//+------------------------------------------------------------------+
//|                                               SlidingPuzzle4.mqh |
//|                                    Copyright (c) 2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                     Based on original Enrico Lambino application |
//|                             https://www.mql5.com/en/users/iceron |
//|                           https://www.mql5.com/ru/articles/7734/ |
//+------------------------------------------------------------------+

#include <Layouts/GridTkEx.mqh>
#include <Layouts/AppDialogResizable.mqh>
#include <Layouts/LayoutStdLib.mqh>
#include <ControlsPlus/Edit.mqh>
#include <ControlsPlus/Button.mqh>


//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSlidingPuzzleDialog: public AppDialogResizable
{
 protected:
  bool m_solved;
  int m_difficulty;

  CGridTkEx m_main;
  CButton m_buttons[16];
  CButton m_button_new;
  CEdit m_label;
  CButton *m_empty_cell;
  
 public:
  CSlidingPuzzleDialog();
  ~CSlidingPuzzleDialog();
  bool CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2);
  virtual bool OnEvent(const int id, const long &lparam, const double &dparam, const string &sparam);
  void Difficulty(int d)
  {
    m_difficulty = d;
  }

 protected:
  virtual bool IsMovable(CButton *button);
  virtual bool HasNorth(CButton *button, int id, bool shuffle = false);
  virtual bool HasSouth(CButton *button, int id, bool shuffle = false);
  virtual bool HasEast(CButton *button, int id, bool shuffle = false);
  virtual bool HasWest(CButton *button, int id, bool shuffle = false);
  virtual void Swap(CButton *source);
  bool OnClickButton(const int i);
  void OnClickButtonNew();
  bool Check();
  void Shuffle();
  
  virtual void SelfAdjustment(const bool restore = false) override;

};

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
EVENT_MAP_BEGIN(CSlidingPuzzleDialog)
  ON_INDEXED_EVENT(ON_CLICK, m_buttons, OnClickButton)
  ON_EVENT(ON_CLICK, m_button_new, OnClickButtonNew)
EVENT_MAP_END(AppDialogResizable)
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CSlidingPuzzleDialog::CSlidingPuzzleDialog():
    m_solved(false),
    m_difficulty(30)
{
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CSlidingPuzzleDialog::~CSlidingPuzzleDialog()
{
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2)
{
  {
    _layout<CSlidingPuzzleDialog> dialog(this, name, x1, y1, x2, y2);
    {
      _layout<CGridTkEx> clientArea(m_main, NULL, 0, 0, ClientAreaWidth(), ClientAreaHeight());
      {
        SimpleSequenceGenerator<long> IDs;
        SimpleSequenceGenerator<string> Captions("0", 15);

        _layout<CButton> block(m_buttons, "block");
        block["background"] <= clrCyan <= IDs <= Captions;
  
        _layout<CButton> start(m_button_new, "New");
        start["background;font"] <= clrYellow <= "Arial Black";
        
        _layout<CEdit> label(m_label);
        label <= "click new" <= true <= ALIGN_CENTER;
      }
      m_main.Init(5, 4, 2, 2);
      m_main.SetGridConstraints(m_button_new, 4, 0, 1, 2);
      m_main.SetGridConstraints(m_label, 4, 2, 1, 2);
      m_main.Pack();
    }
  }
  m_empty_cell = &m_buttons[15];
  
  SelfAdjustment();
  return true;
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::IsMovable(CButton *button)
{
  int id = (int)(button.Id() - m_buttons[0].Id() + 1);
  if(button.Text() == "")
    return (false);
  if(HasNorth(button, id) || HasSouth(button, id) || HasEast(button, id) || HasWest(button, id))
    return (true);
  return (false);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::HasNorth(CButton *button, int id, bool shuffle = false)
{
  if(id == 1 || id == 2 || id == 3 || id == 4)
    return (false);
  CButton *button_adj = m_main.Control(id - 4);
  if(!CheckPointer(button_adj))
    return (false);
  if(!shuffle)
  {
    if(button_adj.Text() != "")
      return (false);
  }
  return (true);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::HasSouth(CButton *button, int id, bool shuffle = false)
{
  if(id == 13 || id == 14 || id == 15 || id == 16)
    return (false);
  CButton *button_adj = m_main.Control(id + 4);
  if(!CheckPointer(button_adj))
    return (false);
  if(!shuffle)
  {
    if(button_adj.Text() != "")
      return (false);
  }
  return (true);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::HasEast(CButton *button, int id, bool shuffle = false)
{
  if(id == 4 || id == 8 || id == 12 || id == 16)
    return (false);
  CButton *button_adj = m_main.Control(id + 1);
  if(!CheckPointer(button_adj))
    return (false);
  if(!shuffle)
  {
    if(button_adj.Text() != "")
      return (false);
  }
  return (true);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::HasWest(CButton *button, int id, bool shuffle = false)
{
  if(id == 1 || id == 5 || id == 9 || id == 13)
    return (false);
  CButton *button_adj = m_main.Control(id - 1);
  if(!CheckPointer(button_adj))
    return (false);
  if(!shuffle)
  {
    if(button_adj.Text() != "")
      return (false);
  }
  return (true);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSlidingPuzzleDialog::Swap(CButton *source)
{
  string source_text = source.Text();
  string target_text = m_empty_cell.Text();
  source.Text(target_text);
  m_empty_cell.Text(source_text);
  m_empty_cell = source;
  m_empty_cell.Text("");
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::OnClickButton(const int i)
{
  CButton *button = &m_buttons[i];
  if(IsMovable(button) && !m_solved)
  {
    Swap(button);
    Check();
  }
  return true;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSlidingPuzzleDialog::OnClickButtonNew(void)
{
  Shuffle();
  m_label.Text("not solved");
  m_solved = false;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSlidingPuzzleDialog::Check(void)
{
  int cnt = 1;
  for(int i = 0; i < m_main.ControlsTotal(); i++)
  {
    CWnd *control = m_main.Control(i);
    if(StringFind(control.Name(), "block") >= 0)
    {
      CButton *button = control;
      if(CheckPointer(button))
      {
        if(button.Text() != IntegerToString(cnt) && cnt < 16)
        {
          m_label.Text("not solved");
          return (false);
        }
      }
      cnt++;
    }
  }
  m_label.Text("solved");
  m_solved = true;
  return (true);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSlidingPuzzleDialog::Shuffle(void)
{
  int cnt = 1;
  for(int i = 1; i < m_main.ControlsTotal(); i++)
  {
    CWnd *control = m_main.Control(i);
    if(StringFind(control.Name(), "block") >= 0)
    {
      CButton *button = control;
      if(cnt < 16)
        button.Text((string)cnt);
      if(button.Id() - m_buttons[0].Id() + 1 == 16)
        m_empty_cell = button;
      cnt++;
    }
  }
  MathSrand((int)TimeLocal());
  CButton *target = NULL;
  for(int i = 0; i < m_difficulty; i++)
  {
    int empty_cell_id = (int)(m_empty_cell.Id() - m_buttons[0].Id() + 1);
    int random = MathRand() % 4 + 1;
    if(random == 1 && HasNorth(m_empty_cell, empty_cell_id, true))
      target = m_main.Control(empty_cell_id - 4);
    else if(random == 2 && HasEast(m_empty_cell, empty_cell_id, true))
      target = m_main.Control(empty_cell_id + 1);
    else if(random == 3 && HasSouth(m_empty_cell, empty_cell_id, true))
      target = m_main.Control(empty_cell_id + 4);
    else if(random == 4 && HasWest(m_empty_cell, empty_cell_id, true))
      target = m_main.Control(empty_cell_id - 1);
    if(CheckPointer(target))
      Swap(target);
  }
}
//+------------------------------------------------------------------+

void CSlidingPuzzleDialog::SelfAdjustment(const bool restore = false)
{
  CSize size;
  size.cx = ClientAreaWidth();
  size.cy = ClientAreaHeight();
  m_main.Size(size);
  m_main.Pack();
}
