// ============================================================================
// File: TimeFilteredSignal.mq5
// Purpose: Time-filtered RSI signal + alerts indicator
// Notes: Uses large Wingdings arrows for visual clarity
//         Signals only appear when IsTradingAllowed(iCTX) == true
// ============================================================================
#property copyright "Clemence Benjamin"
#property strict
#property indicator_chart_window

// We use two plots: one for bullish arrows, one for bearish arrows
#property indicator_plots   2
#property indicator_buffers 2

// --- Plot 1: Bullish RSI signal (up arrow) ---
#property indicator_type1   DRAW_ARROW
#property indicator_color1  clrLime
#property indicator_width1  2
#property indicator_label1  "TF_RSI_Bull"

// --- Plot 2: Bearish RSI signal (down arrow) ---
#property indicator_type2   DRAW_ARROW
#property indicator_color2  clrRed
#property indicator_width2  2
#property indicator_label2  "TF_RSI_Bear"

#include <SessionVisualizer.mqh>
#include <TimeFilters.mqh>

// --------------------------- Inputs -----------------------------------------
// Visual session context (indicator-side)
input bool   InpDrawSessions_i    = true;
input int    InpGMTOffsetHours_i  = 0;
input int    InpLookbackDays_i    = 5;

// RSI configuration
input int    InpRSIPeriod         = 14;
input int    InpRSIOverbought     = 60;
input int    InpRSIOversold       = 40;

// Alert behavior
input bool   InpAlertOnCross      = true;      // enable popup alert on RSI cross
input bool   InpSendPush          = false;     // send mobile push
input bool   InpSendEmail         = false;     // send email
input string InpAlertPrefix       = "TF-RSI";  // prefix tag in messages

// --------------------------- Globals ----------------------------------------
CSessionVisualizer  iSV("TFI_SESS_");
CTimeFilterContext  iCTX;

// Two buffers: one for bullish arrows, one for bearish arrows
double BuffUp[];
double BuffDn[];

int      rsiHandle            = INVALID_HANDLE;
datetime gLastRSIAlertBarTime = 0;   // avoid duplicate alerts per bar

// --------------------------- Alert helper -----------------------------------
void FireRSIAlert(const string direction, const double rsiValue, const datetime barTime)
{
   string timeStr = TimeToString(barTime, TIME_DATE|TIME_SECONDS);

   string msg = StringFormat("%s | %s | %s | RSI=%.2f | Time=%s (inside allowed window)",
                             InpAlertPrefix,
                             _Symbol,
                             direction,
                             rsiValue,
                             timeStr);

   Alert(msg);

   if(InpSendPush)
      SendNotification(msg);

   if(InpSendEmail)
      SendMail(InpAlertPrefix + " " + _Symbol, msg);

   Print("TimeFilteredSignal: ", msg);
}

// --------------------------- OnInit -----------------------------------------
int OnInit()
{
   // Bind buffers
   SetIndexBuffer(0, BuffUp, INDICATOR_DATA);
   SetIndexBuffer(1, BuffDn, INDICATOR_DATA);

   // Use large Wingdings arrows:
   // 233 = up arrow, 234 = down arrow (Wingdings set)
   PlotIndexSetInteger(0, PLOT_ARROW, 233); // Bullish arrow up
   PlotIndexSetInteger(1, PLOT_ARROW, 234); // Bearish arrow down

   // Time filter context
   iCTX.AttachVisualizer(iSV);
   iCTX.SetGMTOffset(InpGMTOffsetHours_i);

   if(InpDrawSessions_i)
      iSV.RefreshSessions(InpLookbackDays_i);

   // Create RSI handle
   rsiHandle = iRSI(_Symbol, _Period, InpRSIPeriod, PRICE_CLOSE);
   if(rsiHandle == INVALID_HANDLE)
   {
      Print("TimeFilteredSignal: Failed to create RSI handle. Error = ", GetLastError());
      return(INIT_FAILED);
   }

   // Initialize buffers as empty
   ArrayInitialize(BuffUp, EMPTY_VALUE);
   ArrayInitialize(BuffDn, EMPTY_VALUE);

   return(INIT_SUCCEEDED);
}

// --------------------------- OnCalculate ------------------------------------
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   if(rates_total <= 2 || rsiHandle == INVALID_HANDLE)
      return(rates_total);

   // Refresh session drawings when new bars appear
   if(InpDrawSessions_i && rates_total != prev_calculated)
      iSV.RefreshSessions(InpLookbackDays_i);

   // Prepare RSI buffer
   static double rsiBuffer[];
   ArrayResize(rsiBuffer, rates_total);

   int copied = CopyBuffer(rsiHandle, 0, 0, rates_total, rsiBuffer);
   if(copied <= 0)
      return(prev_calculated > 0 ? prev_calculated : rates_total);

   // Recalc from prev-1 so last bar updates smoothly
   int start = (prev_calculated > 1 ? prev_calculated - 1 : 1);

   for(int i = start; i < rates_total; ++i)
   {
      // Default: hide both arrows on this bar
      BuffUp[i] = EMPTY_VALUE;
      BuffDn[i] = EMPTY_VALUE;

      // Respect global time filters (same logic as EA)
      if(!IsTradingAllowed(iCTX))
         continue;

      double rsi_prev = rsiBuffer[i - 1];
      double rsi_curr = rsiBuffer[i];

      bool bullCross =
         (rsi_prev < InpRSIOversold && rsi_curr >= InpRSIOversold);

      bool bearCross =
         (rsi_prev > InpRSIOverbought && rsi_curr <= InpRSIOverbought);

      // Bullish RSI recovery: big green up arrow below bar
      if(bullCross)
      {
         BuffUp[i] = low[i] - (_Point * 5);

         // Alert only on the latest bar, once
         if(InpAlertOnCross && i == rates_total - 1 && time[i] != gLastRSIAlertBarTime)
         {
            FireRSIAlert("RSI cross UP from oversold", rsi_curr, time[i]);
            gLastRSIAlertBarTime = time[i];
         }
      }

      // Bearish RSI rejection: big red down arrow above bar
      if(bearCross)
      {
         BuffDn[i] = high[i] + (_Point * 5);

         if(InpAlertOnCross && i == rates_total - 1 && time[i] != gLastRSIAlertBarTime)
         {
            FireRSIAlert("RSI cross DOWN from overbought", rsi_curr, time[i]);
            gLastRSIAlertBarTime = time[i];
         }
      }
   }

   return(rates_total);
}
