//+------------------------------------------------------------------+
//|                                            Nasdaq_NVDA_Coint.mqh |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include <Trade\Trade.mqh>

CTrade trade;

//+------------------------------------------------------------------+
//|          Load strategy parameter from database                   |
//+------------------------------------------------------------------+

/**
 * @brief Loads trading strategy parameters from SQLite database
 *
 * This function queries the specified SQLite database to retrieve strategy parameters
 * including asset symbols, portfolio weights, and timeframe. The data is extracted
 * from JSON arrays stored in the database using SQLite's native json_extract() function.
 *
 * @param db_name Name of the SQLite database file (e.g., "strategies" for "::SQLite\\strategies")
 * @param strat_name Name of the strategy to load from the database
 * @param symbols_arr Output array that will be populated with asset symbols from the strategy
 * @param weights_arr Output array that will be populated with portfolio weights from the strategy
 * @param tf Output variable that will be populated with the strategy's timeframe
 *
 * @return true if strategy was successfully loaded, false otherwise
 *
 * @note The database table must have the following structure:
 *        - name: TEXT (primary key)
 *        - symbols: TEXT (JSON array format, e.g., ["MU", "NVDA", "MPWR", "MCHP"])
 *        - weights: TEXT (JSON array format, e.g., [2.699439, 1.000000, -1.877447, -2.505294])
 *        - timeframe: TEXT (one of the supported timeframe strings)
 *        - lookback: INTEGER (252, 180, 90)
 * @note Supported timeframe strings:
 *        "M1", "M2", "M3", "M4", "M5", "M6", "M10", "M12", "M15", "M20", "M30",
 *        "H1", "H2", "H3", "H4", "H6", "H8", "H12", "D1", "W1", "MN1"
 *
 * @note The function automatically resizes the output arrays to match the number of elements
 *        found in the JSON arrays from the database.
 *
 * @example
 *   string symbols[];
 *   double weights[];
 *   ENUM_TIMEFRAMES tf;
 *   int lb_per;
 *
 *   if(LoadStrategyFromDB("strategies", "MyCointegrationStrategy", symbols, weights, tf, lb_per))
 *   {
 *       Print("Strategy loaded successfully");
 *       Print("Timeframe: ", EnumToString(tf));
 *       for(int i = 0; i < ArraySize(symbols); i++)
 *       {
 *           PrintFormat("Symbol: %s, Weight: %.6f", symbols[i], weights[i]);
 *       }
 *   }
 *   else
 *   {
 *       Print("Failed to load strategy");
 *   }
 *
 * @see DatabaseOpen(), DatabasePrepare(), DatabaseRead(), DatabaseColumnText()
 * @see StringSplit(), StringToDouble()
 *
 * @error On failure, the function returns false and error details are printed to the Experts log
 *        using Print() statements. Common errors include:
 *        - Database not found or inaccessible
 *        - Strategy name not found in database
 *        - Malformed JSON data in symbols or weights columns
 *        - Unsupported timeframe string
 */
bool LoadStrategyFromDB(string db_name,
                        string strat_name,
                        string &symbols_arr[],
                        double &weights_arr[],
                        ENUM_TIMEFRAMES &tf,
                        int &lb_per)

  {
// Open the database (Metatrader's integrated SQLite)
   int db = DatabaseOpen(db_name, DATABASE_OPEN_READONLY);
   if(db == INVALID_HANDLE)
     {
      Print("Failed to open database: %s", GetLastError());
      return false;
     }
// Prepare the SQL query with json_extract
   string query = StringFormat(
                     "SELECT "
                     "json_extract(symbols, '$') as symbols_json, "
                     "json_extract(weights, '$') as weights_json, "
                     "timeframe, "
                     "lookback "
                     "FROM strategy WHERE name = '%s' "
                     "ORDER BY tstamp DESC LIMIT 1 ",
                     strat_name
                  );
// Execute the query
   int result = DatabasePrepare(db, query);
   if(result <= 0)
     {
      Print("Failed to prepare query: ", GetLastError());
      DatabaseClose(db);
      return false;
     }
// Read the results
   ResetLastError();
   if(DatabaseRead(result))
     {
      // Get symbols_arr JSON array and parse it
      string symbolsJson;
      DatabaseColumnText(result, 0, symbolsJson);
      ParseJsonArray(symbolsJson, symbols_arr);
      // Get weights_arr JSON array and parse it
      string weightsJson;
      DatabaseColumnText(result, 1, weightsJson);
      ParseJsonDoubleArray(weightsJson, weights_arr);
      // Get tf string and convert to ENUM_TIMEFRAMES
      string timeframeStr;
      DatabaseColumnText(result, 2, timeframeStr);
      tf = StringToTimeframe(timeframeStr);
      // Get the lookback period
      DatabaseColumnInteger(result, 3, lb_per);
      Print("Successfully loaded strategy: ", strat_name);
      Print("Symbols JSON: ", symbolsJson);
      Print("Weights JSON: ", weightsJson);
      Print("Timeframe: ", timeframeStr);
      Print("Lookback: ", lb_per);
     }
   else
     {
      Print("Failed to read results: ", GetLastError());
      Print("Strategy not found: ", strat_name);
      DatabaseFinalize(result);
      DatabaseClose(db);
      return false;
     }
// Clean up
   DatabaseFinalize(result);
   DatabaseClose(db);
   return true;
  }
//+------------------------------------------------------------------+
//|         overload                                                 |
//+------------------------------------------------------------------+
bool LoadSymbolsAndTimeframeFromDB(string db_name,
                                   string strat_name,
                                   string &symbols_arr[],
                                   ENUM_TIMEFRAMES &tf_arr[])

  {
// Open the database (Metatrader's integrated SQLite)
   int db = DatabaseOpen(db_name, DATABASE_OPEN_READONLY);
   if(db == INVALID_HANDLE)
     {
      Print("Failed to open database: %s", GetLastError());
      return false;
     }
// Prepare the SQL query with json_extract
   string query = StringFormat(
                     "SELECT "
                     "json_extract(symbols, '$') as symbols_json, "
                     "timeframe "
                     "FROM strategy WHERE name = '%s'",
                     strat_name
                  );
// Execute the query
   int result = DatabasePrepare(db, query);
   if(result <= 0)
     {
      Print("Failed to prepare query: ", GetLastError());
      DatabaseClose(db);
      return false;
     }
// Read the results
   if(DatabaseRead(result))
     {
      // Get symbols_arr JSON array and parse it
      string symbolsJson;
      DatabaseColumnText(result, 0, symbolsJson);
      ParseJsonArray(symbolsJson, symbols_arr);
      // Get tf string and convert to ENUM_TIMEFRAMES
      string timeframeStr;
      DatabaseColumnText(result, 1, timeframeStr);
      tf_arr[0] = StringToTimeframe(timeframeStr);
      Print("Successfully loaded strategy: ", strat_name);
      Print("Symbols JSON: ", symbolsJson);
      Print("Timeframe: ", timeframeStr);
     }
   else
     {
      Print("Strategy not found: ", strat_name);
      DatabaseFinalize(result);
      DatabaseClose(db);
      return false;
     }
// Clean up
   DatabaseFinalize(result);
   DatabaseClose(db);
   return true;
  }

// Helper function to parse JSON array of strings
void ParseJsonArray(string json, string &array[])
  {
// Remove brackets and quotes from JSON array
   string cleaned = StringSubstr(json, 1, StringLen(json) - 2); // Remove [ and ]
   StringReplace(cleaned, "\"", ""); // Remove quotes
// Split by commas
   StringSplit(cleaned, ',', array);
// Trim whitespace from each element
   for(int i = 0; i < ArraySize(array); i++)
     {
      array[i] = StringTrim(array[i]);
     }
  }

// Helper function to parse JSON array of doubles
void ParseJsonDoubleArray(string json, double &array[])
  {
// Remove brackets from JSON array
   string cleaned = StringSubstr(json, 1, StringLen(json) - 2); // Remove [ and ]
// Split by commas
   string stringArray[];
   int count = StringSplit(cleaned, ',', stringArray);
// Convert to double array
   ArrayResize(array, count);
   for(int i = 0; i < count; i++)
     {
      array[i] = StringToDouble(StringTrim(stringArray[i]));
     }
  }

// Helper function to trim whitespace
string StringTrim(string str)
  {
// Trim leading whitespace
   while(StringLen(str) > 0 && str[0] == ' ')
     {
      str = StringSubstr(str, 1);
     }
// Trim trailing whitespace
   while(StringLen(str) > 0 && str[StringLen(str) - 1] == ' ')
     {
      str = StringSubstr(str, 0, StringLen(str) - 1);
     }
   return str;
  }

// Helper function to convert tf string to ENUM_TIMEFRAMES
ENUM_TIMEFRAMES StringToTimeframe(string tfStr)
  {
   if(tfStr == "M1")
      return PERIOD_M1;
   if(tfStr == "M2")
      return PERIOD_M2;
   if(tfStr == "M3")
      return PERIOD_M3;
   if(tfStr == "M4")
      return PERIOD_M4;
   if(tfStr == "M5")
      return PERIOD_M5;
   if(tfStr == "M6")
      return PERIOD_M6;
   if(tfStr == "M10")
      return PERIOD_M10;
   if(tfStr == "M12")
      return PERIOD_M12;
   if(tfStr == "M15")
      return PERIOD_M15;
   if(tfStr == "M20")
      return PERIOD_M20;
   if(tfStr == "M30")
      return PERIOD_M30;
   if(tfStr == "H1")
      return PERIOD_H1;
   if(tfStr == "H2")
      return PERIOD_H2;
   if(tfStr == "H3")
      return PERIOD_H3;
   if(tfStr == "H4")
      return PERIOD_H4;
   if(tfStr == "H6")
      return PERIOD_H6;
   if(tfStr == "H8")
      return PERIOD_H8;
   if(tfStr == "H12")
      return PERIOD_H12;
   if(tfStr == "D1")
      return PERIOD_D1;
   if(tfStr == "W1")
      return PERIOD_W1;
   if(tfStr == "MN1")
      return PERIOD_MN1;
   return PERIOD_CURRENT; // Default if not found
  }
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Custom Moving Average Calculation                                |
//+------------------------------------------------------------------+
double CalculateMA(const double &array[], int period)
  {
   double sum = 0.0;
   for(int i = 0; i < period; i++)
      sum += array[i];
   return sum / period;
  }

//+------------------------------------------------------------------+
//| Custom Standard Deviation Calculation                            |
//+------------------------------------------------------------------+
double CalculateStdDev(const double &array[], int period, double mean)
  {
   double variance = 0.0;
   for(int i = 0; i < period; i++)
      variance += MathPow(array[i] - mean, 2);
   return MathSqrt(variance / period);
  }
//+------------------------------------------------------------------+
//| Calculate current spread value                                   |
//+------------------------------------------------------------------+
double CalculateSpread(string &symbols_arr[],
                       ENUM_TIMEFRAMES tf,
                       double &weights_arr[])
  {
   double spread = 0.0;
   for(int i = 0; i < ArraySize(symbols_arr); i++)
     {
      double price = iClose(symbols_arr[i], tf, 0); // Use latest closing price
      spread += weights_arr[i] * price;
     }
   return spread;
  }

//+------------------------------------------------------------------+
//| Execute trade with normalized integer lots                       |
//+------------------------------------------------------------------+
void ExecuteTrade(ENUM_ORDER_TYPE order_type,
                  string &symbols_arr[],
                  double &weights_arr[],
                  double lot_size)
  {
   double vol_arr[];
   ArrayResize(vol_arr, ArraySize(symbols_arr));
   if(!NormalizeVolumeToIntegerLots(vol_arr, symbols_arr, weights_arr, lot_size))
     {
      Print("Volume normalization failed!");
      return;
     }
   for(int i = 0; i < ArraySize(symbols_arr); i++)
     {
      ENUM_ORDER_TYPE leg_type = (weights_arr[i] > 0) ? order_type :
                                 (order_type == ORDER_TYPE_BUY ? ORDER_TYPE_SELL : ORDER_TYPE_BUY);
      trade.PositionOpen(symbols_arr[i], leg_type, vol_arr[i], 0, 0, 0, "NVDA Coint");
     }
  }
//+------------------------------------------------------------------+
//| Custom clamping function (replaces MathClamp)                    |
//+------------------------------------------------------------------+
double Clamp(double value, double min, double max)
  {
   return MathMin(MathMax(value, min), max);
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Normalize floating-point volumes to integer lots                 |
//| while preserving hedge ratios.                                   |
//| Returns true if successful, false if normalization fails.        |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Normalize volumes to integer lots                                |
//+------------------------------------------------------------------+
bool NormalizeVolumeToIntegerLots(double &volumeArray[], const string &symbols_arr[], const double &weights_arr[], double baseLotSize)
  {
   MqlTick tick; // Structure to store bid/ask prices
   double totalDollarExposure = 0.0;
   double dollarExposures[];
   ArrayResize(dollarExposures, ArraySize(symbols_arr));
// Step 1: Calculate dollar exposure for each leg
   for(int i = 0; i < ArraySize(symbols_arr); i++)
     {
      if(!SymbolInfoTick(symbols_arr[i], tick)) // Get latest bid/ask
        {
         Print("Failed to get price for ", symbols_arr[i]);
         return false;
        }
      // Use bid price for short legs, ask for long legs
      double price = (weights_arr[i] > 0) ? tick.ask : tick.bid;
      dollarExposures[i] = MathAbs(weights_arr[i]) * price * baseLotSize;
      totalDollarExposure += dollarExposures[i];
     }
// Step 2: Convert dollar exposure to integer lots
   for(int i = 0; i < ArraySize(symbols_arr); i++)
     {
      double ratio = dollarExposures[i] / totalDollarExposure;
      double targetDollarExposure = ratio * totalDollarExposure;
      // Get min/max lot size and step for the symbol
      double minLot = SymbolInfoDouble(symbols_arr[i], SYMBOL_VOLUME_MIN);
      double maxLot = SymbolInfoDouble(symbols_arr[i], SYMBOL_VOLUME_MAX);
      double lotStep = SymbolInfoDouble(symbols_arr[i], SYMBOL_VOLUME_STEP);
      // Get current price again (for lot calculation)
      if(!SymbolInfoTick(symbols_arr[i], tick))
         return false;
      double price = (weights_arr[i] > 0) ? tick.ask : tick.bid;
      double lots = targetDollarExposure / price;
      lots = MathFloor(lots / lotStep) * lotStep; // Round down to nearest step
      // Clamp to broker constraints using custom Clamp()
      volumeArray[i] = Clamp(lots, minLot, maxLot);
     }
   return true;
  }

//+------------------------------------------------------------------+
//| Close all open positions                                         |
//+------------------------------------------------------------------+
void CloseAllTrades(int max_slippage)
  {
   for(int i = 0; i < PositionsTotal(); i++)
     {
      ulong ticket = PositionGetTicket(i);
      trade.PositionClose(ticket, max_slippage);
     }
  }
//+------------------------------------------------------------------+
//|          Wrapper to load strategy from db                       |
//+------------------------------------------------------------------+
// Load strategy parameters from database
bool UpdateModelParams(string db_name,
                       string strat_name,
                       string &symbols_arr[],
                       double &weights_arr[],
                       ENUM_TIMEFRAMES tf,
                       int &lb_per)
  {
   return LoadStrategyFromDB(db_name, strat_name, symbols_arr, weights_arr, tf, lb_per);
  }
//+------------------------------------------------------------------+
