//+------------------------------------------------------------------+
//|                                                   SignalWVAE.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals of Inference Learning with Stochastic-Oscillator and FrAMA. |
//| Type=SignalAdvanced                                              |
//| Name=IL_Stochastic_FrAMA                                         |
//| ShortName=IL_Stochastic_FrAMA                                    |
//| Class=CSignalIL_Stochastic_FrAMA                                 |
//| Page=signal_il_stochastic_frama                                  |
//| Parameter=Pattern_9,int,50,Pattern 0 [0...100]                   |
//| Parameter=Pattern_6,int,50,Pattern 1 [0...100]                   |
//| Parameter=Pattern_5,int,50,Pattern 5 [0...100]                   |
//| Parameter=ModelType,int,0,Model Type [0...2]                     |
//| Parameter=Pips,int,50,Pips                                       |
//| Parameter=Past,int,30,Past Look Back                             |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalIL_Stochastic_FrAMA.                                |
//| Purpose: Class of generator of trade signals based on            |
//|          Inference Learning with Stochastic-Oscillator and FrAMA.|
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+
//
#resource "Python/84-USDJPY.onnx" as uchar __84_USDJPY[]
#resource "Python/84-XAU.onnx" as uchar __84_XAU[]
#resource "Python/84-SPY.onnx" as uchar __84_SPY[]
//
#define  __MODELS 3
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalIL_Stochastic_FrAMA : public CExpertSignal
{
protected:
   CiStochastic   m_stochastic;
   CiFrAMA        m_frama;

   //long              m_handles[__MODELS];
   //--- adjusted parameters

   //--- "weights" of market models (0-100)
   int               m_pattern_9;      // model 9
   int               m_pattern_6;      // model 6
   int               m_pattern_5;      // model 5
   
   int               m_pips;
   int               m_past;
   double            m_last;

   int               m_model_type;

   //
   //int               m_patterns_usage;   //

public:
   CSignalIL_Stochastic_FrAMA(void);
   ~CSignalIL_Stochastic_FrAMA(void);
   //--- methods of setting adjustable parameters
   //--- methods of adjusting "weights" of market models
   void              Pattern_9(int value)
   {  m_pattern_9 = value;
   }

   void              Pattern_6(int value)
   {  m_pattern_6 = value;
   }

   void              Pattern_5(int value)
   {  m_pattern_5 = value;
   }
   
   void              Pips(int value)
   {  m_pips = value;
   }
   void              Past(int value)
   {  m_past = value;
   }

   void              ModelType(int value)
   {  m_model_type = value;
   }

   long              m_handles[__MODELS];

   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the oscillator and timeseries
   //--- method of creating the oscillator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   //--- methods of getting data
   double            FrAMA(int ind)
   {  //
      m_frama.Refresh(-1);
      double _f = m_frama.Main(ind);
      if(!MathIsValidNumber(_f))
      {  for(int i = ind; i < m_past + ind; i++)
         {  m_frama.Refresh(-1);
            if(MathIsValidNumber(m_frama.Main(i)))
            {  _f = m_frama.Main(i);
            }
         }
      }
      if(MathIsValidNumber(_f))
      {  m_last = _f;
      }
      else
      {  _f = m_last;
      }
      return(_f);
   }
   double  FrAMASlope( int ind)
   {  return FrAMA(ind) - FrAMA(ind + 1);
   }

   double  K( int ind)
   {  m_stochastic.Refresh(-1);
      return m_stochastic.Main(ind);     // %K
   }
   double  D( int ind)
   {  m_stochastic.Refresh(-1);
      return m_stochastic.Signal(ind);   // %D
   }
   //
   bool    CrossUp(double a0, double b0, double a1, double b1)
   {  return (a1 <= b1 && a0 > b0);
   }
   bool    CrossDown(double a0, double b0, double a1, double b1)
   {  return (a1 >= b1 && a0 < b0);
   }

   bool    FlatFrama(int ind);
   bool    FarAboveFrama(const int ind, double mult = 0.0000005);
   bool    FarBelowFrama(const int ind, double mult = 0.0000005);

   bool    BullishDivergence();
   bool    BearishDivergence();

   bool    FramaTurningUp();
   bool    FramaTurningDown();
   //
   double            Close(int ind)
   {  //
      m_close.Refresh(-1);
      return(m_symbol.Bid() + (m_close.GetData(ind) - m_close.GetData(X())));
   }
   double            High(int ind)
   {  //
      m_high.Refresh(-1);
      return(m_high.GetData(ind));
   }
   double            Low(int ind)
   {  //
      m_low.Refresh(-1);
      return(m_low.GetData(ind));
   }
   int               X()
   {  //
      return(StartIndex());
   }
   //--- methods to check for patterns
   bool              IsPattern_9(ENUM_POSITION_TYPE T);
   bool              IsPattern_6(ENUM_POSITION_TYPE T);
   bool              IsPattern_5(ENUM_POSITION_TYPE T);

   double   Infer(ENUM_POSITION_TYPE T, vectorf &X);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalIL_Stochastic_FrAMA::CSignalIL_Stochastic_FrAMA(void) : m_pattern_6(50),
   m_pattern_9(50),
   m_pattern_5(50),
   m_model_type(0)
//m_patterns_usage(255)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_CLOSE + USE_SERIES_TIME;
   PatternsUsage(m_patterns_usage);
//--- create model from static buffer
   m_handles[0] = OnnxCreateFromBuffer(__84_USDJPY, ONNX_DEFAULT);
   m_handles[1] = OnnxCreateFromBuffer(__84_XAU, ONNX_DEFAULT);
   m_handles[2] = OnnxCreateFromBuffer(__84_SPY, ONNX_DEFAULT);
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalIL_Stochastic_FrAMA::~CSignalIL_Stochastic_FrAMA(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::ValidationSettings(void)
{
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   // Set input shapes
   const long _in_shape[] = {1, 6};
   const long _out_shape[] = {1, 1};
   if(!OnnxSetInputShape(m_handles[m_model_type], ONNX_DEFAULT, _in_shape))
   {  Print("OnnxSetInputShape error ", GetLastError());
      return(false);
   }
   // Set output shapes
   if(!OnnxSetOutputShape(m_handles[m_model_type], 0, _out_shape))
   {  Print("OnnxSetOutputShape error ", GetLastError());
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Initialize MA indicators.                                        |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- initialize object
   if(!indicators.Add(GetPointer(m_stochastic)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
   if(!indicators.Add(GetPointer(m_frama)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_stochastic.Create(m_symbol.Name(), m_period, m_past, 3, 3, MODE_SMA, STO_CLOSECLOSE))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_frama.Create(m_symbol.Name(), m_period, m_past, 0, PRICE_CLOSE))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::FlatFrama(int ind)
{  const double tol = m_pips * m_symbol.Point();
   for(int i = ind; i < m_past+ind; i++)
      if(MathAbs(FrAMASlope(i)) > tol) return false;
   return true;
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::FarAboveFrama(const int ind, double mult)
{  double dist = MathAbs(Close(ind) - FrAMA(ind));
   double atr  = High(ind + 1) - Low(ind + 1);
   if(atr <= 0.0) return false;
   //printf(__FUNCTION__ + " dist & atr are: %.5f, & %.5f ", dist, atr);
   return (Close(ind) > FrAMA(ind) && dist > mult * m_symbol.Point() * atr / 4.0);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::FarBelowFrama(const int ind, double mult)
{  const double dist = MathAbs(Close(ind) - FrAMA(ind));
   const double atr  = High(ind + 1) - Low(ind + 1);
   if(atr <= 0.0) return false;
   //printf(__FUNCTION__);
   return (Close(ind) < FrAMA(ind) && dist > mult * m_symbol.Point() * atr / 4.0);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::BullishDivergence()
{  int a = -1, b = -1;
   for(int i = 2; i <= m_past + 2; i++)
   {  if(Low(i) < Low(i - 1) && Low(i) < Low(i + 1))
      {  if(a == -1) a = i;
         else
         {  b = i;
            break;
         }
      }
   }
   if(a == -1 || b == -1) return false;
   bool priceLL = (Low(a) < Low(b));
   bool oscHL   = (K(a) > K(b));
   return (priceLL && oscHL);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::BearishDivergence()
{  int a = -1, b = -1;
   for(int i = 2; i <= m_past + 2; i++)
   {  if(High(i) > High(i - 1) && High(i) > High(i + 1))
      {  if(a == -1) a = i;
         else
         {  b = i;
            break;
         }
      }
   }
   if(a == -1 || b == -1) return false;
   bool priceHH = (High(a) > High(b));
   bool oscLH   = (K(a) < K(b));
   return (priceHH && oscLH);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::FramaTurningUp()
{  return (FrAMASlope(X() + 3) < 0 && FrAMASlope(X() + 2) < 0 && FrAMASlope(X() + 1) > 0 && FrAMASlope(X()) > 0);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::FramaTurningDown()
{  return (FrAMASlope(X() + 3) > 0 && FrAMASlope(X() + 2) > 0 && FrAMASlope(X() + 1) < 0 && FrAMASlope(X()) < 0);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalIL_Stochastic_FrAMA::LongCondition(void)
{  int result  = 0, results = 0;
   vectorf _x(6);
   _x.Fill(0.0f);
//--- if the model 5 is used
   if(IsPattern_5(POSITION_TYPE_BUY))
   {  _x[0] = 1.0f;
      result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used
   if(IsPattern_6(POSITION_TYPE_BUY))
   {  _x[2] = 1.0f;
      result += m_pattern_6;
      results++;
   }
//--- if the model 5 is used
   if(IsPattern_9(POSITION_TYPE_BUY))
   {  _x[4] = 1.0f;
      result += m_pattern_9;
      results++;
   }
   double _y = Infer(POSITION_TYPE_BUY, _x);
//--- return the result
   if(_y > 0.0 && results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalIL_Stochastic_FrAMA::ShortCondition(void)
{  int result  = 0, results = 0;
   vectorf _x(6);
   _x.Fill(0.0f);
//--- if the model 5 is used
   if(IsPattern_5(POSITION_TYPE_SELL))
   {  _x[1] = 1.0f;
      result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used
   if(IsPattern_6(POSITION_TYPE_SELL))
   {  _x[3] = 1.0f;
      result += m_pattern_6;
      results++;
   }
//--- if the model 9 is used
   if(IsPattern_9(POSITION_TYPE_SELL))
   {  _x[5] = 1.0f;
      result += m_pattern_9;
      results++;
   }
   double _y = Infer(POSITION_TYPE_SELL, _x);
//--- return the result
   if(_y < 0.0 && results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| Check for Pattern 9.                                             |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::IsPattern_9(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY)
   {  return((FrAMASlope(X()) < 0 && K(X()) < 10 && K(X()) >= K(X() + 1))  ? true : false);
   }
   else if(T == POSITION_TYPE_SELL)
   {  return((FrAMASlope(X()) > 0 && K(X()) > 90 && K(X()) <= K(X() + 1)) ? true : false);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 6.                                             |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::IsPattern_6(ENUM_POSITION_TYPE T)
{  bool W = (K(X() + 2) < 20 && K(X() + 1) > K(X() + 2) && K(X()) < 20 && K(X()) > K(X() + 1));
   bool M = (K(X() + 2) > 80 && K(X() + 1) < K(X() + 2) && K(X()) > 80 && K(X()) < K(X() + 1));
   if(T == POSITION_TYPE_BUY)
   {  return((FrAMASlope(X()) < 0 && W)  ? true : false);
   }
   else if(T == POSITION_TYPE_SELL)
   {  return((FrAMASlope(X()) > 0 && M) ? true : false);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 5.                                             |
//+------------------------------------------------------------------+
bool CSignalIL_Stochastic_FrAMA::IsPattern_5(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY)
   {  return((FlatFrama(X()) && CrossUp(K(X()), D(X()), K(X() + 1), D(X() + 1)) && K(X()) < 30)  ? true : false);
   }
   else if(T == POSITION_TYPE_SELL)
   {  return((FlatFrama(X()) && CrossDown(K(X()), D(X()), K(X() + 1), D(X() + 1)) && K(X()) > 70) ? true : false);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Inference Learning Forward Pass.                                 |
//+------------------------------------------------------------------+
double CSignalIL_Stochastic_FrAMA::Infer(ENUM_POSITION_TYPE T, vectorf &X)
{  vectorf _y(1);
   _y.Fill(0.0);
   Print(" x: ", __FUNCTION__, X);
   ResetLastError();
   if(!OnnxRun(m_handles[m_model_type], ONNX_NO_CONVERSION, X, _y))
   {  printf(__FUNCSIG__ + " failed to get y forecast, err: %i", GetLastError());
      return(double(_y[0]));
   }
   printf(__FUNCSIG__ + " post y is: ", DoubleToString(_y[0], 5));
   return(double(_y[0]));
}
//+------------------------------------------------------------------+
