
//+------------------------------------------------------------------+
//|                                                  AutoProtect.mq5 |
//|          Copyright 2025, MetaQuotes Ltd. Developer is Chacha Ian |
//|                          https://www.mql5.com/en/users/chachaian |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd. Developer is Chacha Ian"
#property link      "https://www.mql5.com/en/users/chachaian"
#property version   "1.00"

//+------------------------------------------------------------------+
//| Libraries                                                        |
//+------------------------------------------------------------------+
#include <Trade\Trade.mqh>

//+------------------------------------------------------------------+
//| User Input Variables                                             |
//+------------------------------------------------------------------+
input group "Information"
input bool  manageTradesByMagicNumber   = true;
input ulong tradeSelectionMagicNumber   = 254700680002;

input group "Break-Even"
input bool  enableBreakEven            = false;
input int   breakEvenTriggerPoints     = 50;
input int   breakEvenLockPoints        = 0;

input group "Trailing Stop"
input bool  enableTrailingStop         = false;
input int   trailingStartPoints        = 50;
input int   trailingStepPoints         = 20;
input int   trailingDistancePoints     = 100;

input group "Partial Close"
input bool   enablePartialClose        = false;
input int    partialCloseTriggerPoints = 100;
input double partialClosePercent       = 50.0;

//+------------------------------------------------------------------+
//| Data Structures                                                  |
//+------------------------------------------------------------------+
struct MqlTradeInfo{
   ulong ticket;
   ENUM_POSITION_TYPE positionType;
   double openPrice;
   double originalStopLevel;
   double currentStopLevel;
   double nextStopLevel;
   double originalTargetLevel;
   double nextTrailTriggerPrice;
   bool   isMovedToBreakEven;
   bool   isPartialProfitsSecured;
};

//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
CTrade Trade;
MqlTradeInfo tradeInfo[];
double closePriceMinutesData [];
int usefulTrailingStartPoints = trailingStartPoints;
long freezeLevelPoints;
double askPrice;
double bidPrice;
double pointValue;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit(){

   //--- Initialize global variables
   freezeLevelPoints  = SymbolInfoInteger(_Symbol, SYMBOL_TRADE_FREEZE_LEVEL);
   pointValue         = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
   
   //--- If both BreakEven and Trailing are enabled,trailing should only start after break-even has triggered.
   if(enableBreakEven && enableTrailingStop){
      usefulTrailingStartPoints = breakEvenTriggerPoints;
   }
   
   // Set arrays as series
   ArraySetAsSeries(closePriceMinutesData, true);
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason){
   //--- Expert Exit Code
   Print("Expert terminated with Exit Code: ", reason);
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick(){

   //--- Scope variables
   askPrice = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   bidPrice = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   
   //--- Get some minutes data
   if(CopyClose(_Symbol, PERIOD_M1, 0, 7, closePriceMinutesData) == -1){
      Print("Error while copying minutes datas ", GetLastError());
      return;
   }
   
   if(enableBreakEven){
      ManageBreakEven();
   }
   
   if(enableTrailingStop){
      ManageTrailingStop();
   }
   
   if(enablePartialClose){
      ManagePartialClose();
   }
   
}

//+------------------------------------------------------------------+
//| Trade Transaction Events handler                                 |
//+------------------------------------------------------------------+
void OnTradeTransaction(
   const MqlTradeTransaction &trans,
   const MqlTradeRequest     &request,
   const MqlTradeResult      &result){
   
   // When a new position is opened
   if(trans.type   != TRADE_TRANSACTION_DEAL_ADD){
      return;
   }
   
   if(trans.symbol != _Symbol){
      return;
   }
   
   if(trans.deal   == 0){
      return;
   }
   
   bool selected = false;

   HistorySelect(TimeCurrent() - 60, TimeCurrent());
   for(int i = 0; i < 6; i++){
      if(HistoryDealSelect(trans.deal)){
         selected = true;
         break;
      }
      Sleep(5);
      HistorySelect(TimeCurrent() - 60, TimeCurrent());
   }
   
   long entry = -1;
   if(selected){
      entry = (long)HistoryDealGetInteger(trans.deal, DEAL_ENTRY);
   }
   
   if(selected && entry == DEAL_ENTRY_IN){
   
      ulong  positionTicket      = trans.position;
      double openPrice           = 0.000000;
      double originalStopLevel   = 0.000000;
      double originalTargetLevel = 0.000000;
      ulong  magicNumber         = 0;
      
      if(PositionSelectByTicket(positionTicket)){
         openPrice           = PositionGetDouble(POSITION_PRICE_OPEN);
         originalStopLevel   = PositionGetDouble(POSITION_SL);
         originalTargetLevel = PositionGetDouble(POSITION_TP);
         magicNumber         = (ulong)PositionGetInteger(POSITION_MAGIC);
      }
      
      if(manageTradesByMagicNumber){
         if(magicNumber != tradeSelectionMagicNumber){
            return;
         }
      }
   
      if(trans.deal_type == DEAL_TYPE_BUY){
         Print("NEW LONG opened (confirmed via history): deal=", trans.deal, " pos=", trans.position);
         ArrayResize(tradeInfo, ArraySize(tradeInfo) + 1);
         tradeInfo[ArraySize(tradeInfo) - 1].ticket                  = positionTicket;
         tradeInfo[ArraySize(tradeInfo) - 1].positionType            = POSITION_TYPE_BUY;
         tradeInfo[ArraySize(tradeInfo) - 1].openPrice               = openPrice;
         tradeInfo[ArraySize(tradeInfo) - 1].originalStopLevel       = originalStopLevel;
         tradeInfo[ArraySize(tradeInfo) - 1].currentStopLevel        = originalStopLevel;
         tradeInfo[ArraySize(tradeInfo) - 1].nextStopLevel           = originalStopLevel + trailingStepPoints * pointValue;
         tradeInfo[ArraySize(tradeInfo) - 1].originalTargetLevel     = originalTargetLevel;
         tradeInfo[ArraySize(tradeInfo) - 1].nextTrailTriggerPrice   = openPrice + usefulTrailingStartPoints  * pointValue;
         tradeInfo[ArraySize(tradeInfo) - 1].isMovedToBreakEven      = false;
         tradeInfo[ArraySize(tradeInfo) - 1].isPartialProfitsSecured = false;
      }
      
      if(trans.deal_type == DEAL_TYPE_SELL){
         Print("NEW SHORT opened (confirmed via history): deal=", trans.deal, " pos=", trans.position);
         tradeInfo[ArraySize(tradeInfo) - 1].ticket                  = positionTicket;
         tradeInfo[ArraySize(tradeInfo) - 1].positionType            = POSITION_TYPE_SELL;
         tradeInfo[ArraySize(tradeInfo) - 1].openPrice               = openPrice;
         tradeInfo[ArraySize(tradeInfo) - 1].originalStopLevel       = originalStopLevel;
         tradeInfo[ArraySize(tradeInfo) - 1].currentStopLevel        = originalStopLevel;
         tradeInfo[ArraySize(tradeInfo) - 1].nextStopLevel           = originalStopLevel - trailingStepPoints * pointValue;
         tradeInfo[ArraySize(tradeInfo) - 1].originalTargetLevel     = originalTargetLevel;
         tradeInfo[ArraySize(tradeInfo) - 1].nextTrailTriggerPrice   = openPrice - usefulTrailingStartPoints  * pointValue;
         tradeInfo[ArraySize(tradeInfo) - 1].isMovedToBreakEven      = false;
         tradeInfo[ArraySize(tradeInfo) - 1].isPartialProfitsSecured = false;
      }
   }
   
   if(!selected){
   
      if(trans.deal_type == DEAL_TYPE_BUY  && trans.position != 0){
         Print("Probable NEW LONG (fallback): deal=", trans.deal, " pos=", trans.position);
      }
      
      if(trans.deal_type == DEAL_TYPE_SELL && trans.position != 0){
         Print("Probable NEW SHORT (fallback): deal=", trans.deal, " pos=", trans.position);
      }
      
   }
   
}

//--- UTILITY FUNCTIONS
//+------------------------------------------------------------------+
//| Checks if position modification is allowed                       |                               
//+------------------------------------------------------------------+
bool IsTradeModificationAllowed(long freezeLevelPts, ENUM_ORDER_TYPE action, double askPr, double bidPr, double stopLossLvl, double takeProfitLvl){
   double freezeDistance = freezeLevelPts * pointValue;
   
   if(freezeLevelPts == 0){
      return true;
   }
   
   if(action == ORDER_TYPE_BUY) {
      double distanceFromSpotPriceToTP = takeProfitLvl - bidPr;
      double distanceFromSpotPriceToSL = bidPr - stopLossLvl;
      
      if(distanceFromSpotPriceToTP > freezeDistance && distanceFromSpotPriceToSL > freezeDistance){
         return true;
      }
   }
      
   if(action == ORDER_TYPE_SELL){
      double distanceFromSpotPriceToTP = askPr - takeProfitLvl;
      double distanceFromSpotPriceToSL = stopLossLvl - askPr;
      
      if(distanceFromSpotPriceToTP > freezeDistance && distanceFromSpotPriceToSL > freezeDistance){
         return true;
      }    
   }
   return false;
}

//+------------------------------------------------------------------+
//| To detect a crossover                                            |                               
//+------------------------------------------------------------------+
bool IsCrossOver(const double price, const double &closePriceMinsData[]){
   if(closePriceMinsData[1] <= price && closePriceMinsData[0] > price){
      return true;
   }
   return false;
}

//+------------------------------------------------------------------+
//| To detect a crossunder                                           |                               
//+------------------------------------------------------------------+
bool IsCrossUnder(const double price, const double &closePriceMinsData[]){
   if(closePriceMinsData[1] >= price && closePriceMinsData[0] < price){
      return true;
   }
   return false;
}

//+------------------------------------------------------------------+
//| To mange the trade break even functionality                      |                               
//+------------------------------------------------------------------+
void ManageBreakEven   (){
   int totalPositions = PositionsTotal();
   for(int i = totalPositions - 1; i >= 0; i--){
      ulong ticket = PositionGetTicket(i);
      if(ticket != 0){
         // Get some useful position properties
         ENUM_POSITION_TYPE positionType = (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE);
         double openPrice                = PositionGetDouble(POSITION_PRICE_OPEN);
         double currentPrice             = PositionGetDouble(POSITION_PRICE_CURRENT);
         string symbol                   = PositionGetString(POSITION_SYMBOL);
         double stopLevel                = PositionGetDouble(POSITION_SL);
         double takeProfitLevel          = PositionGetDouble(POSITION_TP);
         ulong magicNumber               = PositionGetInteger(POSITION_MAGIC);
         
         if(symbol == _Symbol){
         
            if(!manageTradesByMagicNumber){

               if(positionType == POSITION_TYPE_BUY ){                 
                  for(int j = ArraySize(tradeInfo) - 1; j >= 0; j--){
                     if(tradeInfo[j].ticket == ticket && tradeInfo[j].isMovedToBreakEven == false){
                        if(IsCrossOver(openPrice + breakEvenTriggerPoints * pointValue, closePriceMinutesData)){
                           // Move SL to breakeven + breakEvenLockPoints
                           double newStopLevel = openPrice + (breakEvenLockPoints * pointValue);
                           if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_BUY, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                              if(!Trade.PositionModify(ticket, newStopLevel, takeProfitLevel)){
                                 Print("Error while moving Stop Loss to breakeven! ", GetLastError());
                                 Print(Trade.ResultComment());
                              }
                              tradeInfo[j].isMovedToBreakEven = true;
                           }
                        }
                     }
                  }
               }
       
               if(positionType == POSITION_TYPE_SELL){
                  for(int j = ArraySize(tradeInfo) - 1; i >= 0; i--){
                     if(tradeInfo[j].ticket == ticket && tradeInfo[j].isMovedToBreakEven == false){
                        if(IsCrossUnder(openPrice - breakEvenTriggerPoints * pointValue, closePriceMinutesData)){
                           // Move SL to breakeven + breakEvenLockPoints
                           double newStopLevel = openPrice - (breakEvenLockPoints * pointValue);
                           if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_BUY, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                              if(!Trade.PositionModify(ticket, newStopLevel, takeProfitLevel)){
                                 Print("Error while moving Stop Loss to breakeven! ", GetLastError());
                                 Print(Trade.ResultComment());
                              }
                              tradeInfo[j].isMovedToBreakEven = true;
                           }
                        }
                     }
                  }
               }

            }
            
            if(manageTradesByMagicNumber){
               if(magicNumber == tradeSelectionMagicNumber){
                  if(positionType == POSITION_TYPE_BUY ){                 
                     for(int j = ArraySize(tradeInfo) - 1; j >= 0; j--){
                        if(tradeInfo[j].ticket == ticket && tradeInfo[j].isMovedToBreakEven == false){
                           if(IsCrossOver(openPrice + breakEvenTriggerPoints * pointValue, closePriceMinutesData)){
                              // Move SL to breakeven + breakEvenLockPoints
                              double newStopLevel = openPrice + (breakEvenLockPoints * pointValue);
                              if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_BUY, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                                 if(!Trade.PositionModify(ticket, newStopLevel, takeProfitLevel)){
                                    Print("Error while moving Stop Loss to breakeven! ", GetLastError());
                                    Print(Trade.ResultComment());
                                 }
                                 tradeInfo[j].isMovedToBreakEven = true;
                              }
                           }
                        }
                     }
                  }
          
                  if(positionType == POSITION_TYPE_SELL){
                     for(int j = ArraySize(tradeInfo) - 1; i >= 0; i--){
                        if(tradeInfo[j].ticket == ticket && tradeInfo[j].isMovedToBreakEven == false){
                           if(IsCrossUnder(openPrice - breakEvenTriggerPoints * pointValue, closePriceMinutesData)){
                              // Move SL to breakeven + breakEvenLockPoints
                              double newStopLevel = openPrice - (breakEvenLockPoints * pointValue);
                              if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_BUY, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                                 if(!Trade.PositionModify(ticket, newStopLevel, takeProfitLevel)){
                                    Print("Error while moving Stop Loss to breakeven! ", GetLastError());
                                    Print(Trade.ResultComment());
                                 }
                                 tradeInfo[j].isMovedToBreakEven = true;
                              }
                           }
                        }
                     }
                  }
               }
            }
            
         }else{
            continue;
         }
      }else{
         Print("Error while getting a position ticket!", GetLastError());
         continue;
      }
   }
}

//+------------------------------------------------------------------+
//| To manage the trade trailing stop functionality                  |                               
//+------------------------------------------------------------------+
void ManageTrailingStop(){
   int totalPositions = PositionsTotal();
   // Loop through all open positions
   for(int i = totalPositions - 1; i >= 0; i--){
      ulong ticket = PositionGetTicket(i);
      if(ticket != 0){
         // Get some useful position properties
         ENUM_POSITION_TYPE positionType = (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE);
         double openPrice                = PositionGetDouble (POSITION_PRICE_OPEN);
         double currentPrice             = PositionGetDouble (POSITION_PRICE_CURRENT);
         string symbol                   = PositionGetString (POSITION_SYMBOL);
         double stopLevel                = PositionGetDouble (POSITION_SL);
         double takeProfitLevel          = PositionGetDouble (POSITION_TP);
         ulong magicNumber               = PositionGetInteger(POSITION_MAGIC);
         // Skip positions not matching this financial security
         if(symbol == _Symbol){            
            if(!manageTradesByMagicNumber){

               if(positionType == POSITION_TYPE_BUY ){
                  
                  // 07-10-2025
                  for(int i = ArraySize(tradeInfo) - 1; i >= 0; i--){
                     if(tradeInfo[i].ticket == ticket){
                        if(IsCrossOver(tradeInfo[i].nextTrailTriggerPrice, closePriceMinutesData)){
                           // Physically trail SL only when next SL level is above current SL Level
                           if(tradeInfo[i].nextStopLevel > tradeInfo[i].currentStopLevel){
                              if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_BUY, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                                 if(!Trade.PositionModify(ticket, tradeInfo[i].nextStopLevel, takeProfitLevel)){
                                    Print("Error while trailing Stop Loss! ", GetLastError());
                                    Print(Trade.ResultComment());
                                    Print(Trade.ResultRetcode());
                                 }else{
                                    tradeInfo[i].nextTrailTriggerPrice = tradeInfo[i].nextTrailTriggerPrice + trailingStepPoints * pointValue;
                                 }
                              }
                           }
                           tradeInfo[i].currentStopLevel      = tradeInfo[i].currentStopLevel + trailingStepPoints * pointValue;
                           tradeInfo[i].nextStopLevel         = tradeInfo[i].nextStopLevel    + trailingStepPoints * pointValue;                   
                        }
                     }
                  }
               }
               
               if(positionType == POSITION_TYPE_SELL){
                  // 07-10-2025
                  for(int i = ArraySize(tradeInfo) - 1; i >= 0; i--){
                     if(tradeInfo[i].ticket == ticket){
                        if(IsCrossUnder(tradeInfo[i].nextTrailTriggerPrice, closePriceMinutesData)){
                           // Physically trail SL only when next SL level is above current SL Level
                           if(tradeInfo[i].nextStopLevel < tradeInfo[i].currentStopLevel){
                              if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_SELL, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                                 if(!Trade.PositionModify(ticket, tradeInfo[i].nextStopLevel, takeProfitLevel)){
                                    Print("Error while trailing Stop Loss! ", GetLastError());
                                    Print(Trade.ResultComment());
                                    Print(Trade.ResultRetcode());
                                 }else{
                                    tradeInfo[i].nextTrailTriggerPrice = tradeInfo[i].nextTrailTriggerPrice - trailingStepPoints * pointValue;
                                 }
                              }
                           }
                           tradeInfo[i].currentStopLevel = tradeInfo[i].currentStopLevel - trailingStepPoints * pointValue;
                           tradeInfo[i].nextStopLevel    = tradeInfo[i].nextStopLevel    - trailingStepPoints * pointValue;                          
                        }
                     }
                  }
               }
            }
            
            if( manageTradesByMagicNumber){
               if(magicNumber == tradeSelectionMagicNumber){
 
                  if(positionType == POSITION_TYPE_BUY ){
                     // 07-10-2025
                     for(int i = ArraySize(tradeInfo) - 1; i >= 0; i--){
                        if(tradeInfo[i].ticket == ticket){
                           if(IsCrossOver(tradeInfo[i].nextTrailTriggerPrice, closePriceMinutesData)){
                              // Physically trail SL only when next SL level is above current SL Level
                              if(tradeInfo[i].nextStopLevel > tradeInfo[i].currentStopLevel){
                                 if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_BUY, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                                    if(!Trade.PositionModify(ticket, tradeInfo[i].nextStopLevel, takeProfitLevel)){
                                       Print("Error while trailing Stop Loss! ", GetLastError());
                                       Print(Trade.ResultComment());
                                       Print(Trade.ResultRetcode());
                                    }else{
                                       tradeInfo[i].nextTrailTriggerPrice = tradeInfo[i].nextTrailTriggerPrice + trailingStepPoints * pointValue;
                                    }
                                 }
                              }
                              tradeInfo[i].currentStopLevel = tradeInfo[i].currentStopLevel + trailingStepPoints * pointValue;
                              tradeInfo[i].nextStopLevel    = tradeInfo[i].nextStopLevel    + trailingStepPoints * pointValue;                          
                           }
                        }
                     }
                  }
                  if(positionType == POSITION_TYPE_SELL){
                     // 07-10-2025
                     for(int i = ArraySize(tradeInfo) - 1; i >= 0; i--){
                        if(tradeInfo[i].ticket == ticket){
                           if(IsCrossUnder(tradeInfo[i].nextTrailTriggerPrice, closePriceMinutesData)){
                              // Physically trail SL only when next SL level is above current SL Level
                              if(tradeInfo[i].nextStopLevel < tradeInfo[i].currentStopLevel){
                                 if(IsTradeModificationAllowed(freezeLevelPoints, ORDER_TYPE_SELL, askPrice, bidPrice, stopLevel, takeProfitLevel)){
                                    if(!Trade.PositionModify(ticket, tradeInfo[i].nextStopLevel, takeProfitLevel)){
                                       Print("Error while trailing Stop Loss! ", GetLastError());
                                       Print(Trade.ResultComment());
                                       Print(Trade.ResultRetcode());
                                    }else{
                                       tradeInfo[i].nextTrailTriggerPrice = tradeInfo[i].nextTrailTriggerPrice - trailingStepPoints * pointValue;
                                    }
                                 }
                              }
                              tradeInfo[i].currentStopLevel = tradeInfo[i].currentStopLevel - trailingStepPoints * pointValue;
                              tradeInfo[i].nextStopLevel    = tradeInfo[i].nextStopLevel    - trailingStepPoints * pointValue;                          
                           }
                        }
                     }
                  }
               }else{
                  continue;
               }
            }
            
         }else{
            continue;
         }
      }else{
         Print("Error while getting a position ticket! ", GetLastError());
         continue;
      }
   }
}

//+------------------------------------------------------------------+
//| To manage the trade partial close functionality                  |                               
//+------------------------------------------------------------------+
void ManagePartialClose(){
   int totalPositions = PositionsTotal();
      
   for(int i = totalPositions - 1; i >= 0; i--){
      ulong ticket = PositionGetTicket(i);
      if(ticket != 0){
         // Get some useful position properties
         ENUM_POSITION_TYPE positionType = (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE);
         double openPrice                = PositionGetDouble (POSITION_PRICE_OPEN);
         double currentPrice             = PositionGetDouble (POSITION_PRICE_CURRENT);
         string symbol                   = PositionGetString (POSITION_SYMBOL);
         ulong magicNumber               = PositionGetInteger(POSITION_MAGIC);
         double positionVolume           = PositionGetDouble (POSITION_VOLUME);
         if(symbol == _Symbol){
         
            double volumeToDecrease = NormalizeDouble((partialClosePercent / 100.0) * positionVolume, 2);
            
            if(!manageTradesByMagicNumber){           
               if(positionType == POSITION_TYPE_BUY ){               
                  for(int j = ArraySize(tradeInfo) - 1; j >= 0; j--){
                     if(tradeInfo[j].ticket == ticket && tradeInfo[j].isPartialProfitsSecured == false){
                        if(IsCrossOver(openPrice + partialCloseTriggerPoints * pointValue, closePriceMinutesData)){
                           if(!Trade.PositionClosePartial(ticket, volumeToDecrease)){
                              Print("Error closing partial volume! ", GetLastError());
                              Print(Trade.ResultComment());
                           }
                           tradeInfo[j].isPartialProfitsSecured = true;
                        }
                     }
                  }
               }
               
               if(positionType == POSITION_TYPE_SELL){               
                  for(int j = ArraySize(tradeInfo) - 1; j >= 0; j--){
                     if(tradeInfo[j].ticket == ticket && tradeInfo[j].isPartialProfitsSecured == false){
                        if(IsCrossOver(openPrice - partialCloseTriggerPoints * pointValue, closePriceMinutesData)){
                           if(!Trade.PositionClosePartial(ticket, volumeToDecrease)){
                              Print("Error closing partial volume! ", GetLastError());
                              Print(Trade.ResultComment());
                           }
                           tradeInfo[j].isPartialProfitsSecured = true;
                        }
                     }
                  }
               }
               
            }
            
            if( manageTradesByMagicNumber){
               if(magicNumber == tradeSelectionMagicNumber){
               
                  if(positionType == POSITION_TYPE_BUY ){               
                     for(int j = ArraySize(tradeInfo) - 1; j >= 0; j--){
                        if(tradeInfo[j].ticket == ticket && tradeInfo[j].isPartialProfitsSecured == false){
                           if(IsCrossOver(openPrice + partialCloseTriggerPoints * pointValue, closePriceMinutesData)){
                              if(!Trade.PositionClosePartial(ticket, volumeToDecrease)){
                                 Print("Error closing partial volume! ", GetLastError());
                                 Print(Trade.ResultComment());
                              }
                              tradeInfo[j].isPartialProfitsSecured = true;
                           }
                        }
                     }
                  }
                  
                  if(positionType == POSITION_TYPE_SELL){               
                     for(int j = ArraySize(tradeInfo) - 1; j >= 0; j--){
                        if(tradeInfo[j].ticket == ticket && tradeInfo[j].isPartialProfitsSecured == false){
                           if(IsCrossOver(openPrice - partialCloseTriggerPoints * pointValue, closePriceMinutesData)){
                              if(!Trade.PositionClosePartial(ticket, volumeToDecrease)){
                                 Print("Error closing partial volume! ", GetLastError());
                                 Print(Trade.ResultComment());
                              }
                              tradeInfo[j].isPartialProfitsSecured = true;
                           }
                        }
                     }
                  }
               
               }
            }
            
         }
         
      }else{
         Print("Error while getting a position ticket!", GetLastError());
         continue;
      }
   }
}

//+------------------------------------------------------------------+