//+------------------------------------------------------------------+
//|                              FlexibleMultiTFEngulfingEA.mq5      |
//|                        Expert Advisor for MetaTrader 5            |
//| Description: Trades based on H1/H4/D1 highs/lows purges, waits    |
//| for engulfing candles on H1 or M15, with flexible execution.      |
//+------------------------------------------------------------------+

#property copyright "Your Name"
#property link      "https://www.example.com"
#property version   "1.06"

//--- Input parameters
input double LotSize = 0.1;           // Lot size for trades
input int LookbackBars = 3;           // Number of bars to check for highs/lows
input double StopLossPips = 150.0;    // Stop loss in pips (adjusted for GOLD#)
input double TakeProfitPips = 600.0;  // Take profit in pips (adjusted for GOLD#)
input double TrailingStopPips = 100.0;// Trailing stop in pips
input double EngulfingMinBodyRatio = 0.3; // Min body ratio for engulfing candle
input ENUM_TIMEFRAMES TradeTimeframe = PERIOD_H1; // Primary timeframe for trading
input ENUM_TIMEFRAMES ConfirmationTimeframe = PERIOD_M15; // Lower timeframe for engulfing
input int MaxCandlesPostPurge = 3;    // Max H1 candles to wait for engulfing
input double VolumeThreshold = 1.0;    // Volume multiplier for liquidity confirmation
input bool UseTrendFilter = false;    // Use SMA trend filter
input int SMAPeriod = 50;             // SMA period for trend filter

//--- Global variables
double lastHighH1, lastLowH1, lastHighH4, lastLowH4, lastHighD1, lastLowD1;
datetime lastCandleTime;
bool tradePlaced = false;
double contractSize;
datetime lastPurgeTime = 0;
bool purgeDetected = false;
bool highPurge = false;

//+------------------------------------------------------------------+
//| Expert initialization function                                     |
//+------------------------------------------------------------------+
int OnInit()
{
   if(TradeTimeframe != PERIOD_H1 && TradeTimeframe != PERIOD_H4 && TradeTimeframe != PERIOD_D1)
   {
      Print("Invalid trade timeframe. Use H1, H4, or D1.");
      return(INIT_PARAMETERS_INCORRECT);
   }
   if(ConfirmationTimeframe != PERIOD_M15 && ConfirmationTimeframe != PERIOD_M30)
   {
      Print("Invalid confirmation timeframe. Use M15 or M30.");
      return(INIT_PARAMETERS_INCORRECT);
   }
   
   contractSize = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_CONTRACT_SIZE);
   if(contractSize == 0)
   {
      Print("Failed to get contract size for ", _Symbol);
      return(INIT_PARAMETERS_INCORRECT);
   }
   
   lastHighH1 = 0; lastLowH1 = 0;
   lastHighH4 = 0; lastLowH4 = 0;
   lastHighD1 = 0; lastLowD1 = 0;
   lastCandleTime = 0;
   tradePlaced = false;
   purgeDetected = false;
   lastPurgeTime = 0;
   
   Print("EA Initialized on ", EnumToString(TradeTimeframe), ", Confirmation TF: ", EnumToString(ConfirmationTimeframe), ", Contract Size: ", contractSize);
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert tick function                                              |
//+------------------------------------------------------------------+
void OnTick()
{
   // Check if a new candle has formed on the trade timeframe
   if(!IsNewCandle()) return;
   
   // Reset tradePlaced if no open positions
   if(!PositionSelect(_Symbol)) tradePlaced = false;
   
   // Manage trailing stop for open positions
   ManageTrailingStop();
   
   // Get current and previous candle data for trade timeframe
   MqlRates ratesH1[];
   ArraySetAsSeries(ratesH1, true);
   if(CopyRates(_Symbol, TradeTimeframe, 0, 4, ratesH1) < 4)
   {
      Print("Failed to load H1 rates data");
      return;
   }
   
   // Get candle data for confirmation timeframe (M15)
   MqlRates ratesM15[];
   ArraySetAsSeries(ratesM15, true);
   if(CopyRates(_Symbol, ConfirmationTimeframe, 0, 4, ratesM15) < 4)
   {
      Print("Failed to load M15 rates data");
      return;
   }
   
   // Current H1 candle (index 0)
   double currentOpenH1 = ratesH1[0].open;
   double currentCloseH1 = ratesH1[0].close;
   double currentHighH1 = ratesH1[0].high;
   double currentLowH1 = ratesH1[0].low;
   long currentVolumeH1 = ratesH1[0].tick_volume;
   datetime currentTimeH1 = ratesH1[0].time;
   
   // Update highs and lows for all timeframes
   UpdateHighsLows();
   
   // Check for purge (liquidity sweep) on any timeframe
   bool highPurgedH1 = (currentHighH1 > lastHighH1 && lastHighH1 > 0);
   bool highPurgedH4 = (currentHighH1 > lastHighH4 && lastHighH4 > 0);
   bool highPurgedD1 = (currentHighH1 > lastHighD1 && lastHighD1 > 0);
   bool lowPurgedH1 = (currentLowH1 < lastLowH1 && lastLowH1 > 0);
   bool lowPurgedH4 = (currentLowH1 < lastLowH4 && lastLowH4 > 0);
   bool lowPurgedD1 = (currentLowH1 < lastLowD1 && lastLowD1 > 0);
   bool highPurged = highPurgedH1 || highPurgedH4 || highPurgedD1;
   bool lowPurged = lowPurgedH1 || lowPurgedH4 || lowPurgedD1;
   
   // Update purge status
   if(highPurged || lowPurged)
   {
      purgeDetected = true;
      lastPurgeTime = currentTimeH1;
      highPurge = highPurged;
   }
   
   // Check if within the post-purge window
   bool withinPurgeWindow = false;
   if(purgeDetected)
   {
      int candlesSincePurge = iBarShift(_Symbol, TradeTimeframe, lastPurgeTime, true);
      withinPurgeWindow = candlesSincePurge <= MaxCandlesPostPurge;
      if(!withinPurgeWindow)
      {
         purgeDetected = false; // Reset if window expires
         Print("Purge window expired: ", candlesSincePurge, " candles since last purge");
      }
   }
   
   // Check volume for liquidity confirmation
   bool volumeConfirmed = IsVolumeSpike(currentVolumeH1);
   if(currentVolumeH1 <= 1) 
   {
      Print("Warning: Tick volume is ", currentVolumeH1, ". Possible data issue. Bypassing volume check.");
      volumeConfirmed = true;
   }
   
   // Check trend with SMA
   bool isBullishTrend = UseTrendFilter ? IsBullishTrend() : true;
   
   // Check for engulfing candles on H1 (current + previous 2 candles)
   bool bullishEngulfingH1 = false, bearishEngulfingH1 = false;
   for(int i = 0; i < 3; i++)
   {
      if(IsBullishEngulfing(ratesH1[i], ratesH1[i+1]))
         bullishEngulfingH1 = true;
      if(IsBearishEngulfing(ratesH1[i], ratesH1[i+1]))
         bearishEngulfingH1 = true;
   }
   
   // Check for engulfing candles on M15 (current + previous 2 candles)
   bool bullishEngulfingM15 = false, bearishEngulfingM15 = false;
   for(int i = 0; i < 3; i++)
   {
      if(IsBullishEngulfing(ratesM15[i], ratesM15[i+1]))
         bullishEngulfingM15 = true;
      if(IsBearishEngulfing(ratesM15[i], ratesM15[i+1]))
         bearishEngulfingM15 = true;
   }
   
   // Debug logs
   Print("H1 OHLC: Open=", currentOpenH1, ", High=", currentHighH1, ", Low=", currentLowH1, ", Close=", currentCloseH1);
   Print("H1 High/Low: ", lastHighH1, "/", lastLowH1, 
         " | H4 High/Low: ", lastHighH4, "/", lastLowH4, 
         " | D1 High/Low: ", lastHighD1, "/", lastLowD1);
   Print("Current H1 High/Low: ", currentHighH1, "/", currentLowH1);
   Print("High Purge Check: H1=", currentHighH1-lastHighH1, ", H4=", currentHighH1-lastHighH4, ", D1=", currentHighH1-lastHighD1);
   Print("Low Purge Check: H1=", lastLowH1-currentLowH1, ", H4=", lastLowH4-currentLowH1, ", D1=", lastLowD1-currentLowH1);
   if(highPurgedH1) Print("High purged on H1: ", currentHighH1, " > ", lastHighH1);
   if(highPurgedH4) Print("High purged on H4: ", currentHighH1, " > ", lastHighH4);
   if(highPurgedD1) Print("High purged on D1: ", currentHighH1, " > ", lastHighD1);
   if(lowPurgedH1) Print("Low purged on H1: ", currentLowH1, " < ", lastLowH1);
   if(lowPurgedH4) Print("Low purged on H4: ", currentLowH1, " < ", lastLowH4);
   if(lowPurgedD1) Print("Low purged on D1: ", currentLowH1, " < ", lastLowD1);
   if(!highPurged && !lowPurged) Print("No purge detected");
   if(purgeDetected) Print("Purge detected at ", TimeToString(lastPurgeTime), ", Candles since purge: ", iBarShift(_Symbol, TradeTimeframe, lastPurgeTime, true));
   if(volumeConfirmed) Print("Volume spike confirmed: ", currentVolumeH1);
   else Print("No volume spike detected: ", currentVolumeH1);
   if(UseTrendFilter) Print("Trend: ", isBullishTrend ? "Bullish (Price above SMA)" : "Bearish (Price below SMA)");
   else Print("Trend filter disabled");
   if(bullishEngulfingH1) Print("Bullish engulfing detected on H1");
   if(bearishEngulfingH1) Print("Bearish engulfing detected on H1");
   if(bullishEngulfingM15) Print("Bullish engulfing detected on M15");
   if(bearishEngulfingM15) Print("Bearish engulfing detected on M15");
   if(!bullishEngulfingH1 && !bearishEngulfingH1 && !bullishEngulfingM15 && !bearishEngulfingM15) 
      Print("No engulfing candle detected on H1 or M15");
   if(currentHighH1 == currentLowH1) Print("Warning: Zero-range candle detected on H1");
   
   // Trade logic
   if(purgeDetected && withinPurgeWindow && !tradePlaced)
   {
      if(highPurge && (bullishEngulfingH1 || bullishEngulfingM15) && volumeConfirmed && isBullishTrend)
      {
         Print("Buy signal: High purged, bullish engulfing on ", bullishEngulfingH1 ? "H1" : "M15", ", volume confirmed, bullish trend");
         PlaceTrade(ORDER_TYPE_BUY, currentCloseH1);
      }
      else if(!highPurge && (bearishEngulfingH1 || bearishEngulfingM15) && volumeConfirmed && !isBullishTrend)
      {
         Print("Sell signal: Low purged, bearish engulfing on ", bearishEngulfingH1 ? "H1" : "M15", ", volume confirmed, bearish trend");
         PlaceTrade(ORDER_TYPE_SELL, currentCloseH1);
      }
   }
}

//+------------------------------------------------------------------+
//| Check if a new candle has formed                                  |
//+------------------------------------------------------------------+
bool IsNewCandle()
{
   datetime currentCandleTime = iTime(_Symbol, TradeTimeframe, 0);
   if(currentCandleTime != lastCandleTime)
   {
      lastCandleTime = currentCandleTime;
      return true;
   }
   return false;
}

//+------------------------------------------------------------------+
//| Update highest high and lowest low for H1, H4, D1                 |
//+------------------------------------------------------------------+
void UpdateHighsLows()
{
   // H1
   MqlRates ratesH1[];
   ArraySetAsSeries(ratesH1, true);
   if(CopyRates(_Symbol, PERIOD_H1, 1, LookbackBars, ratesH1) >= LookbackBars)
   {
      lastHighH1 = ratesH1[0].high;
      lastLowH1 = ratesH1[0].low;
      for(int i = 1; i < LookbackBars; i++)
      {
         if(ratesH1[i].high > lastHighH1) lastHighH1 = ratesH1[i].high;
         if(ratesH1[i].low < lastLowH1) lastLowH1 = ratesH1[i].low;
      }
   }
   else Print("Failed to load H1 rates");
   
   // H4
   MqlRates ratesH4[];
   ArraySetAsSeries(ratesH4, true);
   if(CopyRates(_Symbol, PERIOD_H4, 1, LookbackBars, ratesH4) >= LookbackBars)
   {
      lastHighH4 = ratesH4[0].high;
      lastLowH4 = ratesH4[0].low;
      for(int i = 1; i < LookbackBars; i++)
      {
         if(ratesH4[i].high > lastHighH4) lastHighH4 = ratesH4[i].high;
         if(ratesH4[i].low < lastLowH4) lastLowH4 = ratesH4[i].low;
      }
   }
   else Print("Failed to load H4 rates");
   
   // D1
   MqlRates ratesD1[];
   ArraySetAsSeries(ratesD1, true);
   if(CopyRates(_Symbol, PERIOD_D1, 1, LookbackBars, ratesD1) >= LookbackBars)
   {
      lastHighD1 = ratesD1[0].high;
      lastLowD1 = ratesD1[0].low;
      for(int i = 1; i < LookbackBars; i++)
      {
         if(ratesD1[i].high > lastHighD1) lastHighD1 = ratesD1[i].high;
         if(ratesD1[i].low < lastLowD1) lastLowD1 = ratesD1[i].low;
      }
   }
   else Print("Failed to load D1 rates");
}

//+------------------------------------------------------------------+
//| Check for volume spike for liquidity confirmation                 |
//+------------------------------------------------------------------+
bool IsVolumeSpike(long currentVolume)
{
   double avgVolume = 0;
   MqlRates rates[];
   ArraySetAsSeries(rates, true);
   if(CopyRates(_Symbol, TradeTimeframe, 1, LookbackBars, rates) < LookbackBars)
   {
      Print("Failed to load rates for volume check");
      return true;
   }
   
   int validBars = 0;
   for(int i = 0; i < LookbackBars; i++)
   {
      if(rates[i].tick_volume > 1)
      {
         avgVolume += rates[i].tick_volume;
         validBars++;
      }
   }
   if(validBars == 0)
   {
      Print("Warning: No valid volume data. Average volume is 0. Bypassing volume check.");
      return true;
   }
   avgVolume /= validBars;
   
   Print("Current volume: ", currentVolume, ", Average volume: ", avgVolume, ", Valid bars: ", validBars);
   return currentVolume >= VolumeThreshold * avgVolume;
}

//+------------------------------------------------------------------+
//| Check trend using SMA                                            |
//+------------------------------------------------------------------+
bool IsBullishTrend()
{
   double sma[];
   ArraySetAsSeries(sma, true);
   int smaHandle = iMA(_Symbol, TradeTimeframe, SMAPeriod, 0, MODE_SMA, PRICE_CLOSE);
   if(CopyBuffer(smaHandle, 0, 0, 1, sma) < 1)
   {
      Print("Failed to load SMA data");
      return false;
   }
   double currentPrice = iClose(_Symbol, TradeTimeframe, 0);
   Print("Current price: ", currentPrice, ", SMA: ", sma[0]);
   return currentPrice > sma[0];
}

//+------------------------------------------------------------------+
//| Check for bullish engulfing candle                                |
//+------------------------------------------------------------------+
bool IsBullishEngulfing(MqlRates &current, MqlRates &previous)
{
   double currentBody = MathAbs(current.close - current.open);
   double prevBody = MathAbs(previous.close - previous.open);
   
   if(current.close > current.open && 
      previous.close < previous.open && 
      current.open <= previous.close && 
      current.close >= previous.open && 
      currentBody >= EngulfingMinBodyRatio * prevBody)
   {
      return true;
   }
   return false;
}

//+------------------------------------------------------------------+
//| Check for bearish engulfing candle                                |
//+------------------------------------------------------------------+
bool IsBearishEngulfing(MqlRates &current, MqlRates &previous)
{
   double currentBody = MathAbs(current.close - current.open);
   double prevBody = MathAbs(previous.close - previous.open);
   
   if(current.close < current.open && 
      previous.close > previous.open && 
      current.open >= previous.close && 
      current.close <= previous.open && 
      currentBody >= EngulfingMinBodyRatio * prevBody)
   {
      return true;
   }
   return false;
}

//+------------------------------------------------------------------+
//| Place a trade                                                    |
//+------------------------------------------------------------------+
void PlaceTrade(ENUM_ORDER_TYPE orderType, double price)
{
   MqlTradeRequest request = {};
   MqlTradeResult result = {};
   
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = LotSize;
   request.type = orderType;
   request.price = price;
   request.sl = (orderType == ORDER_TYPE_BUY) ? price - StopLossPips * _Point * 100 : price + StopLossPips * _Point * 100;
   request.tp = (orderType == ORDER_TYPE_BUY) ? price + TakeProfitPips * _Point * 100 : price - TakeProfitPips * _Point * 100;
   request.type_filling = ORDER_FILLING_IOC;
   
   if(OrderSend(request, result))
   {
      Print("Trade placed successfully: ", orderType == ORDER_TYPE_BUY ? "BUY" : "SELL", " at ", price, " SL: ", request.sl, " TP: ", request.tp);
      tradePlaced = true;
      purgeDetected = false; // Reset purge after trade
   }
   else
   {
      Print("Trade failed: ", result.retcode);
   }
}

//+------------------------------------------------------------------+
//| Manage trailing stop for open positions                          |
//+------------------------------------------------------------------+
void ManageTrailingStop()
{
   if(!PositionSelect(_Symbol)) return;
   
   double currentPrice = (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) ? 
                         SymbolInfoDouble(_Symbol, SYMBOL_BID) : 
                         SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double openPrice = PositionGetDouble(POSITION_PRICE_OPEN);
   double currentSL = PositionGetDouble(POSITION_SL);
   
   double newSL = 0;
   if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
   {
      newSL = currentPrice - TrailingStopPips * _Point * 100;
      if(newSL > currentSL && newSL > openPrice)
      {
         MqlTradeRequest request = {};
         MqlTradeResult result = {};
         request.action = TRADE_ACTION_SLTP;
         request.position = PositionGetInteger(POSITION_TICKET);
         request.symbol = _Symbol;
         request.sl = newSL;
         request.tp = PositionGetDouble(POSITION_TP);
         if(OrderSend(request, result))
            Print("Trailing stop updated for BUY: ", newSL);
         else
            Print("Failed to update trailing stop: ", result.retcode);
      }
   }
   else // POSITION_TYPE_SELL
   {
      newSL = currentPrice + TrailingStopPips * _Point * 100;
      if((newSL < currentSL || currentSL == 0) && newSL < openPrice)
      {
         MqlTradeRequest request = {};
         MqlTradeResult result = {};
         request.action = TRADE_ACTION_SLTP;
         request.position = PositionGetInteger(POSITION_TICKET);
         request.symbol = _Symbol;
         request.sl = newSL;
         request.tp = PositionGetDouble(POSITION_TP);
         if(OrderSend(request, result))
            Print("Trailing stop updated for SELL: ", newSL);
         else
            Print("Failed to update trailing stop: ", result.retcode);
      }
   }
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   Print("EA Deinitialized, reason: ", reason);
}

//+------------------------------------------------------------------+