//+------------------------------------------------------------------+
//|                                                  SymbolImporter.mq5 |
//|                        Copyright 2024, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#property version   "1.00"
#property script_show_inputs

// Input parameters
input string   InpDatabaseName = "StatArb\\statarb-0.3.db";  // Database filename
input string   InpDataSource = "MetaQuotes"; // Data source (broker or server name)
input string   InpSymbolPath = "Nasdaq\\Stock";         // Symbol path to import
input bool     InpClearExisting = false;        // Clear existing symbols from this source?

//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
// Open or create the database
   int db = DatabaseOpen(InpDatabaseName, DATABASE_OPEN_READWRITE | DATABASE_OPEN_CREATE);
   if(db == INVALID_HANDLE)
     {
      Print("Failed to open database: ", InpDatabaseName, ", Error: ", GetLastError());
      return;
     }
// Clear existing symbols from this path if requested
   if(InpClearExisting)
     {
      if(!ClearSymbolsByPath(db, InpSymbolPath))
        {
         DatabaseClose(db);
         return;
        }
     }
// Get and import symbols from the specified path
   ImportSymbolsFromPath(db, InpSymbolPath);
// Close the database
   DatabaseClose(db);
   Print("Symbol import completed successfully");
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Clear existing symbols from the specified path                  |
//+------------------------------------------------------------------+
bool ClearSymbolsByPath(int db_handle, string path)
  {
   string sql = "DELETE FROM symbol WHERE source = ?";
   int stmt = DatabasePrepare(db_handle, sql);
   if(stmt == INVALID_HANDLE)
     {
      Print("Failed to prepare delete statement: ", GetLastError());
      return false;
     }
   if(!DatabaseBind(stmt, 0, path))
     {
      Print("Failed to bind path parameter: ", GetLastError());
      DatabaseFinalize(stmt);
      return false;
     }
   if(!DatabaseRead(stmt))
     {
      Print("Failed to execute delete: ", GetLastError());
      DatabaseFinalize(stmt);
      return false;
     }
   DatabaseFinalize(stmt);
   Print("Cleared existing symbols from path: ", path);
   return true;
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Import symbols from the specified path                          |
//+------------------------------------------------------------------+
void ImportSymbolsFromPath(int db_handle, string path)
  {
   int total_symbols = SymbolsTotal(false);
   int imported_count = 0;
   Print("Scanning ", total_symbols, " symbols for path: ", path);
   for(int i = 0; i < total_symbols; i++)
     {
      string symbol_name = SymbolName(i, false);
      string symbol_path = SymbolInfoString(symbol_name, SYMBOL_PATH);
      // Check if the symbol belongs to the target path
      if(StringFind(symbol_path, path) != -1)
        {
         if(InsertSymbol(db_handle, symbol_name, InpDataSource))
            imported_count++;
        }
     }
   Print("Imported ", imported_count, " symbols from path: ", path);
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Insert a symbol into the database                               |
//+------------------------------------------------------------------+
bool InsertSymbol(int db_handle, string ticker, string source)
  {
   ResetLastError();
// Get symbol information
   string exchange = SymbolInfoString(ticker, SYMBOL_EXCHANGE);
   string asset_type = GetAssetType(ticker);
   string sector = SymbolInfoString(ticker, SYMBOL_SECTOR_NAME);
   string industry = SymbolInfoString(ticker, SYMBOL_INDUSTRY_NAME);
   string currency = SymbolInfoString(ticker, SYMBOL_CURRENCY_BASE);
   if(currency == "")
      currency = SymbolInfoString(ticker, SYMBOL_CURRENCY_PROFIT);
// Prepare SQL insert statement (symbol_id is auto-generated by SQLite)
   string req = StringFormat(
                   "INSERT INTO symbol(ticker, exchange, asset_type, sector, industry, currency, source)"
                   " VALUES( '%s', '%s', '%s', '%s', '%s', '%s', '%s')",
                   ticker, exchange, asset_type, sector, industry, currency, source);
   if(!DatabaseExecute(db_handle, req))
     {
      printf("Failed to insert symbol: %d", GetLastError());
      return false;
     }
   return true;
  }

//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Determine asset type based on symbol characteristics            |
//+------------------------------------------------------------------+
string GetAssetType(string symbol_name)
  {
// Simple asset type detection - you may want to enhance this
   string description = SymbolInfoString(symbol_name, SYMBOL_DESCRIPTION);
   string path = SymbolInfoString(symbol_name, SYMBOL_PATH);
   if(StringFind(path, "Forex") != -1)
      return "Forex";
   if(StringFind(path, "Stock") != -1)
      return "Stock";
   if(StringFind(path, "Index") != -1)
      return "Index";
   if(StringFind(path, "Future") != -1)
      return "Future";
   if(StringFind(path, "CFD") != -1)
      return "CFD";
   if(StringFind(path, "Crypto") != -1)
      return "Cryptocurrency";
// Fallback based on symbol name patterns
   if(StringLen(symbol_name) == 6 && StringFind(symbol_name, "USD") != -1)
      return "Forex";
   if(StringFind(symbol_name, ".", 0) != -1)
      return "Stock";
   return "Other";
  }
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
