//+------------------------------------------------------------------+
//|                                            OB_&_MitigationOB.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"

#include <Trade/Trade.mqh>
#include <Arrays\ArrayObj.mqh>

CTrade trade;
CArrayObj obList;

#define BullOB clrLime
#define BearOB clrRed
#define ViolatedBullOB clrBlue
#define ViolatedBearOB clrMagenta

//+------------------------------------------------------------------+
//|                           Input parameters                       |
//+------------------------------------------------------------------+
input double Lots = 0.11;
input int takeProfit = 3000;
input double stopLoss = 2000;
input int Time1Hstrt = 1;
input int Time1Hend = 10;

//+------------------------------------------------------------------+
//|                         OrderBlock Class                         |
//+------------------------------------------------------------------+
class COrderBlock : public CObject
{
public:
   int direction;
   datetime time;
   double high;
   double low;
   bool violated;
   datetime violatedTime;
   bool traded;
   bool inTrade;
   string identifier;
   string tradeComment;

   COrderBlock()
   {
      traded = false;
      violated = false;
      inTrade = false;
      tradeComment = "";
   }

   void UpdateIdentifier()
   {
      identifier = IntegerToString(time) + IntegerToString(direction) + IntegerToString(violated);
   }

   void draw(datetime tmS, datetime tmE, color clr)
   {
      UpdateIdentifier();
      string objOB = "OB_REC_" + identifier;
      if(ObjectFind(0, objOB) == -1)
      {
         ObjectCreate(0, objOB, OBJ_RECTANGLE, 0, time, low, tmS, high);
         ObjectSetInteger(0, objOB, OBJPROP_FILL, true);
         ObjectSetInteger(0, objOB, OBJPROP_COLOR, clr);
         ObjectSetInteger(0, objOB, OBJPROP_BACK, true);
      }

      string objtrade = "OB_TRADE_" + identifier;
      if(ObjectFind(0, objtrade) == -1)
      {
         ObjectCreate(0, objtrade, OBJ_RECTANGLE, 0, tmS, high, tmE, low);
         ObjectSetInteger(0, objtrade, OBJPROP_FILL, true);
         ObjectSetInteger(0, objtrade, OBJPROP_COLOR, clr);
         ObjectSetInteger(0, objtrade, OBJPROP_BACK, true);
      }
   }

   void RemoveObjects()
   {
      UpdateIdentifier();
      string objOB = "OB_REC_" + identifier;
      string objtrade = "OB_TRADE_" + identifier;
      if(ObjectFind(0, objOB) >= 0) ObjectDelete(0, objOB);
      if(ObjectFind(0, objtrade) >= 0) ObjectDelete(0, objtrade);
   }
};

//+------------------------------------------------------------------+
//|                           Global variables                       |
//+------------------------------------------------------------------+
COrderBlock *activeMitigationOB = NULL;
int activeMitigationDirection = 0;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   trade.SetExpertMagicNumber(76543);
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   for(int i = obList.Total()-1; i >= 0; i--)
   {
      COrderBlock* ob = obList.At(i);
      ob.RemoveObjects();
      delete ob;
   }
   obList.Clear();
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   if(isNewBar())
   {
      CheckForClosedTradesAndCleanup();
      CheckForViolations();
      getOrderB();
      CheckForTradeEntries();
   }
}

//+------------------------------------------------------------------+
//| Helpers to find bars and swings safely                           |
//+------------------------------------------------------------------+
int BarIndexFromTimeSafely(datetime t)
{
   int idx = iBarShift(_Symbol, _Period, t, true);
   if(idx < 0) idx = Bars(_Symbol, _Period) - 1;
   return(idx);
}

int FindHighestAfter(datetime t, int lookbackBars)
{
   int obIdx = BarIndexFromTimeSafely(t);
   int start = MathMax(1, obIdx - lookbackBars);
   int count = obIdx - start;
   if(count <= 0) return(-1);
   int idx = iHighest(_Symbol, _Period, MODE_HIGH, count, start);
   return(idx);
}

int FindLowestAfter(datetime t, int lookbackBars)
{
   int obIdx = BarIndexFromTimeSafely(t);
   int start = obIdx + 1;
   int maxPossible = Bars(_Symbol, _Period) - start;
   int count = MathMin(lookbackBars, maxPossible);
   if(count <= 0) return(-1);
   int idx = iLowest(_Symbol, _Period, MODE_LOW, count, start);
   return(idx);
}

//+------------------------------------------------------------------+
//| Check for violations of existing orderblocks                     |
//+------------------------------------------------------------------+
void CheckForViolations()
{
   double currentBid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double currentAsk = SymbolInfoDouble(_Symbol, SYMBOL_ASK);

   for(int i = 0; i < obList.Total(); i++)
   {
      COrderBlock* ob = obList.At(i);
      if(ob.violated || ob.traded) continue;

      if(ob.direction == 1 && currentBid < ob.low)
      {
         if(activeMitigationOB != NULL)
         {
            activeMitigationOB = NULL;
            activeMitigationDirection = 0;
         }
         
         ob.violated = true;
         ob.violatedTime = iTime(_Symbol, PERIOD_CURRENT, 0);
         ob.traded = false;
         ob.UpdateIdentifier();
         activeMitigationOB = ob;
         activeMitigationDirection = -1;
         Print("Bullish OB violated and becomes BEARISH mitigation: ", TimeToString(ob.time));
      }
      else if(ob.direction == -1 && currentAsk > ob.high)
      {
         if(activeMitigationOB != NULL)
         {
            activeMitigationOB = NULL;
            activeMitigationDirection = 0;
         }
         
         ob.violated = true;
         ob.violatedTime = iTime(_Symbol, PERIOD_CURRENT, 0);
         ob.traded = false;
         ob.UpdateIdentifier();
         activeMitigationOB = ob;
         activeMitigationDirection = 1;
         Print("Bearish OB violated and becomes BULLISH mitigation: ", TimeToString(ob.time));
      }
   }
}

//+------------------------------------------------------------------+
//| Check for trade entries in orderblocks                           |
//+------------------------------------------------------------------+
void CheckForTradeEntries()
{
   double currentBid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double currentAsk = SymbolInfoDouble(_Symbol, SYMBOL_ASK);

   for(int i = 0; i < obList.Total(); i++)
   {
      COrderBlock* ob = obList.At(i);
      if(ob.traded) continue;

      bool priceInZone = false;
      bool isBullishTrade = false;
      bool isMitigation = ob.violated;

      if(!ob.violated)
      {
         if(ob.direction == 1 && currentAsk < ob.high && currentAsk > ob.low)
         {
            priceInZone = true;
            isBullishTrade = true;
         }
         else if(ob.direction == -1 && currentBid > ob.low && currentBid < ob.high)
         {
            priceInZone = true;
            isBullishTrade = false;
         }
      }
      else
      {
         if(ob.direction == 1 && currentBid > ob.low && currentBid < ob.high)
         {
            priceInZone = true;
            isBullishTrade = false;
         }
         else if(ob.direction == -1 && currentAsk < ob.high && currentAsk > ob.low)
         {
            priceInZone = true;
            isBullishTrade = true;
         }
      }

      if(priceInZone)
      {
         if(isMitigation)
         {
            if(activeMitigationOB == NULL || activeMitigationOB != ob)
            {
               continue;
            }
         }

         if(isBullishTrade)
         {
            ExecuteBuyTrade(ob);
         }
         else
         {
            ExecuteSellTrade(ob);
         }
         ob.traded = true;
         break;
      }
   }
}

//+------------------------------------------------------------------+
//| Execute a buy trade                                              |
//+------------------------------------------------------------------+
void ExecuteBuyTrade(COrderBlock* ob)
{
   double entry = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   int swingHighIdx = FindHighestAfter(ob.time, 20);
   double tp = (swingHighIdx>0) ? getHigh(swingHighIdx) : entry + takeProfit * _Point;
   double sl = NormalizeDouble(ob.low - stopLoss * _Point, _Digits);

   string comment = "Bull_OB_" + ob.identifier;
   ob.tradeComment = comment;

   if(trade.Buy(Lots, _Symbol, entry, sl, tp, comment))
   {
      color obColor = ob.violated ? ViolatedBullOB : BullOB;
      ob.draw(ob.time, iTime(_Symbol, PERIOD_CURRENT, 0), obColor);
      ob.inTrade = true;
      Print("Buy trade executed. Type: ", ob.violated ? "Mitigation" : "Regular", 
            " OB Time: ", TimeToString(ob.time), " Entry: ", entry, " SL: ", sl, " TP: ", tp);
   }
}

//+------------------------------------------------------------------+
//| Execute a sell trade                                             |
//+------------------------------------------------------------------+
void ExecuteSellTrade(COrderBlock* ob)
{
   double entry = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   int swingLowIdx = FindLowestAfter(ob.time, 20);
   double tp = (swingLowIdx>0) ? getLow(swingLowIdx) : entry - takeProfit * _Point;
   double sl = NormalizeDouble(ob.high + stopLoss * _Point, _Digits);

   string comment = "Bear_OB_" + ob.identifier;
   ob.tradeComment = comment;

   if(trade.Sell(Lots, _Symbol, entry, sl, tp, comment))
   {
      color obColor = ob.violated ? ViolatedBearOB : BearOB;
      ob.draw(ob.time, iTime(_Symbol, PERIOD_CURRENT, 0), obColor);
      ob.inTrade = true;
      Print("Sell trade executed. Type: ", ob.violated ? "Mitigation" : "Regular", 
            " OB Time: ", TimeToString(ob.time), " Entry: ", entry, " SL: ", sl, " TP: ", tp);
   }
}

//+------------------------------------------------------------------+
//| Detect new orderblocks                                           |
//+------------------------------------------------------------------+
void getOrderB()
{
   MqlDateTime structTime;
   TimeCurrent(structTime);
   static int prevDay = 0;

   if(structTime.hour >= Time1Hstrt && structTime.hour < Time1Hend)
   {
      if(prevDay != structTime.day)
      {
         prevDay = structTime.day;

         for(int i = 3; i < 100; i++)
         {
            if(i + 3 >= Bars(_Symbol, _Period)) continue;

            if(getOpen(i+2) > getClose(i+2) &&
               getClose(i+1) > getOpen(i+1) &&
               getClose(i) > getOpen(i))
            {
               COrderBlock* ob = new COrderBlock();
               ob.direction = 1;
               ob.time = getTime(i+2);
               ob.high = getHigh(i+2);
               ob.low = getLow(i+2);
               ob.violated = false;
               ob.traded = false;
               ob.UpdateIdentifier();
               obList.Add(ob);
               Print("Bullish Order Block detected at: ", TimeToString(ob.time));
            }
            else if(getOpen(i+2) < getClose(i+2) &&
                    getClose(i+1) < getOpen(i+1) &&
                    getClose(i) < getOpen(i))
            {
               COrderBlock* ob = new COrderBlock();
               ob.direction = -1;
               ob.time = getTime(i+2);
               ob.high = getHigh(i+2);
               ob.low = getLow(i+2);
               ob.violated = false;
               ob.traded = false;
               ob.UpdateIdentifier();
               obList.Add(ob);
               Print("Bearish Order Block detected at: ", TimeToString(ob.time));
            }
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Check for closed trades and cleanup objects after closure        |
//+------------------------------------------------------------------+
void CheckForClosedTradesAndCleanup()
{
   for(int i = obList.Total()-1; i >= 0; i--)
   {
      COrderBlock* ob = obList.At(i);
      if(ob.inTrade)
      {
         if(!PositionWithCommentExists(ob.tradeComment))
         {
            Print("Trade closed for OB: ", ob.identifier, " -> cleaning up drawings");
            ob.RemoveObjects();
            ob.inTrade = false;
            ob.traded = false;
            ob.tradeComment = "";

            if(activeMitigationOB == ob)
            {
               activeMitigationOB = NULL;
               activeMitigationDirection = 0;
               Print("Active mitigation OB cleared after trade closure");
            }

            obList.Delete(i);
            delete ob;
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Check if position with specific comment exists                   |
//+------------------------------------------------------------------+
bool PositionWithCommentExists(string comment)
{
   if(StringLen(comment) == 0) return(false);
   for(int i = PositionsTotal()-1; i >= 0; i--)
   {
      ulong ticket = PositionGetTicket(i);
      if(ticket > 0 && PositionSelectByTicket(ticket))
      {
         string c = PositionGetString(POSITION_COMMENT);
         if(c == comment) return(true);
      }
   }
   return(false);
}

//+------------------------------------------------------------------+
//| Helper functions                                                 |
//+------------------------------------------------------------------+
bool isNewBar()
{
   static datetime lastBar;
   datetime currentBar = iTime(_Symbol, _Period, 0);
   if(lastBar != currentBar)
   {
      lastBar = currentBar;
      return true;
   }
   return false;
}

double getHigh(int index) { return iHigh(_Symbol, _Period, index); }
double getLow(int index) { return iLow(_Symbol, _Period, index); }
double getOpen(int index) { return iOpen(_Symbol, _Period, index); }
double getClose(int index) { return iClose(_Symbol, _Period, index); }
datetime getTime(int index) { return iTime(_Symbol, _Period, index); }