#property copyright "ATR Sentinel Trailing System"
#property version   "1.00"
#property description "Converted from TradingView Pine Script to MQL5 with Trailing Stop Line"
#property indicator_chart_window
#property indicator_buffers 3
#property indicator_plots   3

#property indicator_label1  "Buy"
#property indicator_type1   DRAW_ARROW
#property indicator_color1  clrGreen
#property indicator_style1  STYLE_SOLID
#property indicator_width1  2

#property indicator_label2  "Sell"
#property indicator_type2   DRAW_ARROW
#property indicator_color2  clrRed
#property indicator_style2  STYLE_SOLID
#property indicator_width2  2

#property indicator_label3  "Trailing Stop"
#property indicator_type3   DRAW_LINE
#property indicator_color3  clrBlue
#property indicator_style3  STYLE_SOLID
#property indicator_width3  2

input double a = 1.0;   // Key Value. 'This changes the sensitivity'
input int    c = 10;    // ATR Period
input bool   h = false; // Signals from Heikin Ashi Candles

double BuyBuffer[];
double SellBuffer[];
double TrailBuffer[];

int atr_handle;
double ha_open[], ha_high[], ha_low[], ha_close[];
bool calculate_ha = false;

int OnInit()
{
   SetIndexBuffer(0, BuyBuffer, INDICATOR_DATA);
   PlotIndexSetInteger(0, PLOT_ARROW, 233); // Up arrow
   PlotIndexSetDouble(0, PLOT_EMPTY_VALUE, EMPTY_VALUE);

   SetIndexBuffer(1, SellBuffer, INDICATOR_DATA);
   PlotIndexSetInteger(1, PLOT_ARROW, 234); // Down arrow
   PlotIndexSetDouble(1, PLOT_EMPTY_VALUE, EMPTY_VALUE);

   SetIndexBuffer(2, TrailBuffer, INDICATOR_DATA);
   PlotIndexSetDouble(2, PLOT_EMPTY_VALUE, EMPTY_VALUE);

   atr_handle = iATR(_Symbol, PERIOD_CURRENT, c);
   if (atr_handle == INVALID_HANDLE)
   {
      Print("Failed to create ATR handle");
      return INIT_FAILED;
   }

   calculate_ha = h;
   ArraySetAsSeries(BuyBuffer, false);
   ArraySetAsSeries(SellBuffer, false);
   ArraySetAsSeries(TrailBuffer, false);

   Print("Indicator initialized. Symbol: ", _Symbol, ", Timeframe: ", Period());

   return INIT_SUCCEEDED;
}

void OnDeinit(const int reason)
{
   if (atr_handle != INVALID_HANDLE)
      IndicatorRelease(atr_handle);
}

int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   if (rates_total < c + 2)
   {
      Print("Not enough bars: ", rates_total, " (need ", c + 2, ")");
      return 0;
   }

   ArraySetAsSeries(open, false);
   ArraySetAsSeries(high, false);
   ArraySetAsSeries(low, false);
   ArraySetAsSeries(close, false);
   ArraySetAsSeries(time, false);

   double atr[];
   ArrayResize(atr, rates_total);
   ArraySetAsSeries(atr, false);
   if (CopyBuffer(atr_handle, 0, 0, rates_total, atr) != rates_total)
   {
      Print("Failed to copy ATR buffer");
      return 0;
   }

   // Heikin Ashi calculation
   if (calculate_ha)
   {
      ArrayResize(ha_open, rates_total);
      ArrayResize(ha_high, rates_total);
      ArrayResize(ha_low, rates_total);
      ArrayResize(ha_close, rates_total);
      ArraySetAsSeries(ha_open, false);
      ArraySetAsSeries(ha_high, false);
      ArraySetAsSeries(ha_low, false);
      ArraySetAsSeries(ha_close, false);

      for (int i = 0; i < rates_total; i++)
      {
         ha_close[i] = (open[i] + high[i] + low[i] + close[i]) / 4.0;
         if (i == 0)
         {
            ha_open[i] = open[i];
            ha_high[i] = high[i];
            ha_low[i] = low[i];
         }
         else
         {
            ha_open[i] = (ha_open[i - 1] + ha_close[i - 1]) / 2.0;
            ha_high[i] = MathMax(high[i], MathMax(ha_open[i], ha_close[i]));
            ha_low[i] = MathMin(low[i], MathMin(ha_open[i], ha_close[i]));
         }
      }
   }

   double trail[];
   double pos_arr[];
   ArrayResize(trail, rates_total);
   ArrayResize(pos_arr, rates_total);
   ArraySetAsSeries(trail, false);
   ArraySetAsSeries(pos_arr, false);
   ArrayInitialize(trail, EMPTY_VALUE);
   ArrayInitialize(pos_arr, 0.0);

   // Initialize first trailing stop value
   trail[0] = calculate_ha ? ha_close[0] : close[0];

   // Trailing stop and position calculation
   for (int i = 1; i < rates_total; i++)
   {
      double src = calculate_ha ? ha_close[i] : close[i];
      double src1 = calculate_ha ? ha_close[i - 1] : close[i - 1];
      double nLoss = a * atr[i];

      if (src > trail[i - 1] && src1 > trail[i - 1])
         trail[i] = MathMax(trail[i - 1], src - nLoss);
      else if (src < trail[i - 1] && src1 < trail[i - 1])
         trail[i] = MathMin(trail[i - 1], src + nLoss);
      else if (src > trail[i - 1])
         trail[i] = src - nLoss;
      else
         trail[i] = src + nLoss;

      if (src1 < trail[i - 1] && src > trail[i - 1])
         pos_arr[i] = 1.0;
      else if (src1 > trail[i - 1] && src < trail[i - 1])
         pos_arr[i] = -1.0;
      else
         pos_arr[i] = pos_arr[i - 1];

      Print("Bar ", i, ": src=", src, ", trail=", trail[i], ", pos=", pos_arr[i]);
   }

   ArrayResize(BuyBuffer, rates_total);
   ArrayResize(SellBuffer, rates_total);
   ArrayResize(TrailBuffer, rates_total);
   ArrayInitialize(BuyBuffer, EMPTY_VALUE);
   ArrayInitialize(SellBuffer, EMPTY_VALUE);
   ArrayInitialize(TrailBuffer, EMPTY_VALUE);

   // Signal and trailing stop plotting
   for (int i = 1; i < rates_total; i++)
   {
      double src = calculate_ha ? ha_close[i] : close[i];
      double src1 = calculate_ha ? ha_close[i - 1] : close[i - 1];
      double trail_curr = trail[i];
      double trail1 = trail[i - 1];

      bool above = (src > trail_curr) && (src1 <= trail1);
      bool below = (trail_curr > src) && (trail1 <= src1);

      bool buy_signal = above; // Simplified to ensure signals
      bool sell_signal = below;

      if (buy_signal)
      {
         BuyBuffer[i] = low[i] - Point() * 10; // Offset for visibility
         Print("Buy signal at bar ", i, ", time: ", TimeToString(time[i]), ", low: ", low[i]);
      }
      if (sell_signal)
      {
         SellBuffer[i] = high[i] + Point() * 10; // Offset for visibility
         Print("Sell signal at bar ", i, ", time: ", TimeToString(time[i]), ", high: ", high[i]);
      }

      TrailBuffer[i] = trail[i];
      if (trail[i] != EMPTY_VALUE)
         Print("Trailing stop at bar ", i, ": ", trail[i]);
   }

   // Alerts on the latest bar
   if (rates_total > 1)
   {
      int i = rates_total - 1;
      double src = calculate_ha ? ha_close[i] : close[i];
      double src1 = calculate_ha ? ha_close[i - 1] : close[i - 1];
      double trail_curr = trail[i];
      double trail1 = trail[i - 1];

      bool above = (src > trail_curr) && (src1 <= trail1);
      bool below = (trail_curr > src) && (trail1 <= src1);

      bool buy_signal = above;
      bool sell_signal = below;

      static datetime last_time = 0;
      if (time[i] != last_time)
      {
         if (buy_signal)
         {
            Alert("UT Long at ", TimeToString(time[i]));
            Print("UT Long alert triggered at ", TimeToString(time[i]));
         }
         if (sell_signal)
         {
            Alert("UT Short at ", TimeToString(time[i]));
            Print("UT Short alert triggered at ", TimeToString(time[i]));
         }
         last_time = time[i];
      }
   }

   return rates_total;
}