//+------------------------------------------------------------------+
//|                                                    dmd_utils.mqh |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<np.mqh>
#include<cmath.mqh>
//+------------------------------------------------------------------+
//|  pseudo hankelization                                            |
//+------------------------------------------------------------------+
matrix time_delay_embedding(matrix &data, ulong delay=2)
  {

   if(delay>=data.Cols() || delay<1)
     {
      Print(__FUNCTION__, " invalid input parameters ");
      return matrix::Zeros(0,0);
     }

   ulong rows = data.Rows();
   matrix matrices;

   matrix out(rows*delay,data.Cols()-(delay-1));

   for(ulong i = 0; i<delay; i++)
     {
      matrices = np::sliceMatrixCols(data,long(i),long(i+data.Cols()-(delay-1)));
      if(!np::matrixCopyRows(out,matrices,long(i*rows),long((i*rows)+matrices.Rows())))
        {
         Print(__FUNCTION__, " failed copy operation ");
         return matrix::Zeros(0,0);
        }
     }

   return out;
  }
//+------------------------------------------------------------------+
//| rebuild original shape of data after augmentation                |
//+------------------------------------------------------------------+
matrix time_delay_reconstruction(matrix &data, ulong delay)
  {

   ulong rows = data.Rows();
   ulong cols = data.Cols();

   if(rows<delay)
     {
      Print(__FUNCTION__," invalid inputs");
      return data;
     }

   matrix out(rows/delay,cols + delay - 1);

   matrix splitted[];

   ulong splits = data.Hsplit(delay,splitted);

   for(ulong i = 0; i<splits; i++)
      if(!np::matrixCopyCols(out,splitted[i],long(i),long(i+cols)))
        {
         Print(__FUNCTION__, " failed copy operation ");
         return data;
        }

   return out;
  }

//+------------------------------------------------------------------+
//| DMD wrapper class for built in DMD methods                       |
//+------------------------------------------------------------------+
class CDmd
  {
protected:
   ulong             m_delay;
   bool              m_fitted;
   matrixc           m_modes;
   bool              m_exact_modes;
   matrixc           m_eigenvectors;
   vectorc           m_eigs;
   vectorc           m_amplitudes;
   matrix            m_snapshots_x;
   matrix            m_snapshots_y;
   matrix            m_left_vectors;
   matrixc           m_residual_vectors;
   vector            m_residuals;
   vector            m_og_timesteps;
   vector            m_dmd_timesteps;
   double            m_og_first,m_dmd_first;
   ulong             m_og_last,m_dmd_last;
   double            m_og_dt, m_dmd_dt;

   void              reset(void)
     {
      m_fitted = m_exact_modes = false;
      m_delay = 0;
      m_modes = m_eigenvectors = m_residual_vectors = matrixc::Zeros(0,0);
      m_eigs = m_amplitudes = vectorc::Zeros(0);
      m_snapshots_x = m_snapshots_y = m_left_vectors = matrix::Zeros(0,0);
      m_dmd_timesteps = m_og_timesteps = m_residuals = vector::Zeros(0);
      m_og_dt = m_dmd_dt = m_og_first = m_dmd_first = 0.0;
      m_og_last = m_dmd_last = 0;
     }

   bool              compute_amplitudes(void)
     {
      if(!m_modes.Cols())
         return false;

      vectorc x1 = as_complex(m_snapshots_x.Col(0));
      matrixc A_h = m_modes.TransposeConjugate();
      matrixc B = np::matmul(A_h,m_modes);
      vectorc c = np::matmul(A_h,x1);
      complex right_side[], solution[];
      int pivots[];
      CMatrixComplex lua = B;

      if(!np::vecAsArray(c,right_side))
        {
         Print(__FUNCTION__, " vector as array failure ");
         return false;
        }

      vector check_pivots;
      ResetLastError();
      CAlglib::CMatrixLU(lua,lua.Rows(),lua.Cols(),pivots);
      int info;
      check_pivots.Assign(pivots);
      if(check_pivots.Max() == check_pivots.Min())
        {
         Print(__FUNCTION__," CAlglib::CMatrixLU()  failure ", GetLastError());
         return false;
        }

      CDenseSolverReportShell shell;

      CAlglib::CMatrixLUSolve(lua,pivots,int(pivots.Size()),right_side,info,shell,solution);

      if(info < 1)
        {
         Print(__FUNCTION__, " m_modes matrix is singular, or VERY close to singular. ", GetLastError());
         return false;
        }

      return m_amplitudes.Assign(solution);

     }

public:
                     CDmd(void)
     {
      reset();
     }
   virtual bool              fit(matrix &X,double time_delta, double tolerance, long svd_rank = -1,ulong embedding_dimension = 0,ENUM_DMD_SCALE scale = DMDSCALE_N,ENUM_SVD_ALG svd_method = SVDALG_1, ENUM_DMD_EIGV eigv = DMDEIGV_V,ENUM_DMD_RESIDUALS resid = DMDRESIDUALS_R,ENUM_DMD_REFINE refine = DMDREFINE_R)
     {
      reset();

      matrix z,res_vectors,b,w,s;
      vector residuals;
      m_delay = embedding_dimension;
      matrix snapshots = m_delay>1?time_delay_embedding(X,m_delay):X;
      if(!snapshots.Cols())
         return false;
      m_snapshots_x = np::sliceMatrixCols(snapshots,0,snapshots.Cols()-1);
      m_snapshots_y = np::sliceMatrixCols(snapshots,1);
      if(!m_snapshots_x.DynamicModeDecomposition(m_snapshots_y,scale,eigv,resid,refine,svd_method,svd_rank,tolerance,m_eigs,m_left_vectors,z,m_residuals,res_vectors,b,w,s))
        {
         Print(__FUNCTION__, " DMD error ", GetLastError());
         return false;
        }

      if(eigv != DMDEIGV_N)
         m_modes = compact_to_complex(z,m_eigs);

      m_eigenvectors = compact_to_complex(w,m_eigs);

      if(refine == DMDREFINE_E)
        {
         m_modes = compact_to_complex(b,m_eigs);
         m_exact_modes = true;
        }
      else
         if(refine == DMDREFINE_R)
            m_residual_vectors = compact_to_complex(res_vectors,m_eigs);

      m_og_first = m_dmd_first = 0.0;
      m_og_dt = m_dmd_dt = time_delta;
      m_og_timesteps = m_dmd_timesteps = np::arange(m_snapshots_x.Cols()+1,m_dmd_first,m_dmd_dt);
      m_og_last = m_dmd_last = m_dmd_timesteps.Size()-1;

      m_fitted = compute_amplitudes() ;

      return m_fitted;
     }
   matrixc           dmd_modes(void)
     {
      return m_modes;
     }
   double            dmd_time_delta(void)
     {
      return m_dmd_dt;
     }
   double            dmd_first_time(void)
     {
      return m_dmd_first;
     }
   ulong             dmd_last_time(void)
     {
      return m_dmd_last;
     }
   double            og_first_time(void)
     {
      return m_og_first;
     }
   ulong             og_last_time(void)
     {
      return m_og_last;
     }
   vector            dmd_time_steps(void)
     {
      return m_dmd_timesteps;
     }
   vector            og_time_steps(void)
     {
      return m_og_timesteps;
     }
   vectorc           dmd_eigs(void)
     {
      return m_eigs;
     }
   vectorc           dmd_amplitudes(void)
     {
      return m_amplitudes;
     }
   matrixc           exact_dmd_modes(void)
     {
      if(m_exact_modes)
         return m_modes;
      else
         return matrixc::Zeros(0,0);
     }
   matrixc           eigenvectors(void)
     {
      return m_eigenvectors;
     }
   vector            growth_rate(void)
     {
      return real(m_eigs)/m_dmd_dt;
     }

   vector            frequency(void)
     {
      vectorc f = c_log(m_eigs);
      return imaginary(f)/(2.0*M_PI*m_dmd_dt);
     }
   matrixc           residual_vectors(void)
     {
      return m_residual_vectors;
     }
   vector            residuals(void)
     {
      return m_residuals;
     }
   matrixc           dynamics(void)
     {
      matrixc mat_eigs = np::repeat_vector_as_rows_cols(m_eigs,m_dmd_timesteps.Size(),false);
      vector tpow = (m_dmd_timesteps - m_og_first)/m_og_dt;
      matrixc dyn = mat_eigs;

      for(ulong i = 0; i<mat_eigs.Cols(); i++)
        {
         vectorc column = mat_eigs.Col(i);
         column = c_pow(column,tpow[i]);
         column*=m_amplitudes;
         dyn.Col(column,i);
        }

      return dyn;
     }
   matrix            reconstructed_data(ulong num_future_steps = 0)
     {
      if(num_future_steps)
        {
         m_dmd_last+=num_future_steps;
         m_dmd_timesteps = np::arange(m_dmd_last+1,m_dmd_first,m_dmd_dt);
        }

      matrixc d = dynamics();
      matrixc data = np::matmul(m_modes,d);
      if(!num_future_steps)
         data = np::sliceMatrixCols(data,0,long(m_og_last+1));
      matrix rdata = real(data);
      if(m_delay>1)
         rdata = time_delay_reconstruction(rdata,m_delay);
      return rdata;
     }

   matrix            reconstructed_data(vector& modes_mask, ulong num_future_steps = 0)
     {
      if(num_future_steps)
        {
         m_dmd_last+=num_future_steps;
         m_dmd_timesteps = np::arange(m_dmd_last+1,m_dmd_first,m_dmd_dt);
        }

      matrixc d = dynamics();
      matrixc masked_modes = matrixc::Zeros(m_modes.Rows(),m_modes.Cols());
      if(modes_mask.Size() == m_modes.Cols())
        {
         for(ulong i = 0; i<masked_modes.Cols(); i++)
            if(modes_mask[i])
               masked_modes.Col(m_modes.Col(i),i);
        }
      matrixc data = np::matmul(masked_modes,d);
      if(!num_future_steps)
         data = np::sliceMatrixCols(data,0,long(m_og_last+1));
      matrix rdata = real(data);
      if(m_delay>1)
         rdata = time_delay_reconstruction(rdata,m_delay);
      return rdata;
     }
  };

//+------------------------------------------------------------------+
//| plot the eigenvalues                                             |
//+------------------------------------------------------------------+
void plot_eigs(vectorc& evals,bool points_fill = true,long chart_id=0, int sub_win=0,int x1=0, int y1=0, int x2=750, int y2=500, bool chart_show=true, int duration = 10)
  {
   vector re_evals = real(evals);
   vector im_evals = imaginary(evals);

   vector theta = np::linspace(0.0,2.0*M_PI,100);
   vector x = cos(theta);
   vector y = sin(theta);


   double xa[],ya[],xb[],yb[];

   if(x.Size()!=y.Size() || !np::vecAsArray(x,xa) || !np::vecAsArray(y,ya))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   if(!np::vecAsArray(re_evals,xb) || !np::vecAsArray(im_evals,yb))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   CGraphic graph;
   ChartRedraw();
   string plot_name = "Eigevalues";
   ChartSetInteger(chart_id, CHART_SHOW, chart_show);

   if(!graph.Create(chart_id, plot_name, sub_win, x1, y1, x2, y2))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return ;
     }

   CColorGenerator generator;

   CCurve* circle = graph.CurveAdd(xa, ya, generator.Next(),CURVE_LINES,"Unit Circle");
   CCurve* egs    = graph.CurveAdd(xb, yb, generator.Next(),CURVE_POINTS,"Eigs");

   circle.LinesWidth(1);
   egs.PointsFill(true);
   egs.LinesWidth(3);

   graph.XAxis().Name("Real");
   graph.XAxis().NameSize(13);
   graph.YAxis().Name("Imaginary");
   graph.YAxis().NameSize(13);
   graph.FontSet("Lucida Console", 13);
   graph.CurvePlotAll();
   graph.Update();

   Sleep(duration*1000);
   graph.Destroy();
   ChartRedraw();
   return ;

  }
//+------------------------------------------------------------------+
//| reconstruction                                                   |
//+------------------------------------------------------------------+
void plot_reconstructed(vector& reconstructed_series,vector& input_series, vector& time,string plot_name, long chart_id=0, int sub_win=0,int x1=0, int y1=0, int x2=750, int y2=500, bool chart_show=true,int duration = 10)
  {
   if(reconstructed_series.Size() != time.Size()|| input_series.Size()!=time.Size())
     {
      Print(__FUNCTION__, " Invalid inputs ");
      return;
     }

   double xa[],ya[],yb[];

   if(!np::vecAsArray(time,xa) || !np::vecAsArray(input_series,ya) || !np::vecAsArray(reconstructed_series,yb))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   CGraphic graph;
   ChartRedraw();
   ChartSetInteger(chart_id, CHART_SHOW, chart_show);

   if(!graph.Create(chart_id, plot_name, sub_win, x1, y1, x2, y2))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   CColorGenerator generator;
   graph.HistoryNameWidth(100);
   graph.HistoryNameSize(13);
   CCurve* input_curve = graph.CurveAdd(xa, ya, clrBlue,CURVE_LINES,"Input");
   input_curve.LinesWidth(4);
   input_curve.LinesStyle(STYLE_SOLID);

   CCurve* rec_curve = graph.CurveAdd(xa,yb,clrBlack,CURVE_POINTS,"DMD reconstruction");
   rec_curve.LinesWidth(2);
   rec_curve.PointsFill(true);

   graph.XAxis().Name("Time");
   graph.XAxis().NameSize(13);
   graph.YAxis().Name("Y values");
   graph.YAxis().NameSize(13);

   graph.FontSet("Lucida Console", 13);
   graph.CurvePlotAll();
   graph.Update();

   Sleep(duration*1000);
   graph.Destroy();
   ChartRedraw();
   return ;

  }
//+------------------------------------------------------------------+
//| plot mode dynamics                                               |
//+------------------------------------------------------------------+
void plot_mode_dynamic(vectorc& mode_dynamic,vector& time, string plot_name, long chart_id=0, int sub_win=0,int x1=0, int y1=0, int x2=750, int y2=500, bool chart_show=true,int line_width = 3, ENUM_CURVE_TYPE curve_type=CURVE_LINES,int duration = 10)
  {
   if(time.Size()!=mode_dynamic.Size())
     {
      Print(__FUNCTION__, " Invalid inputs ");
      return;
     }
   vector re = real(mode_dynamic);


   double xa[],ya[];

   if(!np::vecAsArray(time,xa) || !np::vecAsArray(re,ya))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   CGraphic graph;
   ChartRedraw();
   ChartSetInteger(chart_id, CHART_SHOW, chart_show);

   if(!graph.Create(chart_id, plot_name, sub_win, x1, y1, x2, y2))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   graph.HistoryNameWidth(100);
   graph.HistoryNameSize(13);
   CColorGenerator generator;

   CCurve* curve = graph.CurveAdd(xa, ya, generator.Next(),curve_type,plot_name);
   curve.PointsFill(true);
   curve.LinesWidth(line_width);

   graph.XAxis().Name("Time");
   graph.XAxis().NameSize(13);
   graph.YAxis().Name("Spatial");
   graph.YAxis().NameSize(13);
   graph.FontSet("Lucida Console", 13);
   graph.CurvePlotAll();
   graph.Update();
   Sleep(duration*1000);
   graph.Destroy();
   ChartRedraw();
   return ;
  }
//+------------------------------------------------------------------+
