//+------------------------------------------------------------------+
//|                                            DMD_Forecast_Demo.mq5 |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#property script_show_inputs
//---
#include<np.mqh>
#include<cmath.mqh>
//---
input ENUM_DMD_SCALE jobs = DMDSCALE_N;
input ENUM_DMD_EIGV jobz = DMDEIGV_V;
input ENUM_DMD_RESIDUALS jobr = DMDRESIDUALS_R;
input ENUM_DMD_REFINE jobf = DMDREFINE_R;
input ENUM_SVD_ALG whtsvd = SVDALG_1;
input long nrnk = -1;
input double tol_ = 1.e-9;//tol
input ulong Delay = 50;//apply time delay embedding
input ulong Num_future_points = 10;
//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
   ulong series_len = 100;
   double dt = 0.1010101;
   double first_time_point = 0.0;
   vector t = np::arange(series_len,first_time_point,dt);
   vector F = f(t);
   matrix X(1,series_len);
   X.Row(F,0);
//---
   if(Delay)
      X = time_delay_embedding(X,Delay);
//---
   Print("X shape is ", X.Rows(),":",X.Cols());
//---
   matrix X1 = np::sliceMatrixCols(X,0,X.Cols()-1);
   matrix X2 = np::sliceMatrixCols(X,1);
//---
//---
   vectorc eigen_values;
   matrix W;
   matrix B;
   vector residuals;
   matrix left_vectors;
   matrix res_vectors;
   matrix Z,S;

   if(!X1.DynamicModeDecomposition(X2,jobs,jobz,jobr,jobf,whtsvd,nrnk,tol_,eigen_values,left_vectors,Z,residuals,res_vectors,B,W,S))
     {
      Print(" DMD error ",GetLastError());
      return;
     }
//---
   matrixc dmd_modes = compact_to_complex(Z,eigen_values);
//---
   vectorc dmd_amplitudes = compute_amplitudes(X,dmd_modes);
//---
   vector abs_dmd_amplitudes = c_abs(dmd_amplitudes);
//---
   long amp_indices[];
   if(!np::arange(amp_indices,int(abs_dmd_amplitudes.Size())) ||
      !np::quickSortIndices(abs_dmd_amplitudes,false,amp_indices,amp_indices[0],amp_indices[amp_indices.Size()-1]))
     {
      Print(" error ");
      return;
     }
//---
   vector time_steps = np::arange((X1.Cols())+Num_future_points+1,first_time_point,dt);
//---
   matrixc dmd_dynamics = compute_dynamics(dmd_amplitudes,eigen_values,time_steps);
//---
   matrixc reconstruction = reconstructed_data(dmd_dynamics,dmd_modes);
//---
   matrix forecast = real(reconstruction);
   forecast = time_delay_reconstruction(forecast,Delay);
   vector forecast_time = np::arange(series_len+Num_future_points,first_time_point,dt);
//---
   vector actual = f(forecast_time);
//---
   plot_forecast(actual,forecast.Row(0),forecast_time, Num_future_points,"Forecast Demo",0,0,0,0,750,400,true,15);
  }
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| univariate process                                               |
//+------------------------------------------------------------------+
vector f(vector &in)
  {
   return cos(in)*sin(cos(in))*cos(in*0.2);
  }
//+------------------------------------------------------------------+
//|  compute the DMD amplitudes                                      |
//+------------------------------------------------------------------+
vectorc              compute_amplitudes(matrix& snapshots, matrixc& modes)
  {
   vectorc amplitudes(0);

   vectorc x0 = as_complex(snapshots.Col(0));
   matrixc A_h = modes.TransposeConjugate();
   matrixc B = np::matmul(A_h,modes);
   vectorc c = np::matmul(A_h,x0);
   complex right_side[], solution[];
   int pivots[];
   CMatrixComplex lua = B;

   if(!np::vecAsArray(c,right_side))
     {
      Print(__FUNCTION__, " vector as array failure ");
      return amplitudes;
     }

   vector check_pivots;
   ResetLastError();
   CAlglib::CMatrixLU(lua,lua.Rows(),lua.Cols(),pivots);
   int info;
   check_pivots.Assign(pivots);
   if(check_pivots.Max() == check_pivots.Min())
     {
      Print(__FUNCTION__," CAlglib::CMatrixLU()  failure ", GetLastError());
      return amplitudes;
     }

   CDenseSolverReportShell shell;

   CAlglib::CMatrixLUSolve(lua,pivots,int(pivots.Size()),right_side,info,shell,solution);

   if(info < 1)
     {
      Print(__FUNCTION__, " modes matrix is singular, or VERY close to singular. ", GetLastError());
      return amplitudes;
     }

   amplitudes.Assign(solution);

   return amplitudes;

  }
//+------------------------------------------------------------------+
//|   calculate the mode dynamics                                    |
//+------------------------------------------------------------------+
matrixc           compute_dynamics(vectorc& amplitudes,vectorc& eigs, vector& timesteps)
  {
   matrixc mat_eigs = np::repeat_vector_as_rows_cols(eigs,timesteps.Size(),false);
   vector tpow = (timesteps - timesteps[0])/(timesteps[1] - timesteps[0]);
   matrixc dyn = mat_eigs;

   for(ulong i = 0; i<mat_eigs.Cols(); i++)
     {
      vectorc column = mat_eigs.Col(i);
      column = c_pow(column,tpow[i]);
      column*=amplitudes;
      dyn.Col(column,i);
     }

   return dyn;
  }
//+------------------------------------------------------------------+
//|  reconstruct the input data based on the dominant dmd dynamics   |
//+------------------------------------------------------------------+
matrixc            reconstructed_data(matrixc& dynamics, matrixc& modes)
  {
   matrixc rec = np::matmul(modes,dynamics);
   return rec;
  }
//+------------------------------------------------------------------+
//| plot forecasts evaluation                                        |
//+------------------------------------------------------------------+
void plot_forecast(vector& actual,vector& forecast,vector& time,ulong num_forecasts,string plot_name, long chart_id=0, int sub_win=0,int x1=0, int y1=0, int x2=750, int y2=500, bool chart_show=true, int duration = 10)
  {
   if(time.Size()!=actual.Size() || forecast.Size() != time.Size())
     {
      Print(__FUNCTION__, " Invalid inputs ");
      return;
     }
   
   double xa[],ya[],yb[],xc[],yc[];

   if(!np::vecAsArray(time,xa) || !np::vecAsArray(actual,ya) || !np::vecAsArray(forecast,yb))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }
   for(ulong i = time.Size()-num_forecasts;i<time.Size(); i++)
     {
      xc.Push(time[i]);
      yc.Push(forecast[i]);
     }

   CGraphic graph;
   ChartRedraw();
   ChartSetInteger(chart_id, CHART_SHOW, chart_show);

   if(!graph.Create(chart_id, plot_name, sub_win, x1, y1, x2, y2))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   CColorGenerator generator;
   graph.HistoryNameWidth(100);
   graph.HistoryNameSize(13);
   CCurve* actual_curve = graph.CurveAdd(xa, ya, clrYellow,CURVE_LINES,"Actual");
   actual_curve.LinesWidth(6);
   actual_curve.LinesStyle(STYLE_SOLID);

   CCurve* forecast_curve = graph.CurveAdd(xa,yb,clrBlack,CURVE_LINES,"DMD Reconstruction");
   forecast_curve.LinesStyle(STYLE_DOT);
   forecast_curve.LinesWidth(4);
   
   CCurve* forecasts = graph.CurveAdd(xc,yc,clrBlue,CURVE_LINES,"DMD forecast");
   forecasts.LinesStyle(STYLE_DOT);
   forecasts.LinesWidth(4);

   graph.XAxis().Name("Time");
   graph.XAxis().NameSize(13);
   graph.YAxis().Name("Y values");
   graph.YAxis().NameSize(13);

   graph.FontSet("Lucida Console", 13);
   graph.CurvePlotAll();
   graph.Update();

   Sleep(duration*1000);
   graph.Destroy();
   ChartRedraw();
   return ;

  }
//+------------------------------------------------------------------+
//|  pseudo hankelization                                            |
//+------------------------------------------------------------------+
matrix time_delay_embedding(matrix &data, ulong delay=2)
  {

   if(delay>=data.Cols() || delay<1)
     {
      Print(__FUNCTION__, " invalid input parameters ");
      return matrix::Zeros(0,0);
     }

   ulong rows = data.Rows();
   matrix matrices;

   matrix out(rows*delay,data.Cols()-(delay-1));

   for(ulong i = 0; i<delay; i++)
     {
      matrices = np::sliceMatrixCols(data,long(i),long(i+data.Cols()-(delay-1)));
      if(!np::matrixCopyRows(out,matrices,long(i*rows),long((i*rows)+matrices.Rows())))
        {
         Print(__FUNCTION__, " failed copy operation ");
         return matrix::Zeros(0,0);
        }
     }

   return out;
  }
//+------------------------------------------------------------------+
//| rebuild original shape of data after augmentation                |
//+------------------------------------------------------------------+
matrix time_delay_reconstruction(matrix &data, ulong delay)
  {

   ulong rows = data.Rows();
   ulong cols = data.Cols();

   if(rows<delay)
     {
      Print(__FUNCTION__," invalid inputs");
      return matrix::Zeros(0,0);
     }

   matrix out(rows/delay,cols + delay - 1);

   matrix splitted[];

   ulong splits = data.Hsplit(delay,splitted);

   for(ulong i = 0; i<splits; i++)
      if(!np::matrixCopyCols(out,splitted[i],long(i),long(i+cols)))
        {
         Print(__FUNCTION__, " failed copy operation ");
         return matrix::Zeros(0,0);
        }

   return out;
  }
//+------------------------------------------------------------------+
