//+------------------------------------------------------------------+
//|                                                     DMD_Demo.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#property script_show_inputs
//---
#include<np.mqh>
#include<cmath.mqh>
//---
input ENUM_DMD_SCALE jobs = DMDSCALE_N;
input ENUM_DMD_EIGV jobz = DMDEIGV_V;
input ENUM_DMD_RESIDUALS jobr = DMDRESIDUALS_R;
input ENUM_DMD_REFINE jobf = DMDREFINE_R;
input ENUM_SVD_ALG whtsvd = SVDALG_1;
input long nrnk = -1;
input double tol_ = 1.e-9;//tol
input ulong Delay = 50;//apply time delay embedding
//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
//---
   ulong series_len = 100;
   double dt = 0.1010101;
   double first_time_point = 0.0;
   vector t = np::arange(series_len,first_time_point,dt);
   vector F = f(t);
   matrix X(1,series_len);
   X.Row(F,0);
//---
   if(Delay)
      X = time_delay_embedding(X,Delay);
//---
   Print("X shape is ", X.Rows(),":",X.Cols());
//---
   matrix X1 = np::sliceMatrixCols(X,0,X.Cols()-1);
   matrix X2 = np::sliceMatrixCols(X,1);
//---
   vectorc eigen_values;
   matrix W;
   matrix B;
   vector residuals;
   matrix left_vectors;
   matrix res_vectors;
   matrix Z,S;

   if(!X1.DynamicModeDecomposition(X2,jobs,jobz,jobr,jobf,whtsvd,nrnk,tol_,eigen_values,left_vectors,Z,residuals,res_vectors,B,W,S))
     {
      Print(" DMD error ",GetLastError());
      return;
     }
//---
   plot_eigs(eigen_values,true,0,0,0,0,500,400,true,15);
  }
//+------------------------------------------------------------------+
//| univariate process                                               |
//+------------------------------------------------------------------+
vector f(vector &in)
  {
   return cos(in)*sin(cos(in))*cos(in*0.2);
  }
//+------------------------------------------------------------------+
//|  pseudo hankelization                                            |
//+------------------------------------------------------------------+
matrix time_delay_embedding(matrix &data, ulong delay=2)
  {

   if(delay>=data.Cols() || delay<1)
     {
      Print(__FUNCTION__, " invalid input parameters ");
      return matrix::Zeros(0,0);
     }

   ulong rows = data.Rows();
   matrix matrices;

   matrix out(rows*delay,data.Cols()-(delay-1));

   for(ulong i = 0; i<delay; i++)
     {
      matrices = np::sliceMatrixCols(data,long(i),long(i+data.Cols()-(delay-1)));
      if(!np::matrixCopyRows(out,matrices,long(i*rows),long((i*rows)+matrices.Rows())))
        {
         Print(__FUNCTION__, " failed copy operation ");
         return matrix::Zeros(0,0);
        }
     }

   return out;
  }
//+------------------------------------------------------------------+
//| rebuild original shape of data after augmentation                |
//+------------------------------------------------------------------+
matrix time_delay_reconstruction(matrix &data, ulong delay)
  {

   ulong rows = data.Rows();
   ulong cols = data.Cols();

   if(rows<delay)
     {
      Print(__FUNCTION__," invalid inputs");
      return matrix::Zeros(0,0);
     }

   matrix out(rows/delay,cols + delay - 1);

   matrix splitted[];

   ulong splits = data.Hsplit(delay,splitted);

   for(ulong i = 0; i<splits; i++)
      if(!np::matrixCopyCols(out,splitted[i],long(i),long(i+cols)))
        {
         Print(__FUNCTION__, " failed copy operation ");
         return matrix::Zeros(0,0);
        }

   return out;
  }
//+------------------------------------------------------------------+
//| plot the eigenvalues                                             |
//+------------------------------------------------------------------+
void plot_eigs(vectorc& evals,bool points_fill = true,long chart_id=0, int sub_win=0,int x1=0, int y1=0, int x2=750, int y2=500, bool chart_show=true, int duration = 10)
  {
   vector re_evals = real(evals);
   vector im_evals = imaginary(evals);

   vector theta = np::linspace(0.0,2.0*M_PI,100);
   vector x = cos(theta);
   vector y = sin(theta);


   double xa[],ya[],xb[],yb[];

   if(x.Size()!=y.Size() || !np::vecAsArray(x,xa) || !np::vecAsArray(y,ya))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   if(!np::vecAsArray(re_evals,xb) || !np::vecAsArray(im_evals,yb))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return;
     }

   CGraphic graph;
   ChartRedraw();
   string plot_name = "Eigevalues";
   ChartSetInteger(chart_id, CHART_SHOW, chart_show);

   if(!graph.Create(chart_id, plot_name, sub_win, x1, y1, x2, y2))
     {
      Print(__FUNCTION__," Failed to Create graphical object on the Main chart Err ", GetLastError());
      return ;
     }

   CColorGenerator generator;

   CCurve* circle = graph.CurveAdd(xa, ya, generator.Next(),CURVE_LINES,"Unit Circle");
   CCurve* egs    = graph.CurveAdd(xb, yb, generator.Next(),CURVE_POINTS,"Eigs");
   circle.LinesStyle(STYLE_DASH);
   circle.LinesWidth(1);
   egs.PointsFill(true);
   egs.LinesWidth(3);

   graph.XAxis().Name("Real");
   graph.XAxis().NameSize(13);
   graph.YAxis().Name("Imaginary");
   graph.YAxis().NameSize(13);
   graph.FontSet("Lucida Console", 13);
   graph.CurvePlotAll();
   graph.Update();

   Sleep(duration*1000);
   graph.Destroy();
   ChartRedraw();
   return ;

  }
//+------------------------------------------------------------------+
