//+------------------------------------------------------------------+
//|                                                  ProxyExport.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#property script_show_inputs

#include <Trade\TerminalInfo.mqh>
#include <Trade\AccountInfo.mqh>
#include <Trade\DealInfo.mqh>
#include <Trade\HistoryOrderInfo.mqh>
#include <Generic\HashMap.mqh>

input group "choose to export terminal info";
input bool exportTerminalInformations = false; //export terminal informations?
input string terminalInfoName = "Terminal_Info"; //terminal info file or sheet name

input group "choose to export history deals";
input bool exportHistoricalDeals = false; //export historical deals?
input string historyDealsName = "History_Deals"; //history deals file or sheet name

input group "choose to export history orders";
input bool exportHistoricalOrders = false; //export historical orders?
input string historyOrdersName = "History_Orders"; //history orders file or sheet name

input group "choose to export account info";
input bool exportAccountInfos = false; //export account informations?
input string accountInfoName = "Account_Info"; //account info file or sheet name

input group "output to Google Sheet";
input string proxyServerUrl = ""; //valid proxy server url
input string spreadSheetId = ""; //valid spread sheet id

CHashMap<int, string> codePageLanguage;

// storing code page language number to code in hashmap
void initiateCodePageLanguage() {
   codePageLanguage.Add(0, "CP_ACP");
   codePageLanguage.Add(1, "CP_OEMCP");
   codePageLanguage.Add(2, "CP_MACCP");
   codePageLanguage.Add(3, "CP_THREAD_ACP");
   codePageLanguage.Add(42, "CP_SYMBOL");
   codePageLanguage.Add(65000, "CP_UTF7");
   codePageLanguage.Add(65001, "CP_UTF8");
}

string GetTerminalInfo() {
   initiateCodePageLanguage();
   string terminalCodePageLang;
   CTerminalInfo terminalInfo;
   bool terminalCodePageLangFound = codePageLanguage.TryGetValue(terminalInfo.CodePage(), terminalCodePageLang);
   string terminal_infos = "[\"Terminal_Infos\", \"Values\"]";
   string terminalInformations = StringFormat(
      ", [\"build number\", \"%d\"]" +
      ", [\"trade server connected\", \"%s\"]" +
      ", [\"DLLs are allowed\", \"%s\"]" + 
      ", [\"trading enabled\", \"%s\"]" +
      ", [\"email enabled\", \"%s\"]" + 
      ", [\"ftp enabled\", \"%s\"]" +
      ", [\"max bars allowed in chart\", \"%d\"]" +
      ", [\"code page language\", \"%s\"]" +
      ", [\"cpu cores used\", \"%d\"]" +
      ", [\"physical memory\", \"%d MB\"]" +
      ", [\"memory available\", \"%d MB\"]" +
      ", [\"memory used\", \"%d MB\"]" +
      ", [\"memory total\", \"%d MB\"]" +
      ", [\"system is x64\", \"%s\"]" +
      ", [\"OpenCl version\", \"%s\"]" +
      ", [\"free disk space\", \"%d MB\"]" +
      ", [\"terminal language\", \"%s\"]" +
      ", [\"common data folder of all terminals, installed on the computer\", \"%s\"]" +
      ", [\"data folder of the terminal\", \"%s\"]" +
      ", [\"folder of the client terminal\", \"%s\"]" +
      ", [\"company name of the terminal\", \"%s\"]" +
      ", [\"name of the terminal\", \"%s\"]"
      ,
      terminalInfo.Build(),
      terminalInfo.IsConnected() ? "yes" : "no",
      terminalInfo.IsDLLsAllowed() ? "yes" : "no",
      terminalInfo.IsTradeAllowed() ? "yes" : "no",
      terminalInfo.IsEmailEnabled() ? "yes" : "no",
      terminalInfo.IsFtpEnabled() ? "yes" : "no",
      terminalInfo.MaxBars(),
      terminalCodePageLangFound ? terminalCodePageLang : "not found",
      terminalInfo.CPUCores(),
      terminalInfo.MemoryPhysical(),
      terminalInfo.MemoryAvailable(),
      terminalInfo.MemoryUsed(),
      terminalInfo.MemoryTotal(),
      terminalInfo.IsX64() ? "yes" : "no",
      terminalInfo.OpenCLSupport() == 0 ? "OpenCL not supported" : IntegerToString(terminalInfo.OpenCLSupport()),
      terminalInfo.DiskSpace(),
      terminalInfo.Language(),
      terminalInfo.CommonDataPath(),
      terminalInfo.DataPath(),
      terminalInfo.Path(),
      terminalInfo.Company(),
      terminalInfo.Name()
   );
   StringAdd(terminal_infos, terminalInformations);
   return terminal_infos;
}

string GetAccountInfo() {
   CAccountInfo accountInfo;
   string account_info = "[\"symbol\", \"name\", \"currency\", \"company\", \"balance\", \"credit\", " +
      "\"profit\", \"equity\", \"margin\", \"login\", \"trade_mode\", \"leverage\", \"limit_orders\", \"margin_mode\"]";
   StringAdd(account_info, StringFormat(
      ", [\"%s\", \"%s\", \"%s\", \"%s\", \"%f\", \"%f\", \"%f\", \"%f\", \"%f\", \"%d\", \"%d\", \"%d\", \"%d\", \"%d\"]",
      Symbol(),
      accountInfo.Name(), accountInfo.Currency(), accountInfo.Company(),
      accountInfo.Balance(), accountInfo.Credit(), accountInfo.Profit(), accountInfo.Equity(), accountInfo.Margin(),
      accountInfo.Login(), accountInfo.TradeMode(), accountInfo.Leverage(), accountInfo.LimitOrders(),
      accountInfo.MarginMode()
   ));
   return account_info;
}

string GetHistoryDeals() {
   CDealInfo deal;
   HistorySelect(StringToTime("1970.01.01 09:00"), TimeCurrent());
   int totalDeals = HistoryDealsTotal();
   string deals = "[\"ticket\", \"symbol\", \"time\", \"price\", \"profit\", \"type\", \"volume\", \"comment\", " +
      "\"sl\", \"tp\", \"commission\", \"fee\", \"order_id\", \"position_id\", \"magic\"]";
   for(int i = 0; i < totalDeals; i++) {
      deal.SelectByIndex(i);
      ulong ticket = deal.Ticket();
      string dealInfo = StringFormat(
         ", [\"%d\", \"%s\", \"%s\", \"%f\", \"%f\", \"%s\", \"%d\", \"%s\", " +
         "\"%f\", \"%f\", \"%f\", \"%f\", \"%d\", \"%d\", \"%d\"]",
         ticket,
         deal.Symbol(),
         TimeToString(deal.Time()),
         deal.Price(),
         deal.Profit(),
         deal.Type() == 0 ? "buy" : "sell",
         deal.Volume(),
         deal.Comment(),
         HistoryDealGetDouble(ticket, DEAL_SL),
         HistoryDealGetDouble(ticket, DEAL_TP),
         HistoryDealGetDouble(ticket, DEAL_COMMISSION),
         HistoryDealGetDouble(ticket, DEAL_FEE),
         HistoryDealGetInteger(ticket, DEAL_ORDER),
         HistoryDealGetInteger(ticket, DEAL_POSITION_ID),
         HistoryDealGetInteger(ticket, DEAL_MAGIC)
      );
      StringAdd(deals, dealInfo);
   }
   return deals;
}

string GetHistoryOrders() {
   CHistoryOrderInfo order;
   HistorySelect(StringToTime("1970.01.01 09:00"), TimeCurrent());
   int totalOrders = HistoryOrdersTotal();
   string orders = "[\"ticket\", \"symbol\", \"time\", \"price\", \"type\", \"volume\", \"comment\", \"sl\", " +
      "\"tp\", \"state\", \"order_time_done\", \"order_expire\", \"position_id\", \"magic\"]";
   for(int i = 0; i < totalOrders; i++) {
      order.SelectByIndex(i);
      ulong ticket = order.Ticket();
      string orderInfo = StringFormat(
         ", [\"%d\", \"%s\", \"%s\", \"%f\", \"%s\", \"%d\", \"%s\", \"%f\", " +
         "\"%f\", \"%d\", \"%s\", \"%s\", \"%d\", \"%d\"]",
         ticket,
         order.Symbol(),
         TimeToString(order.TimeSetup()),
         order.PriceOpen(),
         order.Type() == 0 ? "buy" : "sell",
         order.VolumeInitial(),
         order.Comment(),
         order.StopLoss(),
         order.TakeProfit(),
         HistoryOrderGetInteger(ticket, ORDER_STATE),
         TimeToString(HistoryOrderGetInteger(ticket,ORDER_TIME_DONE)),
         TimeToString(HistoryOrderGetInteger(ticket, ORDER_TIME_EXPIRATION)),
         HistoryOrderGetInteger(ticket, ORDER_POSITION_ID),
         HistoryOrderGetInteger(ticket, ORDER_MAGIC)
      );
      StringAdd(orders, orderInfo);
   }
   return orders;
}

void WriteToGoogleSheet(string terminalInfos, string historicalDeals, string historicalOrders, string accountInfos) {
   if(terminalInfos != NULL) {
      StringReplace(terminalInfos, "\\", "/");
      SheetExporter(terminalInfoName, terminalInfos);
   }
   if(historicalDeals != NULL) {
      SheetExporter(historyDealsName, historicalDeals);
   }
   if(historicalOrders != NULL) {
      SheetExporter(historyOrdersName, historicalOrders);
   }
   if(accountInfos != NULL) {
      SheetExporter(accountInfoName, accountInfos);
   } 
}

void SheetExporter(string sheetName, string data) {
   string headers = "Content-Type: application/json\r\n";
   char postData[], result[];
   string responseHeaders;
   string jsonDataWithSheetName = StringFormat("{\"spreadSheetId\": \"%s\", \"sheetName\": \"%s\", \"data\": [%s]}", spreadSheetId, sheetName, data);
  
   // Convert to char array
   StringToCharArray(jsonDataWithSheetName, postData, 0, StringLen(jsonDataWithSheetName), CP_UTF8);
   
   // Send POST request
   int res = WebRequest("POST", proxyServerUrl, headers, 5000, postData, result, responseHeaders);
   
   // Check response
   if(res != 200) {
     Alert("Account Infos not exported to Google Sheets returned error ", res, " and get last error is ", GetLastError());
   }
}

//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
   string terminalInfos = NULL;
   string historicalDeals = NULL;
   string historicalOrders = NULL;
   string accountInfos = NULL;
   if(exportTerminalInformations) terminalInfos = GetTerminalInfo();
   if(exportHistoricalDeals) historicalDeals = GetHistoryDeals();
   if(exportHistoricalOrders) historicalOrders = GetHistoryOrders();
   if(exportAccountInfos) accountInfos = GetAccountInfo();
   
   WriteToGoogleSheet(terminalInfos, historicalDeals, historicalOrders, accountInfos);
  }
//+------------------------------------------------------------------+