//+------------------------------------------------------------------+
//|                                                     ParaFrac.mq5 |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek"
#property version   "1.00"

#property indicator_separate_window
#property indicator_buffers 2
#property indicator_plots   2

#property indicator_type1   DRAW_HISTOGRAM
#property indicator_color1  Lime      // UpLevel
#property indicator_width1  2
#property indicator_label1  "UpLevel"

#property indicator_type2   DRAW_HISTOGRAM
#property indicator_color2  Red       // DownLevel
#property indicator_width2  2
#property indicator_label2  "DownLevel"

#property indicator_level1     2.5
#property indicator_level2    -2.5
#property indicator_level3     5.0
#property indicator_level4    -5.0
#property indicator_level5     7.5
#property indicator_level6    -7.5

#property indicator_maximum    10
#property indicator_minimum   -10

//---- indicator buffers
double UpLevelBuffer[];
double DownLevelBuffer[];

double fractalHigh = 0;
double fractalLow = 0;


// Input parameters
input double pstep = 0.02;
input double pMax = 0.2;
int lookback = 20;

// Handles for technical indicators
int sarHandle;
int fractalHandle;


//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{
   // Set indicator buffers
   SetIndexBuffer(0, UpLevelBuffer, INDICATOR_DATA);
   SetIndexBuffer(1, DownLevelBuffer, INDICATOR_DATA);
   
   // Create indicator handles
   sarHandle = iSAR(_Symbol, _Period, pstep, pMax);
   fractalHandle = iFractals(_Symbol, _Period);
   
   if(sarHandle == INVALID_HANDLE || fractalHandle == INVALID_HANDLE || sarHandle==INVALID_HANDLE)
   {
      Print("Error creating indicator handles");
      return(INIT_FAILED);
   }
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   // Check for minimum bars
   if(rates_total < lookback + 50)
      return(0);
      
   // Calculate starting index
   int pStart;
   if(prev_calculated == 0)
   {
      // First calculation - initialize buffer
      ArrayInitialize(UpLevelBuffer, EMPTY_VALUE);
      ArrayInitialize(DownLevelBuffer, EMPTY_VALUE);
      pStart = lookback + 50;
   }
   else
   {
      pStart = prev_calculated - 1;
   }
   
   // Get SAR values
   double sarArray[];
   if(CopyBuffer(sarHandle, 0, 0, rates_total, sarArray) < 0)
      return(0);
   
   // Get Fractal values
   double upperFractals[], lowerFractals[];
   if(CopyBuffer(fractalHandle, 0, 0, rates_total, upperFractals) < 0)
      return(0);
   if(CopyBuffer(fractalHandle, 1, 0, rates_total, lowerFractals) < 0)
      return(0);
   
   // Processing loop
   for(int i = pStart; i < rates_total; i++)
   {
      
      // Find nearest upper fractal within lookback
      for(int j = i; j <= MathMin(i + lookback, rates_total - 1); j++)
      {
         if(upperFractals[j] != EMPTY_VALUE)
         {
            fractalHigh = upperFractals[j];
            break;
         }
      }
      
      // Find nearest lower fractal within lookback
      for(int j = i; j <= MathMin(i + lookback, rates_total - 1); j++)
      {
         if(lowerFractals[j] != EMPTY_VALUE)
         {
            fractalLow = lowerFractals[j];
            break;
         }
      }
      
      // Skip calculation if fractals not found
      if(fractalHigh == EMPTY_VALUE || fractalLow == EMPTY_VALUE)
      {
         UpLevelBuffer[i] = 0;
         DownLevelBuffer[i] = 0;
         continue;
      }
      
      double fractalDiff = MathAbs(fractalHigh - fractalLow);
      if(fractalDiff == 0) fractalDiff = _Point; // Prevent division by zero
      
      double minOC = MathMin(open[i], close[i]);
      double maxOC = MathMax(open[i], close[i]);
      
      double upLevel = (minOC - sarArray[i]) / fractalDiff;
      double downLevel = (maxOC - sarArray[i]) / fractalDiff;
      
      // Set buffer values based on conditions
      if(upLevel > 0)
      {
         UpLevelBuffer[i] = upLevel;
         DownLevelBuffer[i] = 0;
      }
      else if(downLevel < 0)
      {
         DownLevelBuffer[i] = downLevel;
         UpLevelBuffer[i] = 0;
      }
      else
      {
         UpLevelBuffer[i] = 0;
         DownLevelBuffer[i] = 0;
      }
   }
   
   return(rates_total);
}
//+------------------------------------------------------------------+