//+------------------------------------------------------------------+
//|                                                          ssd.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<np.mqh>
#include<Math\Stat\Uniform.mqh>
//+-----------------------------------------------------------------------------------+
//| defines: representing range of random values from random number generator         |
//+-----------------------------------------------------------------------------------+
#define MIN_THRESHOLD 0.0
#define MAX_THRESHOLD 1.0

//+------------------------------------------------------------------+
//| Types of padding                                                 |
//+------------------------------------------------------------------+
enum ENUM_PADDING_TYPE
 {
  PAD_ZERO=0,
  PAD_MEAN,
  PAD_EDGE,
  PAD_MAX,
  PAD_MIN,
  PAD_MEDIAN
 };
//+------------------------------------------------------------------+
//| Permutation spectrum test                                        |
//+------------------------------------------------------------------+
double max_intensity(vector &time_series,uint min_padding=0,bool detrend=false,ENUM_PADDING_TYPE padding_type=PAD_ZERO)
 {
   int               m_length;                   //length of original series
   int               m_padded_len;                //modified length of series
   int               m_half_padded_len;           //half of modded series
   double            m_slope, m_intercept;     //slope and intercept of trend in series
   double            m_buffer[];               //general internal buffer
   complex           m_dft[];                 //general complex buffer
   double series[];
   if(!np::vecAsArray(time_series,series))
    {
     Print(__FUNCTION__, " vector to array failure ");
     return 0.0;
    }
   
   int i;
   int npts = ArraySize(series);
   int pad = (int)MathAbs(min_padding);
//--- check size of series
   if(npts<=0)
     {
      Print("Input array is empty");
      return 0.0;
     }
//---
   m_length = npts ;
   for(m_padded_len=2 ; m_padded_len<INT_MAX ; m_padded_len*=2)
     {
      if(m_padded_len >= npts+pad)
         break ;
     }
//---
   if(m_padded_len<npts+pad)
     {
      Print("Warning, calculated length of modified series is too long");
      return 0.0;
     }
//---
   m_half_padded_len = m_padded_len / 2;
//---
   ArrayResize(m_buffer,m_padded_len);
//---
   if(m_padded_len > npts)            // Any padding needed?
     {

      if(detrend)
        {
         m_intercept = series[0] ;
         m_slope = (series[npts-1] - series[0]) / (npts-1) ;
        }
      else
        {
         m_intercept = m_slope = 0.0 ;
         for(i=0 ; i<npts ; i++)
            m_intercept += series[i] ;
         m_intercept /= npts ;
        }

      for(i=0 ; i<npts ; i++)
        {
         m_buffer[i]=series[i] - m_intercept - m_slope * i ;
        }
      
      double padvalue=-1;
        
      switch(padding_type)
       {
        case PAD_ZERO: 
          padvalue=0.0;
          break;
        case PAD_EDGE:
          padvalue=series[series.Size()-1];
          break;
        case PAD_MAX:
          padvalue=MathMax(series); 
          break;
        case PAD_MIN:
          padvalue=MathMin(series); 
          break;
        case PAD_MEAN:
          padvalue=MathMean(series); 
          break;
        case PAD_MEDIAN:
          padvalue=MathMedian(series); 
          break;   
       } 
       
      ArrayFill(m_buffer,npts,m_padded_len-npts,padvalue);  
     }
   else
     {
      ArrayCopy(m_buffer,series);
      m_intercept = m_slope = 0.0 ;
     }
   
//---Compute the Fourier transform of the padded serie
   CFastFourierTransform::FFTR1D(m_buffer,int(m_padded_len),m_dft);
   vector power(m_dft.Size());
//---
   for(uint i = 0; i<m_dft.Size(); i++)
        power[i] = pow(m_dft[i].real,2.0)+pow(m_dft[i].imag,2.0);
//---   
   power = np::sliceVector(power,1,long(power.Size()-1));
//---   
   return power.Max();
 
 }
//+------------------------------------------------------------------+
//| Function to enable in place permuation of an array               |
//+------------------------------------------------------------------+
bool permute(vector &in_out)
  {
//---resize output array if necessary
   if(!in_out.Size())
     {
      Print("Invalid input parameter supplied ", "Array is Empty");
      return false;
     }
//---
   int i,j;
   double tempvalue;

   i=int(in_out.Size()-1);

   int error_value;
   double unif_rando;

   while(i>1)
     {
      error_value=0;
      unif_rando=MathRandomUniform(MIN_THRESHOLD,MAX_THRESHOLD,error_value);
      if(!MathIsValidNumber(unif_rando))
        {
         Print("Invalid random value ",error_value);
         return(false);
        }
      j=(int)(unif_rando*i);
      if(j>=i)
         j=i-1;
      --i;
      //---swap data randomly
      tempvalue=in_out[i];
      //---
      in_out[i]=in_out[j];
      //---
      in_out[j]=tempvalue;
     }
//---
   return true;
  }
//+------------------------------------------------------------------+
//| permutation spectrum test                                        |
//+------------------------------------------------------------------+
double perm_spec_test(vector &data,ulong n_permutations=100)
 {
  double observed = max_intensity(data);
  double obs;
  vector vdata = data;
  double counter = 0.0;
  
  for(ulong i = 0; i<n_permutations; i++)
     {
       permute(vdata);
       obs = max_intensity(vdata);
       if(obs>=observed)
          counter++;
     }
     
  return counter/double(n_permutations);
 }
//+------------------------------------------------------------------+
//|  Standard Singular Spectrum Analysis Implementation              |
//+------------------------------------------------------------------+
class CSingularSpecDecomp
  {
private:
   ulong             m_windowlen,m_trows,m_components;
   bool              m_initialized;
   matrix            m_tm;
   matrix            m_ls;
   matrix            m_rs;
   vector            m_sv;
   vector            traj2ts(matrix &in);
   matrix            hankelize(matrix &in);

public:
                     CSingularSpecDecomp(void);
                    ~CSingularSpecDecomp(void);

   bool              decompose(vector &series,ulong windowlen);
   vector            relative_contributions(void);
   vector            cumulative_contributions(void);
   matrix            components(ulong num_components=0);
   vector            filter(ulong num_components);
   matrix            comp_corr(void);
  };
//+------------------------------------------------------------------+
//| constructor                                                      |
//+------------------------------------------------------------------+
CSingularSpecDecomp::CSingularSpecDecomp(void)
  {
   m_initialized = false;
  }
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
CSingularSpecDecomp::~CSingularSpecDecomp(void)
  {
  }
//+------------------------------------------------------------------+
//|  decompose a sequence                                            |
//+------------------------------------------------------------------+
bool CSingularSpecDecomp::decompose(vector &series,ulong windowlen)
  {
   m_windowlen = windowlen;
   m_trows = series.Size() - m_windowlen +1;
//---
   m_tm = matrix::Zeros(m_trows,m_windowlen);
   vector eu;
   for(ulong i = 0; i<m_tm.Rows(); i++)
     {
      eu = np::sliceVector(series,i,i+m_windowlen);
      if(!m_tm.Row(eu,i))
        {
         Print(__FUNCTION__, " error ", GetLastError());
         return false;
        }
     }
//---
   CMatrixDouble trajectory = m_tm;
   double singular_values[];
   CMatrixDouble U,Vt;
   if(!CAlglib::RMatrixSVD(trajectory,trajectory.Rows(),trajectory.Cols(),2,2,2,singular_values,U,Vt))
     {
      Print(__FUNCTION__," error ", GetLastError());
      return false;
     }
   else
     {
      m_sv.Assign(singular_values);
      m_ls = U.ToMatrix();
      m_rs = (Vt.ToMatrix()).Transpose();
     }
//---

   m_initialized = true;
//---
   return true;
  }
//+------------------------------------------------------------------+
//| relative contributions                                           |
//+------------------------------------------------------------------+
vector CSingularSpecDecomp::relative_contributions(void)
  {
   vector ps = pow(m_sv,2.0);
   vector relcontribution = (ps/ps.Sum())*100.0;
   return relcontribution;
  }
//+------------------------------------------------------------------+
//| cumulative contributions                                         |
//+------------------------------------------------------------------+
vector CSingularSpecDecomp::cumulative_contributions(void)
  {
   vector ps = pow(m_sv,2.0);
   vector cumcontribution = (ps.CumSum()/ps.Sum())*100.0;
   return cumcontribution;
  }
//+------------------------------------------------------------------+
//| components                                                       |
//+------------------------------------------------------------------+
matrix CSingularSpecDecomp::components(ulong num_components=0)
  {
   ulong d = num_components?num_components:m_tm.Rank();

   matrix hankel,out;
   matrix trj;
   vector comp;
   out = matrix::Zeros(0,0);

   for(ulong i = 0; i<d; i++)
     {
      trj = (m_ls.Col(i).Outer(m_rs.Col(i)))*m_sv[i];
      hankel = hankelize(trj);
      comp = traj2ts(hankel);
      if(!out.Rows())
        {
         if(!out.Resize(comp.Size(),d))
           {
            Print(__FUNCTION__, " error ", GetLastError());
            return matrix::Zeros(0,0);
           }
        }

      if(!out.Col(comp,i))
        {
         Print(__FUNCTION__, " error ", GetLastError());
         return matrix::Zeros(0,0);
        }
     }

   np::reverseMatrixRows(out);

   return out;
  }
//+------------------------------------------------------------------+
//|   hankelize operator                                             |
//+------------------------------------------------------------------+
matrix CSingularSpecDecomp::hankelize(matrix &in)
  {
   ulong r,c;
   r = in.Rows();
   c = in.Cols();
   matrix mat = in;
   bool transposed = false;

   if(r>c)
     {
      mat = in.Transpose();
      r = mat.Rows();
      c = mat.Cols();
      transposed = true;
     }

   matrix hx = matrix::Zeros(r,c);
   ulong ss;
   for(ulong m = 0; m<r; m++)
     {
      for(ulong n = 0; n<c; n++)
        {
         ss = m+n;
         if(ss>=0 && ss<=r-1)
           {
            for(ulong i = 0; i<ss+1; i++)
              {
               hx[m][n] += 1./(ss+1)*mat[i][ss-i];
              }

           }
         else
            if(ss>=r && ss<=c-1)
              {
               for(ulong i = 0; i<r-1; i++)
                  hx[m][n] += 1./(r-1)*mat[i][ss-i];
              }
            else
               if(ss>=c && ss<=c+r-2)
                 {
                  for(ulong i = ss-c+1; i<r; i++)
                     hx[m][n] += 1./(c+r-ss-1)*mat[i][ss-i];
                 }
        }
     }

   if(transposed)
      return hx.Transpose();
   else
      return hx;
  }
//+------------------------------------------------------------------+
//|  hankel to time series                                           |
//+------------------------------------------------------------------+
vector CSingularSpecDecomp::traj2ts(matrix &in)
  {
   matrix rin = in;
   np::reverseMatrixCols(rin);

   vector dv;
   double ts[];
   for(long i = -long(rin.Rows())+1; i<long(rin.Cols()); i++)
     {
      dv = rin.Diag(i);
      ts.Push(dv.Mean());
     }

   dv.Assign(ts);
   return dv;
  }
//+------------------------------------------------------------------+
//| filter by the number of components                               |
//+------------------------------------------------------------------+
vector CSingularSpecDecomp::filter(ulong num_components)
  {
   matrix cccomps = components(num_components);
   return cccomps.Sum(1);
  }
//+------------------------------------------------------------------+
//| component correlations                                           |
//+------------------------------------------------------------------+
matrix CSingularSpecDecomp::comp_corr(void)
  {
   double w[];
   ulong fsize = m_trows+m_windowlen-1;
   for(ulong i = 0; i<fsize; i++)
     {
      if(i>=0 && i<=m_windowlen-1)
         w.Push(i+1);
      else
         if(i>=m_windowlen && i<=m_trows-1)
            w.Push(m_windowlen);
         else
            if(i>=m_trows && i<=fsize-1)
               w.Push(fsize-i);
     }

   vector weights;
   weights.Assign(w);
   ulong d = m_tm.Rank();

   matrix hankel,out;
   matrix trj;
   vector comp[];
   ArrayResize(comp,int(d));
   vector norms(d);
   out = matrix::Identity(d,d);

   for(ulong i = 0; i<d; i++)
     {
      trj = (m_rs.Col(i).Outer(m_ls.Col(i)))*m_sv[i];
      hankel = hankelize(trj);
      comp[i] = traj2ts(hankel);
      norms[i] = weights.Dot(pow(comp[i],2.0));
      norms[i] = pow(norms[i],-0.5);
     }

   for(ulong i = 0; i<d; i++)
     {
      for(ulong j = i+1; j<d; j++)
        {
         out[i][j] = MathAbs(weights.Dot(comp[i]*comp[j])*norms[i]*norms[j]);
         out[j][i] = out[i][j];
        }
     }

   return out;

  }
//+------------------------------------------------------------------+
