//+------------------------------------------------------------------+
//|                                           Multi-Pattern Signal EA|
//|                                   Copyright 2025, MetaQuotes Ltd.|
//|                           https://www.mql5.com/en/users/lynnchris|
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com/en/users/lynnchris"
#property version   "1.0"
#property strict

#include <Trade/Trade.mqh>

//--- Inputs
input int    RSI_Period         = 14;
input double RSI_Overbought     = 70.0;
input double RSI_Oversold       = 30.0;
input double SL_Pips            = 20.0;     // Stop-loss in pips
input double TP_Pips            = 20.0;     // Take-profit in pips
input double EntryBuffer        = 5.0;      // Entry offset in points
input double WickToBodyRatio    = 1.5;      // Pin bar wick length ratio
input double OppositeWickMax    = 0.5;      // Max opposite wick ratio
input double BodyMinPercent     = 0.05;     // Min body as percent of range
input int    MinMoveForWin      = 50;       // Points moved to count as win
input int    LookaheadBars      = 5;        // Bars to look ahead for win
input bool   EnableSound        = true;
input bool   EnablePush         = true;
input bool   EnableEmail        = false;

CTrade trade;

//--- Buffers
double   rsiBuffer[];
datetime timeBuffer[];
double   openBuffer[], highBuffer[], lowBuffer[], closeBuffer[];
int      rsiHandle;
datetime lastBarTime = 0;

//--- Stats
int totalPinbar=0, winPinbar=0;
int totalEngulf=0, winEngulf=0;

//+------------------------------------------------------------------+
int OnInit()
  {
   rsiHandle = iRSI(_Symbol, _Period, RSI_Period, PRICE_CLOSE);
   if(rsiHandle == INVALID_HANDLE)
      return INIT_FAILED;
   ArraySetAsSeries(rsiBuffer,true);
   ArraySetAsSeries(timeBuffer,true);
   ArraySetAsSeries(openBuffer,true);
   ArraySetAsSeries(highBuffer,true);
   ArraySetAsSeries(lowBuffer,true);
   ArraySetAsSeries(closeBuffer,true);
   return INIT_SUCCEEDED;
  }

//+------------------------------------------------------------------+
void OnTick()
  {
   if(Bars(_Symbol,_Period)<20)
      return;
   if(CopyTime(_Symbol,_Period,0,20,timeBuffer)<=0 ||
      CopyOpen(_Symbol,_Period,0,20,openBuffer)<=0 ||
      CopyHigh(_Symbol,_Period,0,20,highBuffer)<=0 ||
      CopyLow(_Symbol,_Period,0,20,lowBuffer)<=0 ||
      CopyClose(_Symbol,_Period,0,20,closeBuffer)<=0)
      return;
   if(timeBuffer[1]==lastBarTime)
      return;
   lastBarTime=timeBuffer[1];
   if(CopyBuffer(rsiHandle,0,0,20,rsiBuffer)<=0)
      return;

   int idx=1;
   int dir=FindSignalBar();
   if(dir==0)
      return;

   bool isBull=(dir>0);
   double entry = isBull ? highBuffer[idx]+EntryBuffer*_Point : lowBuffer[idx]-EntryBuffer*_Point;
   double stopL = isBull ? lowBuffer[idx]-SL_Pips*_Point   : highBuffer[idx]+SL_Pips*_Point;

// Track stats
   if(IsBullishPinBar(idx) || IsBearishPinBar(idx))
     {
      totalPinbar++;
      if(CheckWin(idx,isBull))
         winPinbar++;
     }
   else
     {
      totalEngulf++;
      if(CheckWin(idx,isBull))
         winEngulf++;
     }

   DrawSignal(idx,isBull,entry,stopL);
  }

//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   if(totalPinbar+totalEngulf>0)
     {
      double ratePin = totalPinbar>0 ? (double)winPinbar/totalPinbar*100.0 : 0;
      double rateEng = totalEngulf>0  ? (double)winEngulf/totalEngulf*100.0  : 0;
      PrintFormat("Pin Bars: %d signals, %d wins (%.2f%%)", totalPinbar, winPinbar, ratePin);
      PrintFormat("Engulfings: %d signals, %d wins (%.2f%%)", totalEngulf, winEngulf, rateEng);
      PrintFormat("Overall: %d signals, %d wins (%.2f%%)", totalPinbar+totalEngulf, winPinbar+winEngulf,
                  (double)(winPinbar+winEngulf)/(totalPinbar+totalEngulf)*100.0);
     }
  }

//+------------------------------------------------------------------+
int FindSignalBar()
  {
   bool bullDiv=false,bearDiv=false;
   for(int i=5;i<=15;i++)
     {
      if(lowBuffer[i]>lowBuffer[1] && rsiBuffer[i]<rsiBuffer[1] && rsiBuffer[1]<RSI_Oversold)
         bullDiv=true;
      if(highBuffer[i]<highBuffer[1] && rsiBuffer[i]>rsiBuffer[1] && rsiBuffer[1]>RSI_Overbought)
         bearDiv=true;
     }
   if(!bullDiv&&!bearDiv)
      return 0;
   bool bullPat=IsBullishPinBar(1)||IsBullishEngulfing(1);
   bool bearPat=IsBearishPinBar(1)||IsBearishEngulfing(1);
   if(bullDiv&&bullPat)
      return +1;
   if(bearDiv&&bearPat)
      return -1;
   return 0;
  }

//+------------------------------------------------------------------+
bool IsBullishPinBar(int i)
  {
   double body=fabs(openBuffer[i]-closeBuffer[i]);
   double rng=highBuffer[i]-lowBuffer[i];
   double lw=fmin(openBuffer[i],closeBuffer[i])-lowBuffer[i];
   double uw=highBuffer[i]-fmax(openBuffer[i],closeBuffer[i]);
   return closeBuffer[i]>openBuffer[i] && lw>WickToBodyRatio*body && uw<OppositeWickMax*body && body>BodyMinPercent*rng;
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool IsBearishPinBar(int i)
  {
   double body=fabs(openBuffer[i]-closeBuffer[i]);
   double rng=highBuffer[i]-lowBuffer[i];
   double uw=highBuffer[i]-fmax(openBuffer[i],closeBuffer[i]);
   double lw=fmin(openBuffer[i],closeBuffer[i])-lowBuffer[i];
   return closeBuffer[i]<openBuffer[i] && uw>WickToBodyRatio*body && lw<OppositeWickMax*body && body>BodyMinPercent*rng;
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool IsBullishEngulfing(int i)
  {
   return closeBuffer[i]>openBuffer[i] && openBuffer[i]<=closeBuffer[i+1] && closeBuffer[i]>=openBuffer[i+1];
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool IsBearishEngulfing(int i)
  {
   return closeBuffer[i]<openBuffer[i] && openBuffer[i]>=closeBuffer[i+1] && closeBuffer[i]<=openBuffer[i+1];
  }

//+------------------------------------------------------------------+
bool CheckWin(int i, bool isBull)
  {
   int bufSize = ArraySize(closeBuffer);
   for(int j = 1; j <= LookaheadBars; j++)
     {
      int idx = i + j;
      if(idx >= bufSize)
         break;
      double diff = isBull
                    ? (highBuffer[idx] - closeBuffer[i])
                    : (closeBuffer[i] - lowBuffer[idx]);
      if(diff >= MinMoveForWin * _Point)
         return true;
     }
   return false;
  }
//+------------------------------------------------------------------+
//void DrawSignal corrected placement
//+------------------------------------------------------------------+
void DrawSignal(int i,bool isBull,double entry,double stopL)
  {
   string tag=TimeToString(timeBuffer[i],TIME_SECONDS);
   string a="Arr_"+tag,e="Ent_"+tag,s="SL_"+tag,l="Div_"+tag;
   color c=isBull?clrLime:clrRed;
   int code=isBull?233:234;
   ObjectCreate(0,a,OBJ_ARROW,0,timeBuffer[i],entry);
   ObjectSetInteger(0,a,OBJPROP_COLOR,c);
   ObjectSetInteger(0,a,OBJPROP_ARROWCODE,code);
   ObjectSetInteger(0,a,OBJPROP_WIDTH,2);
   ObjectCreate(0,e,OBJ_HLINE,0,0,entry);
   ObjectSetInteger(0,e,OBJPROP_COLOR,clrAqua);
   ObjectCreate(0,s,OBJ_HLINE,0,0,stopL);
   ObjectSetInteger(0,s,OBJPROP_COLOR,clrOrangeRed);
   ObjectSetInteger(0,s,OBJPROP_STYLE,STYLE_DASH);
   for(int j=i+5;j<i+15;j++)
     {
      if(isBull&&lowBuffer[j]>lowBuffer[i])
        {
         ObjectCreate(0,l,OBJ_TREND,0,timeBuffer[j],lowBuffer[j],timeBuffer[i],lowBuffer[i]);
         ObjectSetInteger(0,l,OBJPROP_COLOR,clrDodgerBlue);
         break;
        }
      if(!isBull&&highBuffer[j]<highBuffer[i])
        {
         ObjectCreate(0,l,OBJ_TREND,0,timeBuffer[j],highBuffer[j],timeBuffer[i],highBuffer[i]);
         ObjectSetInteger(0,l,OBJPROP_COLOR,clrOrange);
         break;
        }
     }
   string pat=isBull? (IsBullishEngulfing(i)?"Engulfing":"PinBar") : (IsBearishEngulfing(i)?"Engulfing":"PinBar");
   string side=isBull?"Buy":"Sell";
   string msg=StringFormat("%s + RSI Divergence %s Signal\nSymbol: %s\nTime: %s\nEntry: %.5f\nSL: %.5f",
                           pat,side,_Symbol,TimeToString(timeBuffer[i],TIME_MINUTES),entry,stopL);
   Alert(msg);
   if(EnableSound)
      PlaySound("alert.wav");
   if(EnablePush)
      SendNotification(msg);
   if(EnableEmail)
      SendMail("Signal EA Alert",msg);
   Print(msg);
  }
//+------------------------------------------------------------------+
