//+------------------------------------------------------------------+
//|                                                       ta-lib.mqh |
//|                                          Copyright 2023, Omegafx |
//|                                           https://www.omegafx.co |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Omegafx"
#property link      "https://www.mql5.com/en/users/omegajoctan/seller"
//+------------------------------------------------------------------+
//| defines                                                          |
//+------------------------------------------------------------------+

#ifndef NaN
#define  NaN double("nan")
#endif

#ifndef inf
#define  inf double("inf")
#endif

#ifndef neg_inf
#define  neg_inf double("-inf")
#endif

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CheckShiftPeriod(uint size, uint period, uint shift)
  {
   if(size < period)
     {
      printf("period %d > price vector's size of %d, Failed to calculate the indicator", period, size);
      return false;
     }

   if(shift > period || shift > size)
     {
      printf("Invalid Shift value of %d", shift);
      return false;
     }

   return true;
  }

template <typename T>
vector<T> ArrayToVector(const T &arr[])
  {
   vector<T> v = vector::Zeros(arr.Size());
   for(uint i = 0; i < arr.Size(); i++)
      v[i] = arr[i];

   return v;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                     Trend Indicators                             |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//+------------------------------------------------------------------+
struct BB_res_struct
  {
   vector            upper_band;
   vector            lower_band;
   vector            middle_band;
  };

struct Envelopes_res_struct
  {
   vector            upper_band;
   vector            lower_band;
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CTrendIndicators
  {

public:
                     CTrendIndicators(void) {};
                    ~CTrendIndicators(void) {};

   //--- For dealing with vectors

   static vector     SMA(const vector &price, uint period, uint shift = 0);
   static vector     EMA(const vector &price, uint period, uint shift = 0);
   static BB_res_struct     BollingerBands(const vector &price, uint period, uint shift = 0, double k = 2.0);
   static vector     ParabolicSAR(const vector &high, const vector &low, const vector &close, double step = 0.02, double max = 0.2);
   static vector     StandardDeviation(const vector &price, uint period, uint shift = 0);
   static Envelopes_res_struct Envelopes(const vector &price, uint period = 14, double deviation = 0.01, uint shift = 0);

   //--- For dealing with arrays

   static vector     SMA(const double &price[], uint period, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return SMA(v, period, shift);
     }
   static vector     EMA(const double &price[], uint period, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return EMA(v, period, shift);
     }
   static BB_res_struct     BollingerBands(const double &price[], uint period, uint shift = 0, double k = 2.0)
     {
      vector v = ArrayToVector(price);
      return BollingerBands(v, period, shift, k);
     }
   static vector     ParabolicSAR(const double &high[], const double &low[], const double &close[], double step = 0.02, double max = 0.2)
     {
      vector high_vector = ArrayToVector(high),
             low_vector = ArrayToVector(low),
             close_vector = ArrayToVector(close);

      return ParabolicSAR(high_vector, low_vector, close_vector, step, max);
     }
   static vector     StandardDeviation(const double &price[], uint period, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return StandardDeviation(v, period, shift);
     }
   static Envelopes_res_struct Envelopes(const double &price[], uint period = 14, double deviation = 0.01, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return Envelopes(v, period, deviation, shift);
     }
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CTrendIndicators::SMA(const vector &price, uint period, uint shift = 0)
  {
   uint size = (uint)price.Size();

   if(!CheckShiftPeriod(size, period, shift))
      return price;

//---

   vector ma(size);
   ma.Fill(NaN);

   for(uint i = shift + period - 1; i < size; i++) //Loop through all the prices considering the period and shift
     {
      double sum = 0;
      for(uint j = i - period + 1; j <= i; j++)
         sum += price[j]; //sum of the prices

      ma[i] = sum / period; //divided by period to find the mean of that period
     }

   return ma;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
BB_res_struct CTrendIndicators::BollingerBands(const vector &price, uint period, uint shift = 0, double k = 2.0)
  {
   uint size = (uint)price.Size();

   BB_res_struct res;

//--- Check for valid parameters
   if(!CheckShiftPeriod(size, period, shift))
      return res;

//--- Initialize vectors

   res.upper_band.Resize(size);
   res.lower_band.Resize(size);

   res.upper_band.Fill(NaN);
   res.lower_band.Fill(NaN);

//--- Calculate the middle band (SMA)

   res.middle_band = SMA(price, period, shift);

//--- Calculate the upper and lower bands

   for(uint i = shift + period - 1; i < size; i++)
     {
      double sum_squared_diff = 0;
      for(uint j = i - period + 1; j <= i; j++)
        {
         sum_squared_diff += MathPow(price[j] - res.middle_band[i], 2);
        }

      double std_dev = MathSqrt(sum_squared_diff / period);

      res.upper_band[i] = res.middle_band[i] + (k * std_dev);
      res.lower_band[i] = res.middle_band[i] - (k * std_dev);
     }

   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CTrendIndicators::EMA(const vector &price, uint period, uint shift = 0)
  {
   uint size = (uint)price.Size();

   if(!CheckShiftPeriod(size, period, shift))
      return price;

//---

   double alpha = 2.0 / (period + 1.0);    // Smoothing factor

   vector res(size);
   res.Fill(NaN);

// Initialize the EMA with the SMA of the first period

   vector sma = SMA(price, period, shift);
   res[period - 1 + shift] = sma[period - 1 + shift];

// Calculate EMA for the rest of the prices

   for(ulong i = period + shift; i < size; i++)
      res[i] = alpha * price[i] + (1 - alpha) * res[i - 1];

   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CTrendIndicators::ParabolicSAR(const vector &high,
                                      const vector &low,
                                      const vector &close,
                                      double step = 0.02,
                                      double max = 0.2)
  {
   uint size = (uint)close.Size();
   vector psar(size);
   psar.Fill(NaN);

// Initialize variables

   double AF = step;    // Acceleration Factor
   double EP = high[0];      // Extreme Point
   double SAR = low[0];      // Initial SAR
   bool isUptrend = true;    // Assume uptrend at the start

// Calculate Parabolic SAR

   for(uint i = 0; i < size; i++)
     {
      // Update SAR
      if(isUptrend)
         SAR = SAR + AF * (EP - SAR);
      else
         SAR = SAR + AF * (SAR - EP);

      // Determine if trend changes
      if(isUptrend && SAR > low[i])
        {
         // Switch to downtrend
         isUptrend = false;
         SAR = EP;            // Reset SAR to the most recent EP
         EP = low[i];         // Reset EP
         AF = step;      // Reset AF
        }

      else
         if(!isUptrend && SAR < high[i])
           {
            // Switch to uptrend
            isUptrend = true;
            SAR = EP;            // Reset SAR to the most recent EP
            EP = high[i];        // Reset EP
            AF = step;      // Reset AF
           }

      // Update EP and AF
      if(isUptrend)
        {
         if(high[i] > EP)
           {
            EP = high[i];
            AF = MathMin(AF + step, max);
           }
        }
      else
        {
         if(low[i] < EP)
           {
            EP = low[i];
            AF = MathMin(AF + step, max);
           }
        }

      // Store the SAR value
      psar[i] = SAR;
     }

   return psar;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CTrendIndicators::StandardDeviation(const vector &price, uint period, uint shift = 0)
  {
   uint size = (uint)price.Size();

// Check if the period and shift are valid
   if(!CheckShiftPeriod(size, period, shift))
      return price;

// Initialize standard deviation vector
   vector std_dev(size);
   std_dev.Fill(NaN);

// Loop through the price data
   for(uint i = shift + period - 1; i < size; i++)
     {
      double sum = 0.0;
      double sum_sq_diff = 0.0;

      // Calculate mean
      for(uint j = i - period + 1; j <= i; j++)
         sum += price[j];
      double mean = sum / period;

      // Calculate squared differences
      for(uint j = i - period + 1; j <= i; j++)
         sum_sq_diff += MathPow(price[j] - mean, 2);

      // Calculate standard deviation
      std_dev[i] = MathSqrt(sum_sq_diff / period);
     }

   return std_dev;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
Envelopes_res_struct CTrendIndicators::Envelopes(const vector &price, uint period = 14, double deviation = 0.01, uint shift = 0)
  {
   uint size = (uint)price.Size();

   Envelopes_res_struct res;
   if(!CheckShiftPeriod(size, period, shift))
     {
      return res;
     }

// Initialize output vectors
   res.upper_band.Resize(size);
   res.lower_band.Resize(size);

   res.upper_band.Fill(NaN);
   res.lower_band.Fill(NaN);

// Calculate the moving average
   vector ma = SMA(price, period, shift);

// Calculate upper and lower bands
   for(uint i = shift + period - 1; i < size; i++)
     {
      res.upper_band[i] = ma[i] * (1 + deviation);
      res.lower_band[i] = ma[i] * (1 - deviation);
     }

   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|            Oscillators Indicators Class                          |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//+------------------------------------------------------------------+
struct MACD_res_struct
  {
   vector            main;     // The MACD Line
   vector            signal;   // The Signal Line
   vector            histogram;    // The MACD Histogram
  };

struct Stochastic_struct
  {
   vector            main;
   vector            signal;
  };

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class COscillatorIndicators
  {
public:
                     COscillatorIndicators(void) {};
                    ~COscillatorIndicators(void) {};

   //--- For dealing with vectors

   static MACD_res_struct MACD(const vector &price, uint fast_ema = 12, uint slow_ema = 26, uint macd_sma = 9, uint shift = 0);
   static vector     RSI(const vector &price, uint period, uint shift = 0);
   static Stochastic_struct StochasticOscillator(const vector &high, const vector &low, const vector &close, uint k_period = 5, uint d_period = 3, uint period = 3,  uint shift = 0);
   static vector     ATR(const vector &high, const vector &low, const vector &close, uint period = 14, uint shift = 0);
   static vector     MomentumIndicator(const vector &price, uint period, uint shift = 0);

   //--- For dealing with arrays

   static MACD_res_struct MACD(const double &price[], uint fast_ema = 12, uint slow_ema = 26, uint macd_sma = 9, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return MACD(v, fast_ema, slow_ema, macd_sma, shift);
     }
   static vector     RSI(const double &price[], uint period, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return RSI(v, period, shift);
     }
   static Stochastic_struct StochasticOscillator(const double &high[], const double &low[], const double &close[], uint k_period = 5, uint d_period = 3, uint period = 3,  uint shift = 0)
     {
      vector high_vector = ArrayToVector(high),
             low_vector = ArrayToVector(low),
             close_vector = ArrayToVector(close);

      return StochasticOscillator(high_vector, low_vector, close_vector, k_period, d_period, period, shift);
     }

   static vector     ATR(const double &high[], const double &low[], const double &close[], uint period = 14, uint shift = 0)
     {
      vector high_vector = ArrayToVector(high),
             low_vector = ArrayToVector(low),
             close_vector = ArrayToVector(close);

      return ATR(high_vector, low_vector, close_vector, period, shift);
     }
   static vector     MomentumIndicator(const double &price[], uint period, uint shift = 0)
     {
      vector v = ArrayToVector(price);
      return MomentumIndicator(v, period, shift);
     }
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
MACD_res_struct COscillatorIndicators::MACD(const vector &price, uint fast_ema = 12, uint slow_ema = 26, uint macd_sma = 9, uint shift = 0)
  {
   uint size = (uint)price.Size();

   MACD_res_struct res;
   if(!CheckShiftPeriod(size, slow_ema, shift))
      return res;

//--- Calculate EMA(short), EMA(long), and MACD Line

   vector fast_ema_vector = CTrendIndicators::EMA(price, fast_ema, shift);
   vector slow_ema_vector = CTrendIndicators::EMA(price, slow_ema, shift);

   res.main.Resize(size);
   res.main.Fill(NaN);

   for(uint i = 0; i < size; i++)
      res.main[i] = fast_ema_vector[i] - slow_ema_vector[i];

//--- Calculate Signal Line (SMA of MACD Line)

   res.signal = CTrendIndicators::SMA(price, macd_sma, shift);

//--- Calculate MACD Histogram

   res.histogram.Resize(size);
   res.histogram.Fill(NaN);

   for(uint i = 0; i < size; i++)
      res.histogram[i] = res.main[i] - res.signal[i];

   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector COscillatorIndicators::RSI(const vector &price, uint period, uint shift = 0)
  {
   uint size = (uint)price.Size();

//--- Check for valid parameters
   if(!CheckShiftPeriod(size, period, shift))
      return price;

//--- Initialize vectors
   vector rsi(size), gains(size), losses(size);
   rsi.Fill(NaN);
   gains.Fill(0.0);
   losses.Fill(0.0);

//--- Calculate gains and losses
   for(uint i = shift + 1; i < size; i++)
     {
      double change = price[i] - price[i - 1];
      gains[i] = (change > 0) ? change : 0;
      losses[i] = (change < 0) ? -change : 0;
     }

//--- Initialize first average gain and loss (simple average for the first period)
   double avg_gain = 0, avg_loss = 0;
   for(uint i = shift + 1; i < shift + 1 + period; i++)
     {
      avg_gain += gains[i];
      avg_loss += losses[i];
     }
   avg_gain /= period;
   avg_loss /= period;

//--- Compute RSI for the rest of the periods
   for(uint i = shift + period; i < size; i++)
     {
      // Apply smoothing for average gain and loss
      avg_gain = ((avg_gain * (period - 1)) + gains[i]) / period;
      avg_loss = ((avg_loss * (period - 1)) + losses[i]) / period;

      // Calculate RSI
      double rs = (avg_loss == 0) ? 0 : avg_gain / avg_loss;
      rsi[i] = (avg_loss == 0) ? 100 : (100 - (100 / (1 + rs)));
     }

   return rsi;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
Stochastic_struct COscillatorIndicators::StochasticOscillator(const vector &high, const vector &low, const vector &close, uint k_period = 5, uint d_period = 3, uint period = 3, uint shift = 0)
  {
   uint size = (uint)close.Size();

   Stochastic_struct res;

// Check for valid parameters
   if(!CheckShiftPeriod(size, period, shift))
      return res;

// Initialize vectors for %K and %D
   vector K(size), D(size);
   K.Fill(NaN);
   D.Fill(NaN);

// Calculate %K
   for(uint i = shift + period - 1; i < size; i++)
     {
      double H_max = -DBL_MAX, L_min = DBL_MAX;

      // Find the highest high and the lowest low over the lookback period
      for(uint j = i - period + 1; j <= i; j++)
        {
         H_max = MathMax(H_max, high[j]);
         L_min = MathMin(L_min, low[j]);
        }

      // Calculate %K
      double K_value = (H_max - L_min != 0) ? ((close[i] - L_min) / (H_max - L_min)) * 100 : 0;
      K[i] = K_value;
     }

// Smooth %K with a simple moving average (k_period)
   vector smoothedK(size);
   smoothedK.Fill(NaN);

   for(uint i = shift + k_period - 1; i < size; i++)
     {
      double sum = 0;
      for(uint j = i - k_period + 1; j <= i; j++)
        {
         sum += K[j];
        }
      smoothedK[i] = sum / k_period;
     }

// Calculate %D (3-period moving average of %K)

   D = CTrendIndicators::SMA(smoothedK, period, shift);

   res.main = K;
   res.signal = D;

   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector COscillatorIndicators::ATR(const vector &high, const vector &low, const vector &close, uint period = 14, uint shift = 0)
  {
   uint size = (uint)close.Size();

// Check for valid parameters
   if(!CheckShiftPeriod(size, period, shift))
      return close;

// Initialize the True Range (TR) and ATR vectors
   vector TR(size);
   TR.Fill(NaN);

// Calculate the True Range for each period
   for(uint i = shift + 1; i < size; i++)
     {
      double H = high[i];
      double L = low[i];
      double C_prev = close[i - 1];

      // Calculate the three possible True Range values
      double TR1 = H - L;
      double TR2 = MathAbs(H - C_prev);
      double TR3 = MathAbs(L - C_prev);

      // True Range is the maximum of the three
      TR[i] = MathMax(TR1, MathMax(TR2, TR3));
     }

//--- Smooth the True Range using a simple moving average (SMA) over the specified period

   return CTrendIndicators::SMA(TR, period, shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector COscillatorIndicators::MomentumIndicator(const vector &price, uint period, uint shift = 0)
  {
   uint size = (uint)price.Size();

// Check for valid input
   if(!CheckShiftPeriod(size, period, shift))
      return price;

// Initialize the momentum vector
   vector momentum(size);
   momentum.Fill(NaN);

// Calculate Momentum
   for(uint i = shift + period; i < size; i++)
     {
      //momentum[i] = price[i] - price[i - period]; // Momentum difference formula

      // using the ratio formula:
      momentum[i] = (price[i] / price[i - period]) * 100;
     }

   return momentum;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                     Volumes Indicator                            |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//|                                                                  |
//+------------------------------------------------------------------+
class CBillWilliamsIndicators
  {
public:
                     CBillWilliamsIndicators(void) {};
                    ~CBillWilliamsIndicators(void) {};

   //--- For dealing with vectors

   static vector     AcceleratorOscillator(const vector &high, const vector &low, uint ao_period_fast = 5, uint ao_period_slow = 34, uint ac_period = 5);
   static vector     AwesomeOscillator(const vector &high, const vector &low, uint fast_period = 5, uint slow_period = 34, uint shift = 0);

   //--- For dealing with arrays

   static vector     AcceleratorOscillator(const double &high[], const double &low[], uint ao_period_fast = 5, uint ao_period_slow = 34, uint ac_period = 5)
     {
      vector high_vector = ArrayToVector(high);
      vector low_vector = ArrayToVector(low);

      return AcceleratorOscillator(high_vector, low_vector, ao_period_fast, ao_period_slow, ac_period);
     }
   static vector     AwesomeOscillator(const double &high[], const double &low[], uint fast_period = 5, uint slow_period = 34, uint shift = 0)
     {
      vector high_vector = ArrayToVector(high);
      vector low_vector = ArrayToVector(low);

      return AwesomeOscillator(high_vector, low_vector, fast_period, slow_period, shift);
     }

  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CBillWilliamsIndicators::AwesomeOscillator(const vector &high, const vector &low, uint fast_period = 5, uint slow_period = 34, uint shift = 0)
  {
   uint size = (uint)high.Size();
   if(size != low.Size())
      return vector::Zeros(0); // Ensure high and low vectors are of the same size

   if(!CheckShiftPeriod(size, slow_period, shift))
      return vector::Zeros(0); // Validate shift and slow period

// Initialize vectors
   vector ao(size), median_price(size);
   ao.Fill(NaN);
   median_price.Fill(NaN);

// Calculate Median Price
   for(uint i = 0; i < size; i++)
      median_price[i] = (high[i] + low[i]) / 2;

// Calculate Fast and Slow SMAs of the Median Price
   vector sma_fast = CTrendIndicators::SMA(median_price, fast_period, shift);
   vector sma_slow = CTrendIndicators::SMA(median_price, slow_period, shift);

// Calculate AO
   for(uint i = 0; i < size; i++)
      ao[i] = sma_fast[i] - sma_slow[i];

   return ao;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CBillWilliamsIndicators::AcceleratorOscillator(const vector &high,
      const vector &low,
      uint ao_period_fast = 5, // Fast period for AO
      uint ao_period_slow = 34, // Slow period for AO
      uint ac_period = 5 // Period for AC SMA
                                                     )
  {
   uint size = (uint)high.Size();
   if(size != low.Size())
      return vector::Zeros(0); // Ensure high and low vectors are of the same size


// Validate shift and period
   if(!CheckShiftPeriod(size, ao_period_slow, 0))
      return vector::Zeros(0);

// Calculate AO (Awesome Oscillator)

   vector ao = AwesomeOscillator(high, low, ao_period_fast, ao_period_slow);

// Calculate AC (Accelerator Oscillator)

   vector ac(size);
   ac.Fill(NaN);

   vector ao_sma_ac = CTrendIndicators::SMA(ao, ac_period);

   for(uint i = 0; i < size; i++)
      ac[i] = ao[i] - ao_sma_ac[i];

   return ac;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
