//+------------------------------------------------------------------+
//|                                         multilayerperceptron.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<np.mqh>
//+------------------------------------------------------------------+
//| Multilayer perceptron classifier                                 |
//+------------------------------------------------------------------+
class CMlp
  {
private:
   int               m_rinfo;
   CMultilayerPerceptronShell *m_net;
   CMLPReportShell   *m_rep;
   CMatrixDouble     m_xy;
   bool              m_trained;
   int               m_nhid1,m_nhid2,m_epochs,m_ninputs,m_noutputs;
   double            m_tol,m_lr,m_decay,m_alpha,m_beta;

public:
                     CMlp(void);
                    ~CMlp(void);
   bool              fit(matrix &predictors,matrix &targets,double alpha=0.3,double beta=0.01,ulong num_neurons_nonlinear_hiddenlayer1=0, ulong num_neurons_nonlinear_hiddenlayer2=0,ulong max_epochs=10000,double decay = 0.001,double convergence_criterion=1.e-8);
   vector            predict(vector &predictors);
   matrix            predict(matrix &predictors);
   double            score(matrix &predictors,matrix &targets,ENUM_LOSS_FUNCTION lossfunction = LOSS_MSE);
   CMultilayerPerceptron *get_networkobj(void) { return m_net.GetInnerObj(); }
  };
//+------------------------------------------------------------------+
//|  constructor                                                     |
//+------------------------------------------------------------------+
CMlp::CMlp(void)
  {
   m_net = new CMultilayerPerceptronShell();
   m_rep = new CMLPReportShell();
  }
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
CMlp::~CMlp(void)
  {
   if(CheckPointer(m_net) == POINTER_DYNAMIC || CheckPointer(m_rep) == POINTER_DYNAMIC)
     {
      delete m_net;
      delete m_rep;
     }
  }
//+------------------------------------------------------------------+
//| fit classifier model on a dataset                                |
//+------------------------------------------------------------------+
bool CMlp::fit(matrix &predictors,matrix &targets,double alpha=0.3,double beta=0.01,ulong num_neurons_nonlinear_hiddenlayer1=0,ulong num_neurons_nonlinear_hiddenlayer2=0,ulong max_epochs=10000,double decay = 0.001,double convergence_criterion=1.e-8)
  {
   m_alpha = alpha;
   m_beta = beta;
   m_trained = false;
   
   m_ninputs = int(predictors.Cols());

   m_noutputs = int(targets.Cols());
   

   if(predictors.Rows()!=targets.Rows() || m_noutputs<1)
     {
      Print(__FUNCTION__," ",__LINE__," Invalid training data ");
      return false;
     }

   m_xy = predictors;

   if(!m_xy.Resize(predictors.Rows(),predictors.Cols()+targets.Cols()))
     {
      Print(__FUNCTION__," ",__LINE__," failed to format training data ");
      return false;
     }
   
   for(int i = 0; i<m_noutputs; i++)
    {
     
     if(!m_xy.Col(i+m_ninputs,targets.Col(ulong(i))))
       {
        Print(__FUNCTION__," ",__LINE__," failed to format training data ");
        return false;
       }
    
    }

   m_nhid1 = int(num_neurons_nonlinear_hiddenlayer1);
   m_nhid2 = int(num_neurons_nonlinear_hiddenlayer2);
   m_tol= convergence_criterion;
   m_decay = decay;
   m_epochs = int(max_epochs);

   if(m_nhid1+m_nhid2 == 0)
      CMLPBase::MLPCreateR0(m_ninputs,m_noutputs,m_alpha,m_beta,m_net.GetInnerObj());
   else
      if((m_nhid1==0 && m_nhid2>0) || (m_nhid2==0 && m_nhid1>0))
         CMLPBase::MLPCreateR1(m_ninputs,m_nhid1+m_nhid2,m_noutputs,m_alpha,m_beta,m_net.GetInnerObj());
      else
         CMLPBase::MLPCreateR2(m_ninputs,m_nhid1,m_nhid2,m_noutputs,m_alpha,m_beta,m_net.GetInnerObj());

   CMLPTrain::MLPTrainLBFGS(m_net.GetInnerObj(),m_xy,m_xy.Size(),m_decay,2,m_tol,m_epochs,m_rinfo,m_rep.GetInnerObj());

   m_trained = (m_rinfo == 2);

   if(!m_trained)
     {
      string emess;
      if(m_rinfo == -8)
         emess = " Misconfigured convergence criteria ";
      else
         if(m_rinfo == -2)
            emess = " Invalid target training data ";
         else
            if(m_rinfo == -1)
               emess = " invalid hyperparameters ";
      Print(__FUNCTION__, " ", __LINE__,emess);
     }

   return m_trained;

  }
//+------------------------------------------------------------------+
//| get the class probabilities for single sample                    |
//+------------------------------------------------------------------+
vector CMlp::predict(vector &predictors)
  {

   vector probs(m_noutputs);

   if(!m_trained)
     {
      Print(__FUNCTION__," ",__LINE__," no trained model available ");
      probs.Fill(EMPTY_VALUE);
      return probs;
     }

   if(predictors.Size() != ulong(m_ninputs))
     {
      Print(__FUNCTION__," ",__LINE__," Invalid inputs ");
      probs.Fill(EMPTY_VALUE);
      return probs;
     }

   CRowDouble predinputs(predictors), ypred;

   CMLPBase::MLPProcess(m_net.GetInnerObj(),predinputs,ypred);

   probs = ypred.ToVector();

   return probs;

  }
//+------------------------------------------------------------------+
//| get the class probabilities for multiple samples                 |
//+------------------------------------------------------------------+
matrix CMlp::predict(matrix &predictors)
  {

   matrix probs(predictors.Rows(),m_noutputs);

   probs.Fill(EMPTY_VALUE);

   vector rowin,rowout;
   for(ulong i = 0; i<probs.Rows(); i++)
     {
      rowin = predictors.Row(i);
      rowout = predict(rowin);

      if(!probs.Row(rowout,i))
        {
         Print(__FUNCTION__,"  ",__LINE__," ",GetLastError());
         probs.Fill(EMPTY_VALUE);
         return probs;
        }
     }

   return probs;
  }
//+------------------------------------------------------------------+
//| score the model                                                  |
//+------------------------------------------------------------------+
double CMlp::score(matrix &predictors,matrix &targets,ENUM_LOSS_FUNCTION lossfunction = LOSS_MSE)
  {

   matrix probs = predict(predictors);

   return (1.0 - probs.Loss(targets,lossfunction));
  }

//+------------------------------------------------------------------+
