//+------------------------------------------------------------------+
//|                                                    SignalSAR.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals of customized SAR indicator.                       |
//| Type=SignalAdvanced                                              |
//| Name=Parabolic Sar                                               |
//| ShortName=SAR                                                    |
//| Class=CSignalSAR                                                 |
//| Page=signal_sar                                                  |
//| Parameter=PeriodMA,int,14,Period of averaging                    |
//| Parameter=Applied,ENUM_APPLIED_PRICE,PRICE_CLOSE,Prices series   |
//| Parameter=AlternateFrame,ENUM_TIMEFRAMES,PERIOD_W1,Alternate TimeFrame |
//| Parameter=Pattern_0,int,50,Pattern 0                             |
//| Parameter=Pattern_1,int,50,Pattern 1                             |
//| Parameter=Pattern_2,int,50,Pattern 2                             |
//| Parameter=Pattern_3,int,50,Pattern 3                             |
//| Parameter=Pattern_4,int,50,Pattern 4                             |
//| Parameter=Pattern_5,int,50,Pattern 5                             |
//| Parameter=Pattern_6,int,50,Pattern 6                             |
//| Parameter=Pattern_7,int,50,Pattern 7                             |
//| Parameter=Pattern_8,int,50,Pattern 8                             |
//| Parameter=Pattern_9,int,50,Pattern 9                             |
//| Parameter=PatternsUsed,int,255,Patterns Used BitMap            |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalSAR.                                                |
//| Purpose: Class of generator of trade signals based on            |
//|          customized SAR indicator.                               |
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+
#define  __COMPRESSION_LIMIT 65
class CSignalSAR : public CExpertSignal 
{
protected:
   CiSAR             m_sar;            // object-indicator
   CiSAR             m_sar_alt;        // object-indicator alternate
   CiMA              m_ma;             // object-indicator
   //--- adjusted parameters
   int               m_ma_period;      // the "period of averaging" parameter of the oscillator
   ENUM_APPLIED_PRICE m_ma_applied;    // the "object of averaging" parameter of the oscillator
   ENUM_TIMEFRAMES   m_alternate;      // the "alternate time frame" parameter of the oscillator
   //--- "weights" of market models (0-100)
   int               m_pattern_0;      // model 0 "Reversal Gap Crossover"
   int               m_pattern_1;      // model 1 "SAR Compression Zone "
   int               m_pattern_2;      // model 2 "Extended Trending SAR"
   int               m_pattern_3;      // model 3 "SAR Flip Fakeout"
   int               m_pattern_4;      // model 4 "Double SAR Flip with Trend Continuation "
   int               m_pattern_5;      // model 5 "SAR Divergence with Moving Average "
   int               m_pattern_6;      // model 6 "Parabolic SAR Channeling "
   int               m_pattern_7;      // model 7 "SAR and Volume Divergence "
   int               m_pattern_8;      // model 8 "Inverted SAR on Higher Timeframe "
   int               m_pattern_9;      // model 9 "SAR and RSI Overlap "

public:
                     CSignalSAR(void);
                    ~CSignalSAR(void);
   //--- methods of setting adjustable parameters
   void              PeriodMA(int value) 
   {  m_ma_period = value;
   }
   void              Applied(ENUM_APPLIED_PRICE value)
   {  m_ma_applied = value;
   }
   void              AlternateFrame(ENUM_TIMEFRAMES value)
   {  m_alternate = value;
   }
   //--- methods of adjusting "weights" of market models
   void              Pattern_0(int value)
   {  m_pattern_0 = value;
   }
   void              Pattern_1(int value)
   {  m_pattern_1 = value;
   }
   void              Pattern_2(int value)
   {  m_pattern_2 = value;
   }
   void              Pattern_3(int value)
   {  m_pattern_3 = value;
   }
   void              Pattern_4(int value)
   {  m_pattern_4 = value;
   }
   void              Pattern_5(int value)
   {  m_pattern_5 = value;
   }
   void              Pattern_6(int value)
   {  m_pattern_6 = value;
   }
   void              Pattern_7(int value)
   {  m_pattern_7 = value;
   }
   void              Pattern_8(int value)
   {  m_pattern_8 = value;
   }
   void              Pattern_9(int value)
   {  m_pattern_9 = value;
   }
   void              PatternsUsed(int value)
   {                 m_patterns_usage = value;
                     PatternsUsage(value);
   }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the oscillator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   //--- method of initialization of the oscillator
   bool              InitSAR(CIndicators *indicators);
   //--- methods of getting data
   double            Alt(int ind)
   {                 m_sar_alt.Refresh(-1);
                     return(m_sar_alt.Main(ind));
   }
   double            Base(int ind)
   {                 m_sar.Refresh(-1);
                     return(m_sar.Main(ind));
   }
   double            Close(int ind)
   {                 m_close.Refresh(-1);
                     return(m_close.GetData(ind));
   }
   double            High(int ind)
   {                 m_high.Refresh(-1);
                     return(m_high.GetData(ind));
   }
   double            Low(int ind)
   {                 m_low.Refresh(-1);
                     return(m_low.GetData(ind));
   }
   double            Range(int ind)
   {                 return(High(ind) - Low(ind));
   }
   double            MA(int ind)
   {                 m_ma.Refresh(-1);
                     return(m_ma.Main(ind));
   }
   bool              Compression(ENUM_POSITION_TYPE T, double &Out)
   {                 Out = 0.0;
                     int _i = StartIndex() + 1, _c = 0;
                     double _last = Base(StartIndex() + 1);
                     double _first = 0.0;
                     if
                     (
                     T == POSITION_TYPE_BUY &&
                     Base(StartIndex()) < Low(StartIndex()) &&
                     Base(_i) < Close(StartIndex()) &&
                     Base(_i) > High(_i)
                     )
                     {  while(Base(_i) > High(_i) && _c < __COMPRESSION_LIMIT)
                        {  _first = Base(_i);
                           _i++;
                           _c++;
                        }
                        if(_c > 0)
                        {  Out = fabs(_first - _last)/_c;
                           return(true);
                        }
                     }
                     else if
                     (
                     T == POSITION_TYPE_SELL &&
                     Base(StartIndex()) > High(StartIndex()) &&
                     Base(_i) > Close(StartIndex()) &&
                     Base(_i) < Low(_i)
                     )
                     {  while(Base(_i) < Low(_i) && _c < __COMPRESSION_LIMIT)
                        {  _first = Base(_i);
                           _i++;
                           _c++;
                        }
                        if(_c > 0)
                        {  Out = fabs(_first - _last)/_c;
                           return(true);
                        }
                     }
                     return(false);
   }
   bool              Channel(ENUM_POSITION_TYPE T)
   {                 vector _max,_max_i;
                     vector _min,_min_i;
                     _max.Init(2);
                     _max.Fill(High(0));
                     _max_i.Init(2);
                     _max_i.Fill(0.0);
                     _min.Init(2);
                     _min.Fill(Low(0));
                     _min_i.Init(2);
                     _min_i.Fill(0.0);
                     for(int i=0;i<m_ma_period;i++)
                     {  if(High(i) > _max[0])
                        {  _max[0] = High(i);
                           _max_i[0] = i;
                        }
                        if(Low(i) < _min[0])
                        {  _min[0] = Low(i);
                           _min_i[0] = i;
                        }
                     }
                     double _slope = (Close(0) - Close(m_ma_period-1))/m_ma_period;
                     double _upper_scale = fabs(_slope);
                     double _lower_scale = fabs(_slope);
                     for(int i=0;i<m_ma_period;i++)
                     {  if(i == _max_i[0])
                        {  continue;
                        }
                        else
                        {  double _i_slope = (High(i) - _max[0])/(i - _max_i[0]);
                           if((_i_slope > 0.0 && _slope > 0.0)||(_i_slope < 0.0 && _slope < 0.0))
                           {  if(fabs(_i_slope-_slope) < _upper_scale)
                              {  _max[1] = High(i);
                                 _max_i[1] = i;
                              } 
                           }
                        }
                     }
                     for(int i=0;i<m_ma_period;i++)
                     {  if(i == _min_i[0])
                        {  continue;
                        }
                        else
                        {  double _i_slope = (Low(i) - _min[0])/(i - _min_i[0]);
                           if((_i_slope > 0.0 && _slope > 0.0)||(_i_slope < 0.0 && _slope < 0.0))
                           {  if(fabs(_i_slope-_slope) < _lower_scale)
                              {  _min[1] = Low(i);
                                 _min_i[1] = i;
                              } 
                           }
                        }
                     }
                     vector _projections;
                     _projections.Init(4);
                     _projections[0] = _max[0] + (_max_i[0]*_slope);
                     _projections[1] = _min[0] + (_min_i[0]*_slope);
                     _projections[2] = _max[1] + (_max_i[1]*_slope);
                     _projections[3] = _min[1] + (_min_i[1]*_slope);
                     if(T == POSITION_TYPE_BUY && Close(0) < Close(m_ma_period) && Close(0) < _projections.Mean())
                     {  return(true);
                     }
                     else if(T == POSITION_TYPE_SELL && Close(0) > Close(m_ma_period) && Close(0) > _projections.Mean())
                     {  return(true);
                     }
                     return(false);
   }
   //--- methods to check for patterns
   bool              IsPattern_0(ENUM_POSITION_TYPE T);
   bool              IsPattern_1(ENUM_POSITION_TYPE T);
   bool              IsPattern_2(ENUM_POSITION_TYPE T);
   bool              IsPattern_3(ENUM_POSITION_TYPE T);
   bool              IsPattern_4(ENUM_POSITION_TYPE T);
   bool              IsPattern_5(ENUM_POSITION_TYPE T);
   bool              IsPattern_6(ENUM_POSITION_TYPE T);
   bool              IsPattern_7(ENUM_POSITION_TYPE T);
   bool              IsPattern_8(ENUM_POSITION_TYPE T);
   bool              IsPattern_9(ENUM_POSITION_TYPE T);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalSAR::CSignalSAR(void) : m_ma_period(14),
   m_ma_applied(PRICE_CLOSE),
   m_alternate(PERIOD_W1),
   m_pattern_0(50),
   m_pattern_1(50),
   m_pattern_2(50),
   m_pattern_3(50),
   m_pattern_4(50),
   m_pattern_5(50),
   m_pattern_6(50),
   m_pattern_7(50),
   m_pattern_8(50),
   m_pattern_9(50)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE;
   PatternsUsage(m_patterns_usage);
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalSAR::~CSignalSAR(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalSAR::ValidationSettings(void)
{
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_ma_period <= 0)
   {  printf(__FUNCTION__ + ": period MA must be greater than 0");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalSAR::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- create and initialize MA oscillator
   if(!InitSAR(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Initialize MA indicators.                                        |
//+------------------------------------------------------------------+
bool CSignalSAR::InitSAR(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_sar)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_sar.Create(m_symbol.Name(), m_period, 0.02, 0.2))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_sar_alt.Create(m_symbol.Name(), m_alternate, 0.02, 0.2))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_ma.Create(m_symbol.Name(), m_period, m_ma_period, 0, MODE_SMA, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalSAR::LongCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Reversal Gap Crossover"
   if(((m_patterns_usage & 0x01) != 0) && IsPattern_0(POSITION_TYPE_BUY))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "SAR Compression Zone"
   if(((m_patterns_usage & 0x02) != 0) && IsPattern_1(POSITION_TYPE_BUY))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "Extended Trending SAR"
   if(((m_patterns_usage & 0x04) != 0) && IsPattern_2(POSITION_TYPE_BUY))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "SAR Flip Fakeout"
   if(((m_patterns_usage & 0x08) != 0) && IsPattern_3(POSITION_TYPE_BUY))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 5 is used and "Double SAR Flip with Trend Continuation"
   if(((m_patterns_usage & 0x10) != 0) && IsPattern_4(POSITION_TYPE_BUY))
   {  result += m_pattern_4;
      results++;
   }
//--- if the model 5 is used and "SAR Divergence with Moving Average"
   if(((m_patterns_usage & 0x20) != 0) && IsPattern_5(POSITION_TYPE_BUY))
   {  result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used and "Parabolic SAR Channeling"
   if(((m_patterns_usage & 0x40) != 0) && IsPattern_6(POSITION_TYPE_BUY))
   {  result += m_pattern_6;
      results++;
   }
//--- if the model 7 is used and "SAR and Volume Divergence""
   if(((m_patterns_usage & 0x80) != 0) && IsPattern_7(POSITION_TYPE_BUY))
   {  result += m_pattern_7;
      results++;
   }
//--- if the model 8 is used and "Inverted SAR on Higher Timeframe"
   if(((m_patterns_usage & 0x100) != 0) && IsPattern_8(POSITION_TYPE_BUY))
   {  result += m_pattern_8;
      results++;
   }
//--- if the model 9 is used and "SAR and RSI Overlap"
   if(((m_patterns_usage & 0x200) != 0) && IsPattern_9(POSITION_TYPE_BUY))
   {  result += m_pattern_9;
      results++;
   }
//--- return the result
   //if(result > 0)printf(__FUNCSIG__+" result is: %i",result);
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalSAR::ShortCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Reversal Gap Crossover"
   if(((m_patterns_usage & 0x01) != 0) && IsPattern_0(POSITION_TYPE_SELL))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "SAR Compression Zone"
   if(((m_patterns_usage & 0x02) != 0) && IsPattern_1(POSITION_TYPE_SELL))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "Extended Trending SAR"
   if(((m_patterns_usage & 0x04) != 0) && IsPattern_2(POSITION_TYPE_SELL))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "SAR Flip Fakeout"
   if(((m_patterns_usage & 0x08) != 0) && IsPattern_3(POSITION_TYPE_SELL))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 5 is used and "Double SAR Flip with Trend Continuation"
   if(((m_patterns_usage & 0x10) != 0) && IsPattern_4(POSITION_TYPE_SELL))
   {  result += m_pattern_4;
      results++;
   }
//--- if the model 5 is used and "SAR Divergence with Moving Average"
   if(((m_patterns_usage & 0x20) != 0) && IsPattern_5(POSITION_TYPE_SELL))
   {  result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used and "Parabolic SAR Channeling"
   if(((m_patterns_usage & 0x40) != 0) && IsPattern_6(POSITION_TYPE_SELL))
   {  result += m_pattern_6;
      results++;
   }
//--- if the model 7 is used and "SAR and Volume Divergence""
   if(((m_patterns_usage & 0x80) != 0) && IsPattern_7(POSITION_TYPE_SELL))
   {  result += m_pattern_7;
      results++;
   }
//--- if the model 8 is used and "Inverted SAR on Higher Timeframe"
   if(((m_patterns_usage & 0x100) != 0) && IsPattern_8(POSITION_TYPE_SELL))
   {  result += m_pattern_8;
      results++;
   }
//--- if the model 9 is used and "SAR and RSI Overlap"
   if(((m_patterns_usage & 0x200) != 0) && IsPattern_9(POSITION_TYPE_SELL))
   {  result += m_pattern_9;
      results++;
   }
//--- return the result
   //if(result > 0)printf(__FUNCSIG__+" result is: %i",result);
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| Check for Pattern 0.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_0(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && Base(StartIndex() + 1) > High(StartIndex() + 1) && Base(StartIndex()) < Low(StartIndex()))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Base(StartIndex() + 1) < Low(StartIndex() + 1) && Base(StartIndex()) > High(StartIndex()))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 1.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_1(ENUM_POSITION_TYPE T)
{  double _compression = 0.0;
   if(Compression(T, _compression))
   {  if(T == POSITION_TYPE_BUY && _compression < 0.02*fabs(Base(StartIndex())-Low(StartIndex())))
      {  return(true);
      }
      else if(T == POSITION_TYPE_SELL && _compression < 0.02*fabs(Base(StartIndex())-High(StartIndex())))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 2.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_2(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && 
   Base(StartIndex()) - Base(StartIndex() + 1) > Base(StartIndex() + 1) - Base(StartIndex() + 2) && 
   Base(StartIndex() + 1) - Base(StartIndex() + 2) > Base(StartIndex() + 2) - Base(StartIndex() + 3) && 
   Base(StartIndex() + 2) - Base(StartIndex() + 3) > Base(StartIndex() + 3) - Base(StartIndex() + 4) 
   )
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && 
   Base(StartIndex() + 1) - Base(StartIndex()) > Base(StartIndex() + 2) - Base(StartIndex() + 1) && 
   Base(StartIndex() + 2) - Base(StartIndex() + 1) > Base(StartIndex() + 3) - Base(StartIndex() + 2) && 
   Base(StartIndex() + 3) - Base(StartIndex() + 2) > Base(StartIndex() + 4) - Base(StartIndex() + 3) 
   )
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 3.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_3(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && Base(StartIndex()) < Low(StartIndex()) && Base(StartIndex() + 1) > High(StartIndex() + 1) && Base(StartIndex() + 2) < Low(StartIndex() + 2))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Base(StartIndex()) > High(StartIndex()) && Base(StartIndex() + 1) < Low(StartIndex() + 1) && Base(StartIndex() + 2) > High(StartIndex() + 2))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 4.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_4(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && 
   Base(StartIndex()) < Low(StartIndex()) && 
   Base(StartIndex() + 1) > High(StartIndex() + 1) && 
   Base(StartIndex() + 2) < Low(StartIndex() + 2) && 
   Base(StartIndex() + 3) > High(StartIndex() + 4) && 
   Base(StartIndex() + 4) < Low(StartIndex() + 5)
   )
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && 
   Base(StartIndex()) > High(StartIndex()) && 
   Base(StartIndex() + 1) < Low(StartIndex() + 1) && 
   Base(StartIndex() + 2) > High(StartIndex() + 2) && 
   Base(StartIndex() + 3) < Low(StartIndex() + 4) && 
   Base(StartIndex() + 4) > High(StartIndex() + 5)
   )
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 5.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_5(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && 
   MA(StartIndex()) <= Base(StartIndex()) && 
   Base(StartIndex()) > Base(StartIndex() + 1) && 
   Base(StartIndex() + 1) > Base(StartIndex() + 2) && 
   Close(StartIndex()) < Close(StartIndex() + 1) && 
   Close(StartIndex() + 1) < Close(StartIndex() + 2)
   )
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && 
   MA(StartIndex()) >= Base(StartIndex()) && 
   Base(StartIndex()) < Base(StartIndex() + 1) && 
   Base(StartIndex() + 1) < Base(StartIndex() + 2) && 
   Close(StartIndex()) > Close(StartIndex() + 1) && 
   Close(StartIndex() + 1) > Close(StartIndex() + 2)
   )
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 6.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_6(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && Base(StartIndex()) < Low(StartIndex()))
   {  return(Channel(T));
   }
   else if(T == POSITION_TYPE_SELL && Base(StartIndex()) > High(StartIndex()))
   {  return(Channel(T));
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 7.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_7(ENUM_POSITION_TYPE T)
{  m_sar.Refresh(-1);
   m_sar_alt.Refresh(-1);
   if(T == POSITION_TYPE_BUY && Base(StartIndex()) <= 30.0 && Alt(StartIndex()) <= 30.0)
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Base(StartIndex()) >= 70.0 && Alt(StartIndex()) >= 70.0)
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 8.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_8(ENUM_POSITION_TYPE T)
{  m_sar.Refresh(-1);
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 9.                                             |
//+------------------------------------------------------------------+
bool CSignalSAR::IsPattern_9(ENUM_POSITION_TYPE T)
{  m_sar.Refresh(-1);
   return(false);
}
//+------------------------------------------------------------------+
