//+------------------------------------------------------------------+
//|       WORKING VERSION OF A SCALABLE MULTI-LAYER-PERCEPTRON CLASS.|
//+------------------------------------------------------------------+
//|                                                    stephen NJUKI |
//|                                                     AUGUST, 2024 |
//+------------------------------------------------------------------+
#property version   "1.00"
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#define __EPSILON 1e-8  // non-zero " learning-rate label"
//+------------------------------------------------------------------+
//| Regularization Type Enumerator                                   |
//+------------------------------------------------------------------+
enum Eregularize
{  REGULARIZE_NONE = -1,
   REGULARIZE_L1 = 1,
   REGULARIZE_L2 = 2,
   REGULARIZE_DROPOUT = 3,
   REGULARIZE_ELASTIC = 4
};
//+------------------------------------------------------------------+
//| Learning Rate Type Enumerator                                    |
//+------------------------------------------------------------------+
enum Elearning
{  LEARNING_FIXED = 0,
   LEARNING_STEP_DECAY = 1,
   LEARNING_EXPONENTIAL_DECAY = 2,
   LEARNING_POLYNOMIAL_DECAY = 3,
   LEARNING_INVERSE_TIME_DECAY = 4,
   LEARNING_COSINE_ANNEALING = 5,
   LEARNING_CYCLICAL = 6,
   LEARNING_ONE_CYCLE = 7,
   LEARNING_ADAPTIVE_GRADIENT = 8,
   LEARNING_ADAPTIVE_MOMENTUM = 9,
   LEARNING_ADAPTIVE_DELTA = 10,
};
//+------------------------------------------------------------------+
//| Learning Struct                                                  |
//+------------------------------------------------------------------+
struct Slearning
{  Elearning         type;
   int               epochs;
   double            rate;
   double            initial_rate;
   double            prior_rate;
   double            min_rate;
   double            decay_rate_a;
   double            decay_rate_b;
   int               decay_epoch_steps;
   double            polynomial_power;

                     Slearning()
   {  type = LEARNING_FIXED;
      rate = 0.005;
      prior_rate = 0.1;
      epochs = 50;
      initial_rate = 0.1;
      min_rate = __EPSILON;
      decay_rate_a = 0.25;
      decay_rate_b = 0.75;
      decay_epoch_steps = 10;
      polynomial_power = 1.0;
   };
                    ~Slearning() {};
};
//+------------------------------------------------------------------+
//| Multi-Layer-Perceptron Struct for Constructor Parameters         |
//+------------------------------------------------------------------+
struct Smlp
{  //arch array must be defined with at least 2 non zero values
   //that represent the size of the input layer and output layer
   //If more values than this are provided in the array then the
   //middle values will define the size(s) of the hidden layer(s)
   //first value (index zero) is size of input layer
   //last value (index size-1) is size of output layer
   int               arch[];
   ENUM_ACTIVATION_FUNCTION   activation;
   ENUM_LOSS_FUNCTION         loss;
   double            initial_weight;
   double            initial_bias;
   double            minimum_learning_rate;
   Eregularize       regularization;
   double            regularization_lambda;
   ENUM_MATRIX_NORM  regularization_norm;
   double            regularization_alpha;

                     Smlp()
   {                 ArrayFree(arch);
      activation = AF_SOFTSIGN;
      loss = LOSS_HUBER;
      initial_weight = 0.005;
      initial_bias = 0.0;
      minimum_learning_rate = __EPSILON;
      regularization = REGULARIZE_NONE;
      regularization_lambda = 1.0;
      regularization_norm = MATRIX_NORM_NUCLEAR;
      regularization_alpha = 0.5;
   };
                    ~Smlp() {};
};
//+------------------------------------------------------------------+
//| Multi-Layer-Perceptron Class Interface                           |
//+------------------------------------------------------------------+
class Cmlp
{
protected:
   matrix            weights[];
   vector            biases[];

   matrix            gradients[];
   vector            deltas[];

   matrix            gradients_1st_moment[];
   matrix            gradients_2nd_moment[];

   vector            deltas_1st_moment[];
   vector            deltas_2nd_moment[];

   matrix            sum_gradients[];
   matrix            sum_gradients_update[];

   vector            sum_deltas[];
   vector            sum_deltas_update[];


   vector            hidden_outputs[];

   int               hidden_layers;

   Smlp              THIS;

   double            WeightsNorm();
   vector            Hadamard(vector &A, vector &B);
   matrix            TransposeRow(vector &V);
   matrix            TransposeCol(vector &V);

   void              AdaptiveGradientUpdate(Slearning &Learning);

   void              AdaptiveMomentumUpdate(Slearning &Learning, int EpochIndex);
   void              AdaptiveDeltaUpdate(Slearning &Learning);

   void              MomentumUpdate(double &Parameter, double &Change, double &FirstMoment, double &SecondMoment, Slearning &Learning, int EpochIndex);
   void              DeltaUpdate(double &Parameter, double &Change, double &SumChange, double &SumChangeUpdate, Slearning &Learning);

public:
   vector            inputs;
   vector            label;
   vector            output;
   bool              validated;


   double            Get(ENUM_REGRESSION_METRIC R)
   {                 return(label.RegressionMetric(output, R));
   }

   void              LearningType(Slearning &Learning, int EpochIndex);
   vector            RegularizeTerm(vector &Loss, double Lambda);

   void              Set(vector &Inputs)
   {                 inputs.Init(0);
      inputs.Copy(Inputs);
   }
   void              Get(vector &Label)
   {                 label.Init(0);
      label.Copy(Label);
   }

   bool              Get(string File, double &Criteria, datetime &Version, bool Log = false);
   bool              Set(string File, double Criteria, bool Log = false);

   void              Forward(bool Training = false);
   void              Backward(Slearning &Learning, int EpochIndex = 1);


   void              Cmlp(Smlp &MLP)
   {  validated = false;
      int _layers =    ArraySize(MLP.arch);
      if(_layers >= 2 && _layers <= USHORT_MAX && MLP.arch[ArrayMinimum(MLP.arch)] > 0 && MLP.arch[ArrayMaximum(MLP.arch)] < USHORT_MAX)
      {  ArrayResize(weights, _layers - 1);
         //
         ArrayResize(gradients, _layers - 1);
         ArrayResize(gradients_1st_moment, _layers - 1);
         ArrayResize(gradients_2nd_moment, _layers - 1);
         ArrayResize(sum_gradients, _layers - 1);
         ArrayResize(sum_gradients_update, _layers - 1);
         //
         //
         ArrayResize(biases, _layers - 1);
         //
         ArrayResize(deltas, _layers - 1);
         ArrayResize(deltas_1st_moment, _layers - 1);
         ArrayResize(deltas_2nd_moment, _layers - 1);
         ArrayResize(sum_deltas, _layers - 1);
         ArrayResize(sum_deltas_update, _layers - 1);
         //
         //
         ArrayResize(hidden_outputs, _layers - 2);
         //
         hidden_layers = _layers - 2;
         bool _norm_validated = true;
         for(int i = 0; i < _layers - 1; i++)
         {  weights[i].Init(MLP.arch[i + 1], MLP.arch[i]);
            weights[i].Fill(MLP.initial_weight);
            biases[i].Init(MLP.arch[i + 1]);
            biases[i].Fill(MLP.initial_bias);
            //
            gradients[i].Init(MLP.arch[i + 1], MLP.arch[i]);
            gradients[i].Fill(0.0);
            //
            gradients_1st_moment[i].Init(MLP.arch[i + 1], MLP.arch[i]);
            gradients_1st_moment[i].Fill(0.0);
            gradients_2nd_moment[i].Init(MLP.arch[i + 1], MLP.arch[i]);
            gradients_2nd_moment[i].Fill(0.0);
            //
            sum_gradients[i].Init(MLP.arch[i + 1], MLP.arch[i]);
            sum_gradients[i].Fill(0.0);
            sum_gradients_update[i].Init(MLP.arch[i + 1], MLP.arch[i]);
            sum_gradients_update[i].Fill(0.0);
            //
            deltas[i].Init(MLP.arch[i + 1]);
            deltas[i].Fill(0.0);
            //
            deltas_1st_moment[i].Init(MLP.arch[i + 1]);
            deltas_1st_moment[i].Fill(0.0);
            deltas_2nd_moment[i].Init(MLP.arch[i + 1]);
            deltas_2nd_moment[i].Fill(0.0);
            //
            sum_deltas[i].Init(MLP.arch[i + 1]);
            sum_deltas[i].Fill(0.0);
            sum_deltas_update[i].Init(MLP.arch[i + 1]);
            sum_deltas_update[i].Fill(0.0);
            //
            if(i < _layers - 2)
            {  hidden_outputs[i].Init(MLP.arch[i + 1]);
               hidden_outputs[i].Fill(0.0);
            }
         }
         output.Init(MLP.arch[_layers - 1]);
         label.Init(MLP.arch[_layers - 1]);
         THIS.activation = MLP.activation;
         THIS.loss = MLP.loss;
         THIS.minimum_learning_rate = MLP.minimum_learning_rate;
         if(_norm_validated)
         {  validated = true;
         }
      }
      else
      {  printf(__FUNCSIG__ +
                " invalid network arch! Settings size is: %i, Max layer size is: %i, Min layer size is: %i, and activation is %s ",
                _layers, MLP.arch[ArrayMaximum(MLP.arch)], MLP.arch[ArrayMinimum(MLP.arch)], EnumToString(MLP.activation)
               );
      }
   };
   void              ~Cmlp(void) { };
};
//+------------------------------------------------------------------+
//| Learning Type Function                                           |
//+------------------------------------------------------------------+
void Cmlp::LearningType(Slearning &Learning, int EpochIndex)
{  if(Learning.type == LEARNING_STEP_DECAY)
   {  int _epoch_index = int(MathFloor((Learning.epochs - EpochIndex) / Learning.decay_epoch_steps));
      Learning.rate = Learning.initial_rate * pow(Learning.decay_rate_a, _epoch_index);
   }
   else if(Learning.type == LEARNING_EXPONENTIAL_DECAY)
   {  Learning.rate = Learning.initial_rate * exp(-1.0 * Learning.decay_rate_a * (Learning.epochs - EpochIndex + 1));
   }
   else if(Learning.type == LEARNING_POLYNOMIAL_DECAY)
   {  Learning.rate = Learning.initial_rate * pow(1.0 - ((Learning.epochs - EpochIndex) / Learning.epochs), Learning.polynomial_power);
   }
   else if(Learning.type == LEARNING_INVERSE_TIME_DECAY)
   {  Learning.rate = Learning.prior_rate / (1.0 + (Learning.decay_rate_a * (Learning.epochs - EpochIndex)));
      Learning.prior_rate = Learning.rate;
   }
   else if(Learning.type == LEARNING_COSINE_ANNEALING)
   {  Learning.rate = Learning.min_rate + (0.5 * (Learning.initial_rate - Learning.min_rate) * (1.0 + MathCos(((Learning.epochs - EpochIndex) * M_PI) / Learning.epochs)));
   }
   else if(Learning.type == LEARNING_CYCLICAL)
   {  double _x = fabs(((2.0 * fmod(Learning.epochs - EpochIndex, Learning.epochs)) / Learning.epochs) - 1.0);
      Learning.rate = Learning.min_rate + ((Learning.initial_rate - Learning.min_rate) * fmax(0.0, (1.0 - _x)));
   }
   else if(Learning.type == LEARNING_ONE_CYCLE)
   {  double _cycle_position = (double)((Learning.epochs - EpochIndex) % (2 * Learning.epochs)) / (2.0 * Learning.epochs);
      if(_cycle_position <= 0.5)
      {  Learning.rate = Learning.min_rate + (2.0 * _cycle_position * (Learning.initial_rate - Learning.min_rate));
      }
      else
      {  Learning.rate = Learning.initial_rate - (2.0 * (_cycle_position - 0.5) * (Learning.initial_rate - Learning.min_rate));
      }
   }
}
//+------------------------------------------------------------------+
//| Regularize Term Function                                         |
//+------------------------------------------------------------------+
vector Cmlp::RegularizeTerm(vector &Label, double Lambda)
{  vector _regularized;
   _regularized.Init(Label.Size());
   double _term = 0.0;
   double _mse = output.Loss(Label, LOSS_MSE);
   double _weights_norm = WeightsNorm();
   if(THIS.regularization == REGULARIZE_L1)
   {  _term = _mse + (Lambda * _weights_norm);
   }
   else if(THIS.regularization == REGULARIZE_L2)
   {  _term = _mse + (Lambda * _weights_norm * _weights_norm);
   }
   else if(THIS.regularization == REGULARIZE_ELASTIC)
   {  _term =  _mse + (THIS.regularization_alpha * (Lambda * _weights_norm)) +
               ((1.0 - THIS.regularization_alpha) * (Lambda * _weights_norm * _weights_norm));
   }
   _regularized.Fill(_term);
   return(_regularized);
}
//+------------------------------------------------------------------+
//| Normalised Weights Sum                                           |
//+------------------------------------------------------------------+
double Cmlp::WeightsNorm(void)
{  double _norm = 0.0;
   for(int h = hidden_layers; h >= 0;  h--)
   {  _norm += weights[h].Norm(THIS.regularization_norm);
   }
   return(_norm);
}
//+------------------------------------------------------------------+
//| Vector to Matrix-Row Tranposer                                   |
//+------------------------------------------------------------------+
matrix Cmlp::TransposeRow(vector &V)
{  matrix _m;
   _m.Init(1, V.Size());
   _m.Fill(0.0);
   for(int i = 0; i < int(V.Size()); i++)
   {  _m[0][i] = V[i];
   }
   return(_m);
}
//+------------------------------------------------------------------+
//| Vector to Matrix-Column Tranposer                                |
//+------------------------------------------------------------------+
matrix Cmlp::TransposeCol(vector &V)
{  matrix _m;
   _m.Init(V.Size(), 1);
   _m.Fill(0.0);
   for(int i = 0; i < int(V.Size()); i++)
   {  _m[i][0] = V[i];
   }
   return(_m);
}
//+------------------------------------------------------------------+
//| Hadamard Vector Product                                          |
//+------------------------------------------------------------------+
vector Cmlp::Hadamard(vector &A, vector &B)
{  vector _c;
   _c.Init(A.Size());
   _c.Fill(0.0);
   if(A.Size() == B.Size() && B.Size() > 0)
   {  _c = A * B;
   }
   return(_c);
}
//+------------------------------------------------------------------+
//| FORWARD PROPAGATION THROUGH THE MULTI-LAYER-PERCEPTRON.          |
//+------------------------------------------------------------------+
//|                                                                  |
//| -Extra Validation check of MLP architecture settings is performed|
//|  at run-time.                                                    |
//|  Chcecking of 'validation' parameter should ideally be performed |
//|  at class instance initialisation.                               |
//|                                                                  |
//| -Input data is normalized if normalization type was selected at  |
//|  class instance initialisation.                                  |
//+------------------------------------------------------------------+
void Cmlp::Forward(bool Training = false)
{  if(!validated)
   {  printf(__FUNCSIG__ + " invalid network arch! ");
      return;
   }
//
   for(int h = 0; h <= hidden_layers; h++)
   {  vector _output;
      _output.Init(output.Size());
      if(h == 0)
      {  _output = weights[h].MatMul(inputs);
      }
      else if(h > 0)
      {  _output = weights[h].MatMul(hidden_outputs[h - 1]);
      }
      if(Training && THIS.regularization == REGULARIZE_DROPOUT)
      {  int _drop = MathRand() % int(_output.Size());
         _output[_drop] = 0.0;
      }
      _output += biases[h];
      if(h < hidden_layers)
      {  _output.Activation(hidden_outputs[h], THIS.activation);
      }
      else if(h == hidden_layers)
      {  _output.Activation(output, THIS.activation);
      }
   }
}
//+------------------------------------------------------------------+
//| BACKWARD PROPAGATION OF THE MULTI-LAYER-PERCEPTRON.              |
//+------------------------------------------------------------------+
//|                                                                  |
//| -Extra Validation check of MLP architecture settings is performed|
//|  at run-time.                                                    |
//|  Chcecking of 'validation' parameter should ideally be performed |
//|  at class instance initialisation.                               |
//|                                                                  |
//| -Run-time Validation of learning rate, decay rates and epoch     |
//|  index is performed as these are optimisable inputs.             |
//+------------------------------------------------------------------+
void Cmlp::Backward(Slearning &Learning, int EpochIndex = 1)
{  if(!validated)
   {  printf(__FUNCSIG__ + " invalid network arch! ");
      return;
   }
   if(Learning.rate <= 0.0 || Learning.rate >= 1.0)
   {  printf(__FUNCSIG__ + " Learning Rate out of bounds! Should be between 0.0 to +1.0 range. ");
      return;
   }
   if(Learning.decay_rate_a <= 0.0 || Learning.decay_rate_a >= 1.0)
   {  printf(__FUNCSIG__ + " Decay Rate A out of bounds! Should be between 0.0 to +1.0 range. ");
      return;
   }
   if(Learning.decay_rate_b <= 0.0 || Learning.decay_rate_b >= 1.0)
   {  printf(__FUNCSIG__ + " Decay Rate B out of bounds! Should be between 0.0 to +1.0 range. ");
      return;
   }
   if(EpochIndex < 0)
   {  printf(__FUNCSIG__ + " Epoch Index Should start from 1. ");
      return;
   }
//COMPUTE DELTAS
   vector _last, _last_derivative;
   _last.Init(inputs.Size());
   if(hidden_layers == 0)
   {  _last = weights[hidden_layers].MatMul(inputs);
   }
   else if(hidden_layers > 0)
   {  _last = weights[hidden_layers].MatMul(hidden_outputs[hidden_layers - 1]);
   }
   _last.Derivative(_last_derivative, THIS.activation);
   vector _last_loss = output.LossGradient(label, THIS.loss);
   _last_loss += RegularizeTerm(label, THIS.regularization_lambda);
   deltas[hidden_layers] = Hadamard(_last_loss, _last_derivative);
   if(hidden_layers > 0)
   {  for(int h = hidden_layers - 1; h >= 0;  h--)
      {  vector _next, _next_derivative;
         _next.Init(weights[h].Rows());
         if(h - 1 >= 0)
         {  _next = weights[h].MatMul(hidden_outputs[h - 1]);
         }
         else
         {  _next = weights[h].MatMul(inputs);
         }
         _next.Derivative(_next_derivative, THIS.activation);
         matrix _next_weights;
         _next_weights.Copy(weights[h + 1]);
         _next_weights.Transpose();
         vector _next_product = deltas[h + 1].MatMul(_next_weights);
         deltas[h] = Hadamard(_next_product, _next_derivative);
      }
   }
//COMPUTE GRADIENTS
   for(int h = hidden_layers; h >= 0;  h--)
   {  if(h > 0)
      {  gradients[h] = TransposeCol(deltas[h]).MatMul(TransposeRow(hidden_outputs[h - 1]));
      }
      else if(h == 0)
      {  gradients[h] = TransposeCol(deltas[h]).MatMul(TransposeRow(inputs));
      }
   }
// UPDATE WEIGHTS AND BIASES
// Non-Adaptive
   if(int(Learning.type) < 8)
   {  for(int h = hidden_layers; h >= 0;  h--)
      {  weights[h] -= Learning.rate * gradients[h];
         biases[h] -= Learning.rate * deltas[h];
      }
   }
// Adaptive
   else if(int(Learning.type) >= 8)
   {  if(Learning.type == LEARNING_ADAPTIVE_GRADIENT)
      {  AdaptiveGradientUpdate(Learning);
      }
      else if(Learning.type == LEARNING_ADAPTIVE_MOMENTUM)
      {  AdaptiveMomentumUpdate(Learning, EpochIndex);
      }
      else if(Learning.type == LEARNING_ADAPTIVE_DELTA)
      {  AdaptiveDeltaUpdate(Learning);
      }
   }
}
//+------------------------------------------------------------------+
//| Adaptive Gradient Update function                                |
//+------------------------------------------------------------------+
void Cmlp::AdaptiveGradientUpdate(Slearning &Learning)
{  for(int h = hidden_layers; h >= 0;  h--)
   {  for(int i = 0; i < int(weights[h].Rows()); i++)
      {  for(int j = 0; j < int(weights[h].Cols()); j++)
         {  sum_gradients[h][i][j] += (gradients[h][i][j] * gradients[h][i][j]);
            weights[h][i][j] -= ((Learning.rate / (MathSqrt(sum_gradients[h][i][j]) + THIS.minimum_learning_rate)) * gradients[h][i][j]);
         }
         biases[h][i] -= Learning.rate * deltas[h][i];
      }
   }
}
//+------------------------------------------------------------------+
//| Adaptive Momentum Estimation function                            |
//+------------------------------------------------------------------+
void Cmlp::AdaptiveMomentumUpdate(Slearning &Learning, int EpochIndex)
{  for(int h = hidden_layers; h >= 0;  h--)
   {  for(int i = 0; i < int(weights[h].Rows()); i++)
      {  for(int j = 0; j < int(weights[h].Cols()); j++)
         {  MomentumUpdate(weights[h][i][j], gradients[h][i][j], gradients_1st_moment[h][i][j], gradients_2nd_moment[h][i][j], Learning, EpochIndex);
         }
         MomentumUpdate(biases[h][i], deltas[h][i], deltas_1st_moment[h][i], deltas_2nd_moment[h][i], Learning, EpochIndex);
      }
   }
}
//+------------------------------------------------------------------+
//| Adaptive Delta Update function                                   |
//+------------------------------------------------------------------+
void Cmlp::AdaptiveDeltaUpdate(Slearning &Learning)
{  for(int h = hidden_layers; h >= 0;  h--)
   {  for(int i = 0; i < int(weights[h].Rows()); i++)
      {  for(int j = 0; j < int(weights[h].Cols()); j++)
         {  DeltaUpdate(weights[h][i][j], gradients[h][i][j], sum_gradients[h][i][j], sum_gradients_update[h][i][j], Learning);
         }
         DeltaUpdate(biases[h][i], deltas[h][i], sum_deltas[h][i], sum_deltas_update[h][i], Learning);
      }
   }
}
//+------------------------------------------------------------------+
//| Momentum Update method
//+------------------------------------------------------------------+
void Cmlp::MomentumUpdate(double &Parameter, double &Change, double &FirstMoment, double &SecondMoment, Slearning &Learning, int EpochIndex)
{  FirstMoment = (Learning.decay_rate_a * FirstMoment) + ((1.0 - Learning.decay_rate_a) * Change);
   SecondMoment = (Learning.decay_rate_b * SecondMoment) + ((1.0 - Learning.decay_rate_b) * Change * Change);
   double _1st_gradient = FirstMoment / (1.0 - pow(Learning.decay_rate_a, EpochIndex));
   double _2nd_gradient = SecondMoment / (1.0 - pow(Learning.decay_rate_b, EpochIndex));
   Parameter -= Learning.rate * ((_1st_gradient + THIS.minimum_learning_rate) / (sqrt(_2nd_gradient) + THIS.minimum_learning_rate)) * Change;
}
//+------------------------------------------------------------------+
//| Delta Update method
//+------------------------------------------------------------------+
void Cmlp::DeltaUpdate(double &Parameter, double &Change, double &SumChange, double &SumChangeUpdate, Slearning &Learning)
{  SumChange = (Learning.decay_rate_a * SumChange * SumChange) + ((1.0 - Learning.decay_rate_a) * Change * Change);
   double _update = ((sqrt(SumChange) + THIS.minimum_learning_rate) / (sqrt(SumChangeUpdate) + THIS.minimum_learning_rate)) * Change;
   Parameter -= _update;
   SumChangeUpdate = (Learning.decay_rate_a * SumChangeUpdate * SumChangeUpdate) + ((1.0 - Learning.decay_rate_a) * _update * _update);
}
//+------------------------------------------------------------------+
//| Get Weights and Biases in import                                 |
//+------------------------------------------------------------------+
bool Cmlp::Get(string File, double &Criteria, datetime &Version, bool Log = false)
{  ResetLastError();
   int _handle = FileOpen(MQLInfoString(MQL_PROGRAM_NAME) + "_" + File + ".bin", FILE_READ | FILE_BIN | FILE_COMMON);
   if(_handle != INVALID_HANDLE)
   {  Criteria = FileReadDouble(_handle);
      Version = FileReadInteger(_handle);
      if(Log)
      {  PrintFormat(__FUNCSIG__ + " for the file: %s, we have the criteria: %.2f, logged on: %s ",
                     File, Criteria, TimeToString(Version));
      }
      for(int i = 0; i < int(weights.Size()); i++)
      {  for(int j = 0; j < int(weights[i].Rows()); j++)
         {  for(int k = 0; k < int(weights[i].Cols()); k++)
            {  weights[i][j][k] = FileReadDouble(_handle);
               if(Log)
               {  PrintFormat(__FUNCSIG__ +
                              " weight is: %.2f, for layer %i, at row: %i, and col: %i",
                              weights[i][j][k], i, j, k);
               }
            }
         }
      }
      for(int i = 0; i < int(biases.Size()); i++)
      {  for(int j = 0; j < int(biases[i].Size()); j++)
         {  biases[i][j] = FileReadDouble(_handle);
            if(Log)
            {  PrintFormat(__FUNCSIG__ +
                           " %.5f is the bias for layer %i, at row: %i",
                           biases[i][j], i, j);
            }
         }
      }
      FileClose(_handle);
   }
   else
   {  printf(__FUNCSIG__ + " failed to create get handle for: " + File + ", err: " + IntegerToString(GetLastError()));
      FileClose(_handle);
      return(false);
   }
   FileClose(_handle);
   return(true);
}
//+------------------------------------------------------------------+
//| Set Weights and Biases for export                                |
//+------------------------------------------------------------------+
bool Cmlp::Set(string File, double Criteria, bool Log = false)
{  ResetLastError();
   int _handle = FileOpen(MQLInfoString(MQL_PROGRAM_NAME) + "_" + File + ".bin", FILE_WRITE | FILE_BIN | FILE_COMMON);
   if(_handle != INVALID_HANDLE)
   {  FileWriteDouble(_handle, Criteria);
      FileWriteInteger(_handle, int(TimeCurrent()));
      if(Log)
      {  PrintFormat(__FUNCSIG__ + " for the file: %s, we have the criteria: %.2f, logged on: %s ",
                     File, Criteria, TimeToString(TimeCurrent()));
      }
      for(int i = 0; i < int(weights.Size()); i++)
      {  for(int j = 0; j < int(weights[i].Rows()); j++)
         {  for(int k = 0; k < int(weights[i].Cols()); k++)
            {  FileWriteDouble(_handle, weights[i][j][k]);
               if(Log)
               {  PrintFormat(__FUNCSIG__ +
                              " weight is: %.2f, for layer %i, at row: %i, and col: %i",
                              weights[i][j][k], i, j, k);
               }
            }
         }
      }
      for(int i = 0; i < int(biases.Size()); i++)
      {  for(int j = 0; j < int(biases[i].Size()); j++)
         {  FileWriteDouble(_handle, biases[i][j]);
            if(Log)
            {  PrintFormat(__FUNCSIG__ +
                           " %.5f is the bias for layer %i, at row: %i",
                           biases[i][j], i, j);
            }
         }
      }
      FileClose(_handle);
   }
   else
   {  printf(__FUNCSIG__ + " failed to create set handle for: " + File + ", err: " + IntegerToString(GetLastError()));
      FileClose(_handle);
      return(false);
   }
   FileClose(_handle);
   return(true);
}
//+------------------------------------------------------------------+
