//+------------------------------------------------------------------+
//|                                                      NewsTrading |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                            https://www.mql5.com/en/users/kaaiblo |
//+------------------------------------------------------------------+
#include <Trade\Trade.mqh>
#include <Trade\OrderInfo.mqh>
#include <Trade\SymbolInfo.mqh>
#include "RiskManagement.mqh"
#include "TimeManagement.mqh"
//+------------------------------------------------------------------+
//|TradeManagement class                                             |
//+------------------------------------------------------------------+
class CTradeManagement:CRiskManagement
  {
private:
   CTrade            Trade;//Trade class object
   CSymbolProperties CSymbol;//SymbolProperties class object
   CTimeManagement   CTime;//TimeManagement class object
   bool              TradeResult;//boolean to store trade result
   double            mySL;//double variable to store Stoploss
   double            myTP;//double variable to store Takeprofit
public:
   //--- Class constructor
                     CTradeManagement(string SYMBOL=NULL)
     {
      //--- Set symbol name
      CSymbol.SetSymbolName(SYMBOL);
     }
   //--- Class destructor
                    ~CTradeManagement(void) {}
   //--- Will retrieve if there are any open trades
   bool              OpenTrade(ENUM_POSITION_TYPE Type,ulong Magic,string COMMENT=NULL);
   //--- Will retrieve if there are any deals
   bool              OpenedDeal(ENUM_DEAL_TYPE Type,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open buy trade
   bool              Buy(double SL,double TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open buy trade with integer SL
   bool              Buy(int SL,double TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open buy trade with integer TP
   bool              Buy(double SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open buy trade with integer SL & TP
   bool              Buy(int SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open sell trade
   bool              Sell(double SL,double TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open sell trade with integer SL
   bool              Sell(int SL,double TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open sell trade with integer TP
   bool              Sell(double SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt open sell trade with integer SL & TP
   bool              Sell(int SL,int TP,ulong Magic,string COMMENT=NULL);
  };

//+------------------------------------------------------------------+
//|Will retrieve if there are any open trades                        |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenTrade(ENUM_POSITION_TYPE Type,ulong Magic,string COMMENT=NULL)
  {
//--- Iterate through all open positions
   for(int i=0; i<PositionsTotal(); i++)
     {
      //--- Check if Position ticket is above zero
      if(PositionGetTicket(i)>0)
        {
         //--- Check if the Position's Symbol,Magic,Type,Comment is correct
         if(PositionGetString(POSITION_SYMBOL)==CSymbol.GetSymbolName()&&PositionGetInteger(POSITION_MAGIC)==Magic
            &&PositionGetInteger(POSITION_TYPE)==Type&&PositionGetString(POSITION_COMMENT)==COMMENT)
           {
            //--- Return true when there is an open position
            return true;
           }
        }
     }
//--- No open positions found.
   return false;
  }

//+------------------------------------------------------------------+
//|Will retrieve if there are any deals                              |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenedDeal(ENUM_DEAL_TYPE Type,ulong Magic,string COMMENT=NULL)
  {
//--- Check History starting from 2 minutes ago
   if(HistorySelect(CTime.TimeMinusOffset(TimeTradeServer(),CTime.MinutesS(2)),TimeTradeServer()))
     {
      //--- Iterate through all history deals
      for(int i=0; i<HistoryDealsTotal(); i++)
        {
         //--- Assign history deal ticket
         ulong ticket = HistoryDealGetTicket(i);
         //--- Check if ticket is more than zero
         if(ticket>0)
           {
            //--- Check if the Deal's Symbol,Magic,Type,Comment is correct
            if(HistoryDealGetString(ticket,DEAL_SYMBOL)==CSymbol.GetSymbolName()&&
               HistoryDealGetInteger(ticket,DEAL_MAGIC)==Magic&&HistoryDealGetInteger(ticket,DEAL_TYPE)==Type
               &&HistoryDealGetString(ticket,DEAL_COMMENT)==COMMENT)
              {
               //--- Return true when there are any deals
               return true;
              }
           }
        }
     }
//--- No deals found.
   return false;
  }

//+------------------------------------------------------------------+
//|Will attempt open buy trade                                       |
//+------------------------------------------------------------------+
bool CTradeManagement::Buy(double SL,double TP,ulong Magic,string COMMENT=NULL)
  {
//--- Normalize the SL Price
   CSymbol.NormalizePrice(SL);
//--- Normalize the TP Price
   CSymbol.NormalizePrice(TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_BUY);
//--- Set open price for Risk management calculation
   OpenPrice = CSymbol.Ask();
//--- Set close price for Risk management calculation
   ClosePrice = SL;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals already
   if(!OpenTrade(POSITION_TYPE_BUY,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_BUY,Magic,COMMENT))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin();i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         //--- Open trade with a Lot-size not more than max-lot
         TradeResult = Trade.Buy((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,CSymbol.GetSymbolName(),CSymbol.Ask(),SL,TP,COMMENT);
         //--- Check if trade failed.
         if(!TradeResult)
           {
            return TradeResult;
           }
        }
     }
   else
     {
      //--- Trade failed because there is an open trade or opened deal
      return false;
     }
//--- Return trade result.
   return TradeResult;
  }

//+------------------------------------------------------------------+
//|Will attempt open buy trade with integer SL                       |
//+------------------------------------------------------------------+
bool CTradeManagement::Buy(int SL,double TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL=CSymbol.Bid()-mySL;
//--- Open trade
   return Buy(mySL,TP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will attempt open buy trade with integer TP                       |
//+------------------------------------------------------------------+
bool CTradeManagement::Buy(double SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP+=CSymbol.Ask();
//--- Open trade
   return Buy(SL,myTP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will attempt open buy trade with integer SL & TP                  |
//+------------------------------------------------------------------+
bool CTradeManagement::Buy(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL=CSymbol.Bid()-mySL;
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP+=CSymbol.Ask();
//--- Open trade
   return Buy(mySL,myTP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will attempt open sell trade                                      |
//+------------------------------------------------------------------+
bool CTradeManagement::Sell(double SL,double TP,ulong Magic,string COMMENT=NULL)
  {
//--- Normalize the SL Price
   CSymbol.NormalizePrice(SL);
//--- Normalize the TP Price
   CSymbol.NormalizePrice(TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_SELL);
//--- Set open price for Risk management calculation
   OpenPrice = CSymbol.Bid();
//--- Set close price for Risk management calculation
   ClosePrice = SL;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals already
   if(!OpenTrade(POSITION_TYPE_SELL,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_SELL,Magic,COMMENT))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin();i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         //--- Open trade with a Lot-size not more than max-lot
         TradeResult = Trade.Sell((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,CSymbol.GetSymbolName(),CSymbol.Bid(),SL,TP,COMMENT);
         //--- Check if trade failed.
         if(!TradeResult)
           {
            return TradeResult;
           }
        }
     }
   else
     {
      //--- Trade failed because there is an open trade or opened deal
      return false;
     }
//--- Return trade result.
   return TradeResult;
  }

//+------------------------------------------------------------------+
//|Will attempt open sell trade with integer SL                      |
//+------------------------------------------------------------------+
bool CTradeManagement::Sell(int SL,double TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL+=CSymbol.Ask();
//--- Open trade
   return Sell(mySL,TP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will attempt open sell trade with integer TP                      |
//+------------------------------------------------------------------+
bool CTradeManagement::Sell(double SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP=CSymbol.Bid()-myTP;
//--- Open trade
   return Sell(SL,myTP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will attempt open sell trade with integer SL & TP                 |
//+------------------------------------------------------------------+
bool CTradeManagement::Sell(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL+=CSymbol.Ask();
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP=CSymbol.Bid()-myTP;
//--- Open trade
   return Sell(mySL,myTP,Magic,COMMENT);
  }
//+------------------------------------------------------------------+
