//+------------------------------------------------------------------+
//|                                                      NewsTrading |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                            https://www.mql5.com/en/users/kaaiblo |
//+------------------------------------------------------------------+
#include <Trade/SymbolInfo.mqh>
//Enumeration for Spread rating
enum SpreadRating
  {
   SpreadRating_Terrible,//Terrible
   SpreadRating_Bad,//Bad
   SpreadRating_Normal,//Normal
   SpreadRating_Good,//Good
   SpreadRating_Excellent//Excellent
  };
bool isLightMode;//Variable to configure Chart color mode
//+------------------------------------------------------------------+
//|SymbolProperties class                                            |
//+------------------------------------------------------------------+
class CSymbolProperties
  {
private:
   double            ASK;//Store Ask Price
   double            BID;//Store Bid Price
   double            LOTSMIN;//Store Minimum Lotsize
   double            LOTSMAX;//Store Maximum Lotsize
   double            LOTSSTEP;//Store Lotsize Step
   double            LOTSLIMIT;//Store Lotsize Limit(Maximum sum of Volume)
   long              STOPLEVEL;//Store Stop level
   long              FREEZELEVEL;//Store Freeze level
   long              TIME;//Store time
   long              DIGITS;//Store Digits
   double            POINT;//Store Point
   double            ORDERSVOLUME;//Store Orders volume
   double            POSITIONSVOLUME;//Store Positions volume
   long              CUSTOM;//Store if Symbol is Custom
   long              BACKGROUND_CLR;//Store Symbol's background color
   CSymbolInfo       CSymbol;//Creating class CSymbolInfo's Object

public:
                     CSymbolProperties(void);//Constructor
   //-- Set Symbol Name
   bool              SetSymbolName(string SYMBOL)
     {
      //-- If Symbol's name was successfully set.
      if(!CSymbol.Name((SYMBOL==NULL)?Symbol():SYMBOL))
        {
         Print("Invalid Symbol: ",SYMBOL);
         return false;
        }
      return true;
     }

   //-- Retrieve Symbol's name
   string            GetSymbolName()
     {
      return CSymbol.Name();
     }
   double            Ask(string SYMBOL=NULL);//Retrieve Ask Price
   double            Bid(string SYMBOL=NULL);//Retrieve Bid Price
   double            ContractSize(string SYMBOL=NULL);//Retrieve Contract Size
   double            LotsMin(string SYMBOL=NULL);//Retrieve Min Volume
   double            LotsMax(string SYMBOL=NULL);//Retrieve Max Volume
   double            LotsStep(string SYMBOL=NULL);//Retrieve Volume Step
   double            LotsLimit(string SYMBOL=NULL);//Retrieve Volume Limit
   int               Spread(string SYMBOL=NULL);//Retrieve Spread
   bool              SpreadFloat(string SYMBOL=NULL);//Retrieve Spread Float
   SpreadRating      SpreadValue(string SYMBOL=NULL);//Retrieve Spread Rating
   color             SpreadColor(string SYMBOL=NULL);//Retrieve Spread Color
   string            SpreadDesc(string SYMBOL=NULL);//Retrieve Spread Description
   int               StopLevel(string SYMBOL=NULL);//Retrieve Stop Level
   int               FreezeLevel(string SYMBOL=NULL);//Retrieve Freeze Level
   datetime          Time(string SYMBOL=NULL);//Retrieve Symbol's Time
   //-- Normalize Price
   double            NormalizePrice(const double price,string SYMBOL=NULL);
   int               Digits(string SYMBOL=NULL);//Retrieve Symbol's Digits
   double            Point(string SYMBOL=NULL);//Retrieve Symbol's Point
   ENUM_SYMBOL_TRADE_MODE TradeMode(string SYMBOL=NULL);//Retrieve Symbol's Trade Mode
   double            OrdersVolume(string SYMBOL=NULL);//Retrieve Symbol's Orders Volume
   double            PositionsVolume(string SYMBOL=NULL);//Retrieve Symbol's Positions Volume
   string            CurrencyBase(string SYMBOL=NULL);//Retrieve Symbol's Currency Base
   string            CurrencyProfit(string SYMBOL=NULL);//Retrieve Symbol's Currency Profit
   string            CurrencyMargin(string SYMBOL=NULL);//Retrieve Symbol's Currency Margin
   bool              Custom(string SYMBOL=NULL);//Retrieve Symbol's Custom status
   color             Background(string SYMBOL=NULL,bool allow_black=false);//Retrieve Symbol's Background color
   string            Description(string SYMBOL=NULL);//Retrieve Symbol's Description
  };

//+------------------------------------------------------------------+
//|Constructor                                                       |
//+------------------------------------------------------------------+
//Initializing Variables
CSymbolProperties::CSymbolProperties(void):ASK(0.0),BID(0.0),LOTSMIN(0.0),
   LOTSMAX(0.0),LOTSSTEP(0.0),LOTSLIMIT(0.0),DIGITS(0),
   STOPLEVEL(0),ORDERSVOLUME(0.0),FREEZELEVEL(0),
   TIME(0),POINT(0.0),POSITIONSVOLUME(0.0),CUSTOM(0),
   BACKGROUND_CLR(0)
  {
  }

//+------------------------------------------------------------------+
//|Retrieve Ask Price                                                |
//+------------------------------------------------------------------+
double CSymbolProperties::Ask(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_ASK,ASK))
        {
         return ASK;
        }
     }
   Print("Unable to retrieve Symbol's Ask Price");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Bid Price                                                |
//+------------------------------------------------------------------+
double CSymbolProperties::Bid(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_BID,BID))
        {
         return BID;
        }
     }
   Print("Unable to retrieve Symbol's Bid Price");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Contract Size                                            |
//+------------------------------------------------------------------+
double CSymbolProperties::ContractSize(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.Refresh())
        {
         return CSymbol.ContractSize();
        }
     }
   Print("Unable to retrieve Symbol's Contract size");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Min Volume                                               |
//+------------------------------------------------------------------+
double CSymbolProperties::LotsMin(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_VOLUME_MIN,LOTSMIN))
        {
         return LOTSMIN;
        }
     }
   Print("Unable to retrieve Symbol's LotsMin");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Max Volume                                               |
//+------------------------------------------------------------------+
double CSymbolProperties::LotsMax(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_VOLUME_MAX,LOTSMAX))
        {
         return LOTSMAX;
        }
     }
   Print("Unable to retrieve Symbol's LotsMax");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Volume Step                                              |
//+------------------------------------------------------------------+
double CSymbolProperties::LotsStep(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_VOLUME_STEP,LOTSSTEP))
        {
         return LOTSSTEP;
        }
     }
   Print("Unable to retrieve Symbol's LotsStep");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Volume Limit                                             |
//+------------------------------------------------------------------+
double CSymbolProperties::LotsLimit(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_VOLUME_LIMIT,LOTSLIMIT))
        {
         return LOTSLIMIT;
        }
     }
   Print("Unable to retrieve Symbol's LotsLimit");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Spread                                                   |
//+------------------------------------------------------------------+
int CSymbolProperties::Spread(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      return CSymbol.Spread();
     }
   Print("Unable to retrieve Symbol's Spread");
   return 0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Spread Float                                             |
//+------------------------------------------------------------------+
bool CSymbolProperties::SpreadFloat(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      return CSymbol.SpreadFloat();
     }
   Print("Unable to retrieve Symbol's Spread Float");
   return false;//Retrieve false when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Spread Rating                                            |
//+------------------------------------------------------------------+
SpreadRating CSymbolProperties::SpreadValue(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(SpreadFloat(SYMBOL))//Check if Symbol has a floating Spread
        {

         //Declarations
         vector Spreads;
         int SpreadArray[],SpreadAvg=0,SpreadMax=0,SpreadMin=0,
                           SpreadUpper=0,SpreadLower=0,SpreadAvgUpper=0,
                           SpreadAvgLower=0,SpreadMidUpper=0,SpreadMidLower=0;

         //Get Spread data from CopySpread built-in function for 2 weeks using M1 timeframe.
         if(CopySpread(GetSymbolName(),PERIOD_M1,iTime(GetSymbolName(),PERIOD_W1,2),
                       iTime(GetSymbolName(),PERIOD_M1,0),SpreadArray)==-1)
           {
            Print("Error trying to retrieve spread values");
            return SpreadRating_Normal;//Retrieve default value when failed.
           }
         else
           {
            Spreads.Assign(SpreadArray);//Assign spread array into Spreads vector variable

            SpreadMax = int(Spreads.Max());//Assign max spread
            SpreadMin = int(Spreads.Min());//Assign min spread
            SpreadAvg = int(Spreads.Median());//Assign average spread

            //Divide Spread into sectors based of different averages.
            SpreadMidUpper = int((SpreadAvg+SpreadMax)/2);
            SpreadMidLower = int((SpreadAvg+SpreadMin)/2);
            SpreadAvgUpper = int((SpreadAvg+SpreadMidUpper)/2);
            SpreadAvgLower = int((SpreadAvg+SpreadMidLower)/2);
            SpreadUpper = int((SpreadMidUpper+SpreadMax)/2);
            SpreadLower = int((SpreadMidLower+SpreadMin)/2);

            int Spread = Spread(SYMBOL);//Assign Symbol's Spread

            if(Spread<SpreadLower||Spread==SpreadMin)//Excellent
              {
               return SpreadRating_Excellent;
              }
            else
               if(Spread>=SpreadLower&&Spread<SpreadAvgLower)//Good
                 {
                  return SpreadRating_Good;
                 }
               else
                  if(Spread>=SpreadAvgLower&&Spread<=SpreadAvgUpper)//Normal
                    {
                     return SpreadRating_Normal;
                    }
                  else
                     if(Spread>SpreadAvgUpper&&Spread<=SpreadUpper)//Bad
                       {
                        return SpreadRating_Bad;
                       }
                     else//Terrible
                       {
                        return SpreadRating_Terrible;
                       }
           }
        }
      else
        {
         return SpreadRating_Normal;//Retrieve default value when spread is fixed.
        }
     }
   Print("Unable to retrieve Symbol's Spread Rating");
   return SpreadRating_Normal;//Retrieve default value when failed.*/
  }

//+------------------------------------------------------------------+
//|Retrieve Spread Color                                             |
//+------------------------------------------------------------------+
color CSymbolProperties::SpreadColor(string SYMBOL=NULL)
  {
   switch(SpreadValue(SYMBOL))//Get Spread Rating value
     {
      case SpreadRating_Excellent://Excellent Spread
         return (isLightMode)?clrBlue:clrLightCyan;
         break;
      case SpreadRating_Good://Good Spread
         return (isLightMode)?clrCornflowerBlue:clrLightGreen;
         break;
      case SpreadRating_Normal://Normal Spread
         return (isLightMode)?clrBlack:clrWheat;
         break;
      case SpreadRating_Bad://Bad Spread
         return clrOrange;
         break;
      case SpreadRating_Terrible://Terrible Spread
         return clrRed;
         break;
      default://failed to be identified
         return (isLightMode)?clrBlack:clrWheat;//Retrieve default color when failed.
         break;
     }
  }

//+------------------------------------------------------------------+
//|Retrieve Spread Description                                       |
//+------------------------------------------------------------------+
string CSymbolProperties::SpreadDesc(string SYMBOL=NULL)
  {
   switch(SpreadValue(SYMBOL))//Get Spread Rating value
     {
      case SpreadRating_Excellent://Excellent Spread
         return "Excellent";
         break;
      case SpreadRating_Good://Good Spread
         return "Good";
         break;
      case SpreadRating_Normal://Normal Spread
         return "Normal";
         break;
      case SpreadRating_Bad://Bad Spread
         return "Bad";
         break;
      case SpreadRating_Terrible://Terrible Spread
         return "Terrible";
         break;
      default://failed to be identified
         return "Unknown";//Retrieve default value when failed.
         break;
     }
  }

//+------------------------------------------------------------------+
//|Retrieve Stop Level                                               |
//+------------------------------------------------------------------+
int CSymbolProperties::StopLevel(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoInteger(SYMBOL_TRADE_STOPS_LEVEL,STOPLEVEL))
        {
         return int(STOPLEVEL);
        }
     }
   Print("Unable to retrieve Symbol's StopLevel");
   return 0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Freeze Level                                             |
//+------------------------------------------------------------------+
int CSymbolProperties::FreezeLevel(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoInteger(SYMBOL_TRADE_FREEZE_LEVEL,FREEZELEVEL))
        {
         return int(FREEZELEVEL);
        }
     }
   Print("Unable to retrieve Symbol's FreezeLevel");
   return 0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Time                                            |
//+------------------------------------------------------------------+
datetime CSymbolProperties::Time(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoInteger(SYMBOL_TIME,TIME))
        {
         return datetime(TIME);
        }
     }
   Print("Unable to retrieve Symbol's Time");
   TIME=0;
   return datetime(TIME);//Retrieve zero datetime when failed.
  }

//+------------------------------------------------------------------+
//|Normalize Price                                                   |
//+------------------------------------------------------------------+
double CSymbolProperties::NormalizePrice(const double price,string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.Refresh()&&CSymbol.RefreshRates())
        {
         return CSymbol.NormalizePrice(price);
        }
     }
   Print("Unable to Normalize Symbol's Price");
   return price;//Retrieve input parameter when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Digits                                          |
//+------------------------------------------------------------------+
int CSymbolProperties::Digits(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoInteger(SYMBOL_DIGITS,DIGITS))
        {
         return int(DIGITS);
        }
     }
   Print("Unable to retrieve Symbol's Digits");
   return 0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Point                                           |
//+------------------------------------------------------------------+
double CSymbolProperties::Point(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoDouble(SYMBOL_POINT,POINT))
        {
         return POINT;
        }
     }
   Print("Unable to retrieve Symbol's Point");
   return 0.0;//Retrieve zero when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Trade Mode                                      |
//+------------------------------------------------------------------+
ENUM_SYMBOL_TRADE_MODE CSymbolProperties::TradeMode(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.Refresh())
        {
         return CSymbol.TradeMode();
        }
     }
   Print("Unable to retrieve Symbol's TradeMode");
   return SYMBOL_TRADE_MODE_DISABLED;//Retrieve default value when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Orders Volume                                   |
//+------------------------------------------------------------------+
double CSymbolProperties::OrdersVolume(string SYMBOL=NULL)
  {
   ORDERSVOLUME=0.0;
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      for(int i=0; i<OrdersTotal(); i++)
        {
         if(OrderSelect(OrderGetTicket(i)))
           {
            if(OrderGetString(ORDER_SYMBOL)==GetSymbolName())
              {
               ORDERSVOLUME+=OrderGetDouble(ORDER_VOLUME_CURRENT);
              }
           }
        }
     }
   else
     {
      Print("Unable to retrieve Symbol's OrdersVolume");
      return 0.0;//Retrieve zero when failed.
     }
   return ORDERSVOLUME;//Retrieve assigned value.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Positions Volume                                |
//+------------------------------------------------------------------+
double CSymbolProperties::PositionsVolume(string SYMBOL=NULL)
  {
   POSITIONSVOLUME=0.0;
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      for(int i=0; i<PositionsTotal(); i++)
        {
         if(PositionGetTicket(i)>0)
           {
            if(PositionGetString(POSITION_SYMBOL)==GetSymbolName())
              {
               POSITIONSVOLUME+=PositionGetDouble(POSITION_VOLUME);
              }
           }
        }
     }
   else
     {
      Print("Unable to retrieve Symbol's PositionsVolume");
      return 0.0;//Retrieve zero when failed.
     }
   return POSITIONSVOLUME;//Retrieve assigned value.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Currency Base                                   |
//+------------------------------------------------------------------+
string CSymbolProperties::CurrencyBase(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.Refresh())
        {
         return CSymbol.CurrencyBase();
        }
     }
   Print("Unable to retrieve Symbol's CurrencyBase");
   return "";//Retrieve an empty string when failed.
  }
//+------------------------------------------------------------------+
//|Retrieve Symbol's Currency Profit                                 |
//+------------------------------------------------------------------+
string CSymbolProperties::CurrencyProfit(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.Refresh())
        {
         return CSymbol.CurrencyProfit();
        }
     }
   Print("Unable to retrieve Symbol's CurrencyProfit");
   return "";//Retrieve an empty string when failed.
  }
//+------------------------------------------------------------------+
//|Retrieve Symbol's Currency Margin                                 |
//+------------------------------------------------------------------+
string CSymbolProperties::CurrencyMargin(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.Refresh())
        {
         return CSymbol.CurrencyMargin();
        }
     }
   Print("Unable to retrieve Symbol's CurrencyMargin");
   return "";//Retrieve an empty string when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Custom status                                   |
//+------------------------------------------------------------------+
bool CSymbolProperties::Custom(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoInteger(SYMBOL_CUSTOM,CUSTOM))
        {
         return bool(CUSTOM);
        }
     }
   Print("Unable to retrieve if Symbol is Custom");
   return false;//Retrieve false when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Background color                                |
//+------------------------------------------------------------------+
color CSymbolProperties::Background(string SYMBOL=NULL,bool allow_black=false)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      if(CSymbol.InfoInteger(SYMBOL_BACKGROUND_COLOR,BACKGROUND_CLR))
        {
         /*Avoid any Symbol black background color */
         BACKGROUND_CLR = ((ColorToString(color(BACKGROUND_CLR))=="0,0,0")&&!allow_black)?
                          long(StringToColor("236,236,236")):BACKGROUND_CLR;
         return color(BACKGROUND_CLR);
        }
     }
   Print("Unable to retrieve Symbol's Background color");
   return color(StringToColor("236,236,236"));//Retrieve a lightish gray color when failed.
  }

//+------------------------------------------------------------------+
//|Retrieve Symbol's Description                                     |
//+------------------------------------------------------------------+
string CSymbolProperties::Description(string SYMBOL=NULL)
  {
   if(SetSymbolName(SYMBOL))//Set Symbol
     {
      return CSymbol.Description();
     }
   Print("Unable to retrieve Symbol's Description");
   return "";//Retrieve an empty string when failed.
  }
//+------------------------------------------------------------------+
