//+------------------------------------------------------------------+
//|                                                      NewsTrading |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                            https://www.mql5.com/en/users/kaaiblo |
//+------------------------------------------------------------------+
//--- width and height of the canvas (used for drawing)
#define IMG_WIDTH  200
#define IMG_HEIGHT 100
//--- enable to set color format
ENUM_COLOR_FORMAT clr_format=COLOR_FORMAT_XRGB_NOALPHA;
//--- drawing array (buffer)
uint ExtImg[IMG_WIDTH*IMG_HEIGHT];

#include "News.mqh"
CNews NewsObject;//Class object for News
#include "TimeManagement.mqh"
CTimeManagement CTM;//Class object for Time Management
#include "WorkingWithFolders.mqh"
CFolders Folder;//Class object for Folders
#include "ChartProperties.mqh"
CChartProperties Chart;//Class object for Chart Properties
#include "RiskManagement.mqh"
CRiskManagement CRisk;//Class object for Risk Management
#include "CommonGraphics.mqh"
CCommonGraphics *CGraphics;//Class pointer object for Common Graphics
CCandleProperties *CP;//Class pointer object for Candle Properties
#include "TradeManagement.mqh"
CTradeManagement Trade;//Class object for Trade Management

//--- used to separate Input Menu
enum iSeparator
  {
   Delimiter//__________________________
  };

//--- for chart color Mode selection
enum DisplayMode
  {
   Display_LightMode,//LIGHT MODE
   Display_DarkMode//DARK MODE
  };

sinput group "+--------|   DISPLAY   |--------+";
sinput DisplayMode iDisplayMode=Display_LightMode;//CHART COLOUR MODE
sinput Choice iDisplay_NewsInfo=Yes;//DISPLAY NEWS INFO
sinput Choice iDisplay_EventObj=Yes;//DISPLAY EVENT OBJ
sinput Choice iDisplay_Spread=Yes;//DISPLAY SPREAD RATING
sinput Choice iDisplay_Date=Yes;//DISPLAY DATE
sinput group "";
sinput group "+--------|   DST SCHEDULE   |--------+";
input DSTSchedule ScheduleDST=AutoDst_Selection;//SELECT DST OPTION
sinput iSeparator iCustomSchedule=Delimiter;//__________________________
sinput iSeparator iCustomScheduleL=Delimiter;//CUSTOM DST
input DST_type CustomSchedule=DST_NONE;//SELECT CUSTOM DST
sinput group "";
sinput group "+--------| RISK MANAGEMENT |--------+";
input RiskOptions RISK_Type=MINIMUM_LOT;//SELECT RISK OPTION
input RiskFloor RISK_Mini=RiskFloorMin;//RISK FLOOR
input double RISK_Mini_Percent=75;//MAX-RISK [100<-->0.01]%
input RiskCeil  RISK_Maxi=RiskCeilMax;//RISK CEILING
sinput iSeparator iRisk_1=Delimiter;//__________________________
sinput iSeparator iRisk_1L=Delimiter;//PERCENTAGE OF [BALANCE | FREE-MARGIN]
input double Risk_1_PERCENTAGE=3;//[100<-->0.01]%
sinput iSeparator iRisk_2=Delimiter;//__________________________
sinput iSeparator iRisk_2L=Delimiter;//AMOUNT PER [BALANCE | FREE-MARGIN]
input double Risk_2_VALUE=1000;//[BALANCE | FREE-MARGIN]
input double Risk_2_AMOUNT=10;//EACH AMOUNT
sinput iSeparator iRisk_3=Delimiter;//__________________________
sinput iSeparator iRisk_3L=Delimiter;//LOTSIZE PER [BALANCE | FREE-MARGIN]
input double Risk_3_VALUE=1000;//[BALANCE | FREE-MARGIN]
input double Risk_3_LOTSIZE=0.1;//EACH LOTS(VOLUME)
sinput iSeparator iRisk_4=Delimiter;//__________________________
sinput iSeparator iRisk_4L=Delimiter;//CUSTOM LOTSIZE
input double Risk_4_LOTSIZE=0.01;//LOTS(VOLUME)
sinput iSeparator iRisk_5=Delimiter;//__________________________
sinput iSeparator iRisk_5L=Delimiter;//PERCENTAGE OF MAX-RISK
input double Risk_5_PERCENTAGE=1;//[100<-->0.01]%
sinput group "";
sinput group "+--------| NEWS SETTINGS |--------+";
input Calendar_Importance iImportance=Calendar_Importance_High;//CALENDAR IMPORTANCE
input Event_Frequency iFrequency=Event_Frequency_ALL;//EVENT FREQUENCY
input Event_Sector iSector=Event_Sector_ALL;//EVENT SECTOR
input Event_Type iType=Event_Type_Indicator;//EVENT TYPE
input Event_Currency iCurrency=Event_Currency_Symbol;//EVENT CURRENCY
sinput group "";
sinput group "+--------| TRADE SETTINGS |--------+";
input uint iStoploss=500;//STOPLOSS [0=NONE]
input uint iTakeprofit=500;//TAKEPROFIT [0=NONE]
input uint iSecondsPreEvent=5;//PRE-ENTRY SEC
input DayOfTheWeek TradingDay=AllDays;//TRADING DAY OF WEEK
sinput group "";
//--- to keep track of start-up time
datetime Startup_date;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- Assign if in LightMode or not
   isLightMode=(iDisplayMode==Display_LightMode)?true:false;
//--- call function for common initialization procedure
   InitCommon();
//--- store Init result
   int InitResult;
   if(!MQLInfoInteger(MQL_TESTER))//Checks whether the program is in the strategy tester
     {
      //--- initialization procedure outside strategy tester
      InitResult=InitNonTester();
     }
   else
     {
      //--- initialization procedure inside strategy tester
      InitResult=InitTester();
     }
//--- Create DB in memory
   NewsObject.CreateEconomicDatabaseMemory();
//--- Initialize Common graphics class pointer object
   CGraphics = new CCommonGraphics(Answer(iDisplay_Date),Answer(iDisplay_Spread),Answer(iDisplay_NewsInfo),Answer(iDisplay_EventObj));
   CGraphics.GraphicsRefresh(iSecondsPreEvent);//-- Create chart objects
//--- Initialize Candle properties pointer object
   CP = new CCandleProperties();
//--- Store start-up time.
   Startup_date = TimeTradeServer();
//--- return Init result
   return InitResult;
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   delete CGraphics;//delete pointer for CCommonGraphics class
   delete CP;//delete pointer for CCandleProperties class
//---
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//--- Run procedures
   Execution();
  }

//+------------------------------------------------------------------+
//|Execute program procedures                                        |
//+------------------------------------------------------------------+
void Execution()
  {
//--- Update realtime Graphic every 1 min
   if(CP.NewCandle(1,PERIOD_M1))
     {
      CGraphics.Block_2_Realtime(iSecondsPreEvent);
     }
//--- function to open trades
   EnterTrade();
//--- Check if not start-up date
   if(!CTM.DateisToday(Startup_date))
     {
      //--- Run every New Daily Candle
      if(CP.NewCandle(2,PERIOD_D1))
        {
         //--- Check if not in strategy tester
         if(!MQLInfoInteger(MQL_TESTER))
           {
            //--- Update/Create DB in Memory
            NewsObject.CreateEconomicDatabaseMemory();
           }
         CGraphics.GraphicsRefresh(iSecondsPreEvent);//-- Create/Re-create chart objects
         //--- Update Realtime Graphics
         CGraphics.Block_2_Realtime(iSecondsPreEvent);
        }
      //--- Check if not in strategy tester
      if(!MQLInfoInteger(MQL_TESTER))
        {
         //--- Run every New Hourly Candle
         if(CP.NewCandle(3,PERIOD_H1))
           {
            //--- Check if DB in Storage needs an update
            if(NewsObject.UpdateRecords())
              {
               //--- initialization procedure outside strategy tester
               InitNonTester();
              }
           }
        }
     }
   else
     {
      //--- Run every New Daily Candle
      if(CP.NewCandle(4,PERIOD_D1))
        {
         //--- Update Event objects on chart
         CGraphics.NewsEvent();
        }
     }
  }

//+------------------------------------------------------------------+
//|function for common initialization procedure                      |
//+------------------------------------------------------------------+
void InitCommon()
  {
//Initializing CRiskManagement variable for Risk options
   RiskProfileOption = RISK_Type;
//Initializing CRiskManagement variable for Risk floor
   RiskFloorOption = RISK_Mini;
//Initializing CRiskManagement variable for RiskFloorMax
   RiskFloorPercentage = (RISK_Mini_Percent>100)?100:
                         (RISK_Mini_Percent<0.01)?0.01:RISK_Mini_Percent;//Percentage cannot be more than 100% or less than 0.01%
//Initializing CRiskManagement variable for Risk ceiling
   RiskCeilOption = RISK_Maxi;
//Initializing CRiskManagement variable for Risk options (PERCENTAGE OF BALANCE and PERCENTAGE OF FREE-MARGIN)
   Risk_Profile_1 = (Risk_1_PERCENTAGE>100)?100:
                    (Risk_1_PERCENTAGE<0.01)?0.01:Risk_1_PERCENTAGE;//Percentage cannot be more than 100% or less than 0.01%
//Initializing CRiskManagement variables for Risk options (AMOUNT PER BALANCE and AMOUNT PER FREE-MARGIN)
   Risk_Profile_2.RiskAmountBoF = Risk_2_VALUE;
   Risk_Profile_2.RiskAmount = Risk_2_AMOUNT;
//Initializing CRiskManagement variables for Risk options (LOTSIZE PER BALANCE and LOTSIZE PER FREE-MARGIN)
   Risk_Profile_3.RiskLotBoF = Risk_3_VALUE;
   Risk_Profile_3.RiskLot = Risk_3_LOTSIZE;
//Initializing CRiskManagement variable for Risk option (CUSTOM LOTSIZE)
   Risk_Profile_4 = Risk_4_LOTSIZE;
//Initializing CRiskManagement variable for Risk option (PERCENTAGE OF MAX-RISK)
   Risk_Profile_5 = (Risk_5_PERCENTAGE>100)?100:
                    (Risk_5_PERCENTAGE<0.01)?0.01:Risk_5_PERCENTAGE;//Percentage cannot be more than 100% or less than 0.01%
//--- Initializing DST Schedule variables
   MyDST = ScheduleDST;
   MySchedule = CustomSchedule;
//--- Initializing News filter variables
   myFrequency=iFrequency;
   myImportance=iImportance;
   mySector=iSector;
   myType=iType;
   myCurrency=iCurrency;
   Chart.ChartRefresh();//Load chart configurations
  }

//+------------------------------------------------------------------+
//|function for initialization procedure outside strategy tester     |
//+------------------------------------------------------------------+
int InitNonTester()
  {
//--- Check if in Strategy tester!
   if(MQLInfoInteger(MQL_TESTER))
     {
      //--- Initialization failed.
      return(INIT_SUCCEEDED);
     }
//--- create OBJ_BITMAP_LABEL object for drawing
   ObjectCreate(0,"STATUS",OBJ_BITMAP_LABEL,0,0,0);
   ObjectSetInteger(0,"STATUS",OBJPROP_XDISTANCE,5);
   ObjectSetInteger(0,"STATUS",OBJPROP_YDISTANCE,22);
//--- specify the name of the graphical resource
   ObjectSetString(0,"STATUS",OBJPROP_BMPFILE,"::PROGRESS");
   uint   w,h;          // variables for receiving text string sizes
   uint    x,y;          // variables for calculation of the current coordinates of text string anchor points
   /*
   In the Do while loop below, the code will check if the terminal is connected to the internet.
   If the the program is stopped the loop will break, if the program is not stopped and the terminal
   is connected to the internet the function CreateEconomicDatabase will be called from the News.mqh header file's
   object called NewsObject and the loop will break once called.
   */
   bool done=false;
   do
     {
      //--- clear the drawing buffer array
      ArrayFill(ExtImg,0,IMG_WIDTH*IMG_HEIGHT,0);

      if(!TerminalInfoInteger(TERMINAL_CONNECTED))
        {
         //-- integer dots used as a loading animation
         static int dots=0;
         //--- set the font
         TextSetFont("Arial",-150,FW_EXTRABOLD,0);
         TextGetSize("Waiting",w,h);//get text width and height values
         //--- calculate the coordinates of the 'Waiting' text
         x=10;//horizontal alignment
         y=IMG_HEIGHT/2-(h/2);//alignment for the text to be centered vertically
         //--- output the 'Waiting' text to ExtImg[] buffer
         TextOut("Waiting",x,y,TA_LEFT|TA_TOP,ExtImg,IMG_WIDTH,IMG_HEIGHT,ColorToARGB(CSymbol.Background()),clr_format);
         //--- calculate the coordinates for the dots after the 'Waiting' text
         x=w+13;//horizontal alignment
         y=IMG_HEIGHT/2-(h/2);//alignment for the text to be centered vertically
         TextSetFont("Arial",-160,FW_EXTRABOLD,0);
         //--- output of dots to ExtImg[] buffer
         TextOut(StringSubstr("...",0,dots),x,y,TA_LEFT|TA_TOP,ExtImg,IMG_WIDTH,IMG_HEIGHT,ColorToARGB(CSymbol.Background()),clr_format);
         //--- update the graphical resource
         ResourceCreate("::PROGRESS",ExtImg,IMG_WIDTH,IMG_HEIGHT,0,0,IMG_WIDTH,clr_format);
         //--- force chart update
         Chart.Redraw();
         dots=(dots==3)?0:dots+1;
         //-- Notify user that program is waiting for connection
         Print("Waiting for connection...");
         Sleep(500);
         continue;
        }
      else
        {
         //--- set the font
         TextSetFont("Arial",-120,FW_EXTRABOLD,0);
         TextGetSize("Getting Ready",w,h);//get text width and height values
         x=20;//horizontal alignment
         y=IMG_HEIGHT/2-(h/2);//alignment for the text to be centered vertically
         //--- output the text 'Getting Ready...' to ExtImg[] buffer
         TextOut("Getting Ready...",x,y,TA_LEFT|TA_TOP,ExtImg,IMG_WIDTH,IMG_HEIGHT,ColorToARGB(CSymbol.Background()),clr_format);
         //--- update the graphical resource
         ResourceCreate("::PROGRESS",ExtImg,IMG_WIDTH,IMG_HEIGHT,0,0,IMG_WIDTH,clr_format);
         //--- force chart update
         Chart.Redraw();
         //-- Notify user that connection is successful
         Print("Connection Successful!");
         NewsObject.CreateEconomicDatabase();//calling the database create function
         done=true;
        }
     }
   while(!done&&!IsStopped());
//-- Delete chart object
   ObjectDelete(0,"STATUS");
//-- force chart to update
   Chart.Redraw();
//--- Initialization succeeded.
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//|function for initialization procedure inside strategy tester      |
//+------------------------------------------------------------------+
int InitTester()
  {
//--- Check if not in Strategy tester!
   if(!MQLInfoInteger(MQL_TESTER))
     {
      //--- Initialization failed.
      return(INIT_SUCCEEDED);
     }
//Checks whether the database file exists
   if(!FileIsExist(NEWS_DATABASE_FILE,FILE_COMMON))
     {
      //--- Warning messages
      Print("Necessary Files Do not Exist!");
      Print("Run Program outside of the Strategy Tester");
      Print("Necessary Files Should be Created First");
      //--- Initialization failed.
      return(INIT_FAILED);
     }
   else
     {
      //Checks whether the latest database date includes the time and date being tested
      datetime latestdate = CTM.TimeMinusOffset(NewsObject.GetLatestNewsDate(),CTM.DaysS());//Day before the latest recorded time in the database
      if(latestdate<TimeTradeServer())
        {
         Print("Necessary Files outdated!");
         Print("To Update Files: Run Program outside of the Strategy Tester");
        }
      Print("Database Dates End at: ",latestdate);
      PrintFormat("Dates after %s will not be available for backtest",TimeToString(latestdate));
     }
//--- Initialization succeeded.
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//|function to open trades                                           |
//+------------------------------------------------------------------+
void EnterTrade()
  {
//--- static variable for storing upcoming event Impact value
   static ENUM_CALENDAR_EVENT_IMPACT Impact=CALENDAR_IMPACT_NA;
//--- Check if Upcoming news date has passed and if upcoming news is not null and if new minute candle has formed.
   if(datetime(UpcomingNews.EventDate)<TimeTradeServer()&&UpcomingNews.CountryName!=NULL&&CP.NewCandle(5,PERIOD_M1))
     {
      //--- Update for next upcoming news
      NewsObject.EconomicNextEvent();
      //--- Get impact value for upcoming news
      Impact=NewsObject.GetImpact();
     }
//--- Check if upcoming news date is about to occur and if it is the trading day of week
   if(CTM.TimePreEvent(CTM.TimeMinusOffset(datetime(UpcomingNews.EventDate),(iSecondsPreEvent==0)?1:iSecondsPreEvent)
                       ,datetime(UpcomingNews.EventDate))
      &&CTM.isDayOfTheWeek(TradingDay))
     {
      //--- Check each Impact value type
      switch(Impact)
        {
         //--- When Impact news is negative
         case CALENDAR_IMPACT_NEGATIVE:
            //--- Check if profit currency is news event currency
            if(UpcomingNews.EventCurrency==CSymbol.CurrencyProfit())
              {
               //--- Open buy trade with Event id as Magic number
               Trade.Buy(iStoploss,iTakeprofit,ulong(UpcomingNews.EventId),"NewsTrading");
              }
            else
              {
               //--- Open sell trade with Event id as Magic number
               Trade.Sell(iStoploss,iTakeprofit,ulong(UpcomingNews.EventId),"NewsTrading");
              }
            break;
         //--- When Impact news is positive
         case CALENDAR_IMPACT_POSITIVE:
            //--- Check if profit currency is news event currency
            if(UpcomingNews.EventCurrency==CSymbol.CurrencyProfit())
              {
               //--- Open sell trade with Event id as Magic number
               Trade.Sell(iStoploss,iTakeprofit,ulong(UpcomingNews.EventId),"NewsTrading");
              }
            else
              {
               //--- Open buy trade with Event id as Magic number
               Trade.Buy(iStoploss,iTakeprofit,ulong(UpcomingNews.EventId),"NewsTrading");
              }
            break;
         //--- Unknown
         default:
            break;
        }
     }
  }
//+------------------------------------------------------------------+
