//+------------------------------------------------------------------+
//|                                                   SignalCGAN.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <My\Cgan.mqh>
//+------------------------------------------------------------------+
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on Conditional GAN with Learning Rate Schedules. |
//| Type=SignalAdvanced                                              |
//| Name=CGAN                                                        |
//| ShortName=CGAN                                                   |
//| Class=CSignalCGAN                                                |
//| Page=signal_cgan                                                 |
//| Parameter=TrainSet,int,30,Training Set Size                      |
//| Parameter=Epochs,int,6,Epochs                                    |
//| Parameter=LearningRate,double,0.05,Learning Rate                 |
//| Parameter=DecayEpochSteps,int,10,Decay Epoch Steps               |
//| Parameter=DecayRate,double,0.1,Decay Rate                        |
//| Parameter=PolynomialPower,int,2,Polynomial Power                 |
//| Parameter=MinLearningRate,double,0.05,Minimum Learning Rate      |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalCGAN.                                               |
//| Purpose: Class of Conditional GAN with Learning Rate Schedules.  |
//|            Derives from class CExpertSignal.                     |
//+------------------------------------------------------------------+
enum ENUM_LEARNING
{  LEARNING_FIXED = 0,
   LEARNING_STEP_DECAY = 1,
   LEARNING_EXPONENTIAL_DECAY = 2,
   LEARNING_POLYNOMIAL_DECAY = 3,
   LEARNING_INVERSE_TIME_DECAY = 4,
   LEARNING_COSINE_ANNEALING = 5,
   LEARNING_CYCLICAL = 6,
   LEARNING_ONE_CYCLE = 7,
   LEARNING_ADAPTIVE = 8
};
enum ENUM_ADAPTIVE
{  ADAPTIVE_GRAD = 0,
   ADAPTIVE_RMS = 1,
   ADAPTIVE_ME = 2,
   ADAPTIVE_DELTA = 3,
};
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#define  __LAYERS 1
#define  __SIZES 8
#define  __GEN_INPUTS 5
#define  __GEN_OUTPUTS 1
int      __GEN_SETTINGS[2 + __LAYERS] //generator network settings

=
{  __GEN_INPUTS,  __SIZES,    __GEN_OUTPUTS
};
#define  __DIS_OUTPUTS 1
int      __DIS_SETTINGS[2 + __LAYERS] //discriminator settings

=
{  __GEN_INPUTS + __GEN_OUTPUTS,  __SIZES,   __DIS_OUTPUTS
};
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalCGAN   : public CExpertSignal
{
protected:

   int                           m_train_set;                       //
   int                           m_epochs;                              // Epochs
   double                        m_learning_rate;                       // Epsilon

   ENUM_LEARNING                 m_learning_type;

   double                        m_train_deposit;
   double                        m_train_criteria;

   int                           m_decay_epoch_steps;
   double                        m_decay_rate;
   int                           m_polynomial_power;
   double                        m_min_learning_rate;                       // Epsilon
   ENUM_ADAPTIVE                 m_adaptive_type;
   
   double                        m_prior_learning_rate;

public:
                     CSignalCGAN(void);
                    ~CSignalCGAN(void);

   //--- methods of setting adjustable parameters
   void                          TrainSet(int value)
   {  m_train_set = value;
   }
   void                          Epochs(int value)
   {  m_epochs = value;
   }
   void                          LearningRate(double value)
   {  m_learning_rate = value;
   }
   void                          LearningType(ENUM_LEARNING value)
   {  m_learning_type = value;
   }
   void                          DecayEpochSteps(int value)
   {  m_decay_epoch_steps = value;
   }
   void                          DecayRate(double value)
   {  m_decay_rate = value;
   }
   void                          PolynomialPower(int value)
   {  m_polynomial_power = value;
   }
   void                          MinLearningRate(double value)
   {  m_min_learning_rate = value;
   }
   void                          AdaptiveType(ENUM_ADAPTIVE value)
   {  m_adaptive_type = value;
   }

   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   void                          GetOutput(double &GenOut, bool &DisOut);
   void                          SetOutput();

   void                          F(vector &IN, vector &OUT, double LearningRate);
   void                          R(vector &IN, vector &OUT, double LearningRate);

   Cgan                          *GEN;
   Cgan                          *DIS;
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CSignalCGAN::CSignalCGAN(void) :    m_train_set(390),
   m_epochs(30),
   m_learning_rate(0.05),
   m_learning_type(LEARNING_STEP_DECAY),
   m_decay_epoch_steps(10),
   m_decay_rate(0.1),
   m_polynomial_power(2),
   m_min_learning_rate(0.00005),
   m_adaptive_type(ADAPTIVE_GRAD)

{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_SPREAD + USE_SERIES_TIME;
   m_prior_learning_rate = m_learning_rate;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CSignalCGAN::~CSignalCGAN(void)
{  delete GEN;
   delete DIS;
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalCGAN::ValidationSettings(void)
{  if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_epochs < 1)
   {  printf(__FUNCSIG__ + " epochs size should at least be 1! ");
      return(false);
   }
   if(m_learning_rate < 0.0)
   {  printf(__FUNCSIG__ + " learning rate should be greater than 0.0! ");
      return(false);
   }
   GEN = new Cgan(__GEN_SETTINGS, 0.1, 0.01);
   DIS = new Cgan(__DIS_SETTINGS, 0.1, 0.01);
   m_train_deposit = AccountInfoDouble(ACCOUNT_BALANCE);
   m_train_criteria = 0.0;
   m_prior_learning_rate = m_learning_rate;
//read best weights
//datetime _version;
//N.Get(m_symbol.Name()+"_"+EnumToString(m_period), m_train_criteria, _version);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalCGAN::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalCGAN::LongCondition(void)
{  int result = 0;
   double _gen_out = 0.0;
   bool _dis_out = false;
   GetOutput(_gen_out, _dis_out);
   _gen_out *= 100.0;
   if(_dis_out && _gen_out > 50.0)
   {  result = int(_gen_out);
   }
   //printf(__FUNCSIG__ + " generator output is: %.5f, which is backed by discriminator as: %s", _gen_out, string(_dis_out));return(0);
   return(result);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalCGAN::ShortCondition(void)
{  int result = 0;
   double _gen_out = 0.0;
   bool _dis_out = false;
   GetOutput(_gen_out, _dis_out);
   _gen_out *= 100.0;
   if(_dis_out && _gen_out < 50.0)
   {  result = int(fabs(_gen_out));
   }
   //printf(__FUNCSIG__ + " generator output is: %.5f, which is backed by discriminator as: %s", _gen_out, string(_dis_out));return(0);
   return(result);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSignalCGAN::GetOutput(double &GenOut, bool &DisOut)
{  GenOut = 0.0;
   DisOut = false;
   double _learning_rate = m_learning_rate;
   for(int i = m_epochs; i >= 0; i--)
   {  if(m_learning_type == LEARNING_STEP_DECAY)
      {  int _epoch_index = int(MathFloor((m_epochs - i) / m_decay_epoch_steps));
         _learning_rate = m_learning_rate * pow(m_decay_rate, _epoch_index);
      }
      else if(m_learning_type == LEARNING_EXPONENTIAL_DECAY)
      {  _learning_rate = m_learning_rate * exp(-1.0 * m_decay_rate * (m_epochs - i + 1));
      }
      else if(m_learning_type == LEARNING_POLYNOMIAL_DECAY)
      {  _learning_rate = m_learning_rate * pow(1.0 - ((m_epochs - i) / m_epochs), m_polynomial_power);
      }
      else if(m_learning_type == LEARNING_INVERSE_TIME_DECAY)
      {  _learning_rate = m_prior_learning_rate / (1.0 + (m_decay_rate * (m_epochs - i)));
         m_prior_learning_rate = _learning_rate;
      }
      else if(m_learning_type == LEARNING_COSINE_ANNEALING)
      {  _learning_rate = m_min_learning_rate + (0.5 * (m_learning_rate - m_min_learning_rate) * (1.0 + MathCos(((m_epochs - i) * M_PI) / m_epochs)));
      }
      else if(m_learning_type == LEARNING_CYCLICAL)
      {  double _x = fabs(((2.0 * fmod(m_epochs - i, m_epochs))/m_epochs) - 1.0);
         _learning_rate = m_min_learning_rate + ((m_learning_rate - m_min_learning_rate) * fmax(0.0, (1.0 - _x)));
      }
      else if(m_learning_type == LEARNING_ADAPTIVE)
      {  if(m_adaptive_type == ADAPTIVE_GRAD)
         {
         }
         else if(m_adaptive_type == ADAPTIVE_RMS)
         {
         }
         else if(m_adaptive_type == ADAPTIVE_ME)
         {
         }
         else if(m_adaptive_type == ADAPTIVE_DELTA)
         {
         }
      }
      for(int ii = m_train_set; ii >= 0; ii--)
      {  vector _in, _out;
         vector _in_new, _out_new, _in_old, _out_old;
         _in_new.CopyRates(m_symbol.Name(), m_period, 8, ii + 1, __GEN_INPUTS);
         _in_old.CopyRates(m_symbol.Name(), m_period, 8, ii + 1 + 1, __GEN_INPUTS);
         _in = Norm(_in_new, _in_old);
         GEN.Set(_in);
         GEN.Forward();
         if(ii > 0)// train
         {  _out_new.CopyRates(m_symbol.Name(), m_period, 8, ii, __GEN_OUTPUTS);
            _out_old.CopyRates(m_symbol.Name(), m_period, 8, ii + 1, __GEN_OUTPUTS);
            _out = Norm(_out_new, _out_old);
            //
            int _dis_sort = ((_in_new[__GEN_INPUTS - 1] - _in_new[__GEN_INPUTS - 2] > 0.0 && _in_new[__GEN_INPUTS - 1] - _in_new[0] > 0.0)
                             ||
                             (_in_new[__GEN_INPUTS - 1] - _in_new[__GEN_INPUTS - 2] < 0.0 && _in_new[__GEN_INPUTS - 1] - _in_new[0] < 0.0) ? 0 : 1);
            if(_dis_sort == 0)
            {  F(_in, GEN.output, _learning_rate);
               GEN.Get(_out);
               GEN.Backward(DIS.output, _learning_rate);
               R(_in, _out, _learning_rate);
            }
            else if(_dis_sort == 1)
            {  R(_in, _out, _learning_rate);
               GEN.Get(_out);
               GEN.Backward(DIS.output, _learning_rate);
               F(_in, GEN.output, _learning_rate);
            }
         }
         else if(ii == 0 && i == 0)
         {  GenOut = GEN.output[0];
            DisOut = (((DIS.output[0] >= 0.5 && GenOut >= 0.5) || (DIS.output[0] < 0.5 && GenOut < 0.5)) ? true : false);
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Process Fake Data in Discriminator                               |
//+------------------------------------------------------------------+
void CSignalCGAN::F(vector &IN, vector &OUT, double LearningRate)
{  vector _out_f, _out_fake, _in_fake;
   _out_f.Copy(OUT);
   _in_fake.Copy(IN);
   Sum(_in_fake, _out_f);
   DIS.Set(_in_fake);
   DIS.Forward();
   _out_fake.Resize(__DIS_OUTPUTS);
   _out_fake.Fill(0.0);
   DIS.Get(_out_fake);
   DIS.Backward(LearningRate);
}
//+------------------------------------------------------------------+
//| Process Real Data in Discriminator                               |
//+------------------------------------------------------------------+
void CSignalCGAN::R(vector &IN, vector &OUT, double LearningRate)
{  vector _out_r, _out_real, _in_real;
   _out_r.Copy(OUT);
   _in_real.Copy(IN);
   Sum(_in_real, _out_r);
   DIS.Set(_in_real);
   DIS.Forward();
   _out_real.Resize(__DIS_OUTPUTS);
   _out_real.Fill(1.0);
   DIS.Get(_out_real);
   DIS.Backward(LearningRate);
}
//+------------------------------------------------------------------+
//| Weights & Biases exporting function for network                  |
//+------------------------------------------------------------------+
void CSignalCGAN::SetOutput(void)
{  if(AccountInfoDouble(ACCOUNT_EQUITY) - m_train_deposit > m_train_criteria && m_train_criteria >= 0.0)
   {  GEN.Set(m_symbol.Name() + "_" + EnumToString(m_period), AccountInfoDouble(ACCOUNT_EQUITY) - m_train_deposit);
      DIS.Set(m_symbol.Name() + "_" + EnumToString(m_period), AccountInfoDouble(ACCOUNT_EQUITY) - m_train_deposit);
   }
}
//+------------------------------------------------------------------+
//| Normalization                                                    |
//+------------------------------------------------------------------+
vector Norm(vector &A, vector &B)
{  vector _n;
   _n.Init(A.Size());
   if(A.Size() > 0 && B.Size() > 0 && A.Size() == B.Size() && A.Min() > 0.0 && B.Min() > 0.0)
   {  int _size = int(A.Size());
      _n.Fill(0.5);
      for(int i = 0; i < _size; i++)
      {  if(A[i] > B[i])
         {  _n[i] += (0.5 * ((A[i] - B[i]) / A[i]));
         }
         else if(A[i] < B[i])
         {  _n[i] -= (0.5 * ((B[i] - A[i]) / B[i]));
         }
      }
   }
   return(_n);
}
//+------------------------------------------------------------------+
//| Add Vector B to A                                                |
//+------------------------------------------------------------------+
void Sum(vector &A, vector &B)
{  int _a_size = int(A.Size());
   A.Resize(_a_size + B.Size());
   for(int i = _a_size; i < int(_a_size + B.Size()); i++)
   {  A[i] = B[i - _a_size];
   }
}
//+------------------------------------------------------------------+
