//+------------------------------------------------------------------+
//|                                         SmoothedCandlesticks.mq5 |
//|                          Copyright 2024, Wanateki Solutions Ltd. |
//|                                         https://www.wanateki.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, Wanateki Solutions Ltd."
#property link      "https://www.wanateki.com"
#property version   "1.00"
#property description "Created for an article guide demonstrating how to"
#property description "create a Smoothed Candlesticks custom indicator."
#property description ""
#property description "This indicator eliminates market noise and makes"
#property description "it easy to quickly detect the trend direction."

//--- specify where to display the indicator
#property indicator_separate_window
//#property indicator_chart_window

//--- indicator buffers
#property indicator_buffers 6

//--- indicator plots
#property indicator_plots   2

//--- plots1 details for the smoothed candles
#property indicator_type1   DRAW_COLOR_CANDLES
#property indicator_color1  clrDodgerBlue, clrTomato, clrDarkGray
#property indicator_label1  "Smoothed Candle Open;Smoothed Candle High;Smoothed Candle Low;Smoothed Candle Close;"

//--- plots2 details for the smoothing line
#property indicator_label2  "Smoothing Line"
#property indicator_type2   DRAW_LINE
#property indicator_color2  clrGoldenrod
#property indicator_style2  STYLE_SOLID
#property indicator_width2  2

//--- user input parameters for the moving averages
input int                _maPeriod = 50;                    // Period
input ENUM_APPLIED_PRICE _maAppliedPrice = PRICE_CLOSE;     // Applied Price

//--- indicator buffers
double openBuffer[];
double highBuffer[];
double lowBuffer[];
double closeBuffer[];
double candleColorBuffer[];

//Moving average dynamic array (buffer) and variables
double iMA_Buffer[];
int maHandle; //stores the handle of the iMA indicator

//--- integer to store the number of values in the moving average indicator
int barsCalculated = 0;

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- call the custom initialization function
   if(!GetInit())
     {
      return(INIT_FAILED); //-- if initialization failed terminate the app
     }

//---
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
  {
//--- declare a int to save the number of values copied from the iMA indicator
   int iMA_valuesToCopy;

//--- find the number of values already calculated in the indicator
   int iMA_calculated = BarsCalculated(maHandle);
   if(iMA_calculated <= 0)
     {
      PrintFormat("BarsCalculated() for iMA handle returned %d, error code %d", iMA_calculated, GetLastError());
      return(0);
     }

   int start;
//--- check if it's the indicators first call of OnCalculate() or we have some new uncalculated data
   if(prev_calculated == 0)
     {
      //--- set all the buffers to the first index
      lowBuffer[0] = low[0];
      highBuffer[0] = high[0];
      openBuffer[0] = open[0];
      closeBuffer[0] = close[0];
      start = 1;


      if(iMA_calculated > rates_total)
         iMA_valuesToCopy = rates_total;
      else   //--- copy the calculated bars which are less than the indicator buffers data
         iMA_valuesToCopy = iMA_calculated;
     }
   else
      start = prev_calculated - 1;

   iMA_valuesToCopy = (rates_total - prev_calculated) + 1;

//--- fill the iMA_Buffer array with values of the Moving Average indicator
//--- reset error code
   ResetLastError();
//--- copy a part of iMA_Buffer array with data in the zero index of the the indicator buffer
   if(CopyBuffer(maHandle, 0, 0, iMA_valuesToCopy, iMA_Buffer) < 0)
     {
      //--- if the copying fails, print the error code
      PrintFormat("Failed to copy data from the iMA indicator, error code %d", GetLastError());
      //--- exit the function with zero result to specify that the indicator calculations were not executed
      return(0);
     }

//--- iterate through the main calculations loop and execute all the calculations
   for(int x = start; x < rates_total && !IsStopped(); x++)
     {
      //--- save all the candle array prices in new non-array variables for quick access
      double candleOpen = open[x];
      double candleClose = close[x];
      double candleHigh = high[x];
      double candleLow  = low[x];

      lowBuffer[x] = candleLow;
      highBuffer[x] = candleHigh;
      openBuffer[x] = candleOpen;
      closeBuffer[x] = candleClose;

      //--- scan for the different trends signals and set the required candle color
      candleColorBuffer[x] = 2.0; // set color clrDarkGray - default (signal for no established trend)
      if(candleOpen > iMA_Buffer[x] && candleClose > iMA_Buffer[x] && candleHigh > iMA_Buffer[x] && candleLow > iMA_Buffer[x])
         candleColorBuffer[x]=0.0; // set color clrDodgerBlue - signal for a long/buy trend

      if(candleOpen < iMA_Buffer[x] && candleClose < iMA_Buffer[x] && candleHigh < iMA_Buffer[x] && candleLow < iMA_Buffer[x])
         candleColorBuffer[x]=1.0; // set color clrTomato - signal for a short/sell trend
     }

//--- return the rates_total which includes the prev_calculated value for the next call
   return(rates_total);
  }

//+------------------------------------------------------------------+
//| Indicator deinitialization function                              |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   if(maHandle != INVALID_HANDLE)
     {
      IndicatorRelease(maHandle);//-- clean up and release the iMA handle
     }
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| User custom function for custom indicator initialization         |
//+------------------------------------------------------------------+
bool GetInit()
  {
//--- set the indicator buffer mapping by assigning the indicator buffer array
   SetIndexBuffer(0, openBuffer, INDICATOR_DATA);
   SetIndexBuffer(1, highBuffer, INDICATOR_DATA);
   SetIndexBuffer(2, lowBuffer, INDICATOR_DATA);
   SetIndexBuffer(3, closeBuffer, INDICATOR_DATA);
   SetIndexBuffer(4, candleColorBuffer, INDICATOR_COLOR_INDEX);

//--- buffer for iMA
   SetIndexBuffer(5, iMA_Buffer, INDICATOR_DATA);

//--- set the price display precision to digits similar to the symbol prices
   IndicatorSetInteger(INDICATOR_DIGITS, _Digits);

//--- set the symbol, timeframe, period and smoothing applied price of the indicator as the short name
   string indicatorShortName = StringFormat("SmoothedCandles(%s, Period %d, %s)", _Symbol,
                               _maPeriod, EnumToString(_maAppliedPrice));
   IndicatorSetString(INDICATOR_SHORTNAME, indicatorShortName);
//IndicatorSetString(INDICATOR_SHORTNAME, "Smoothed Candlesticks");

//--- set line drawing to an empty value
   PlotIndexSetDouble(0, PLOT_EMPTY_VALUE, 0.0);

//--- create the maHandle of the smoothing indicator
   maHandle = iMA(_Symbol, PERIOD_CURRENT, _maPeriod, 0, MODE_SMMA, _maAppliedPrice);

//--- check if the maHandle is created or it failed
   if(maHandle == INVALID_HANDLE)
     {
      //--- creating the handle failed, output the error code
      ResetLastError();
      PrintFormat("Failed to create maHandle of the iMA for symbol %s, error code %d",
                  _Symbol, GetLastError());
      //--- we terminate the program and exit the init function
      return(false);
     }

   return(true); // return true, initialization of the indicator ok
  }
//+------------------------------------------------------------------+
