//+------------------------------------------------------------------+
//|                                                      Returns.mqh |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#include<Arrays\ArrayDouble.mqh>
#include<Files\FileBin.mqh>


//+------------------------------------------------------------------+
//| Enumeration specifying granularity of return                     |
//+------------------------------------------------------------------+
enum ENUM_RETURNS_TYPE
  {
   ENUM_RETURNS_ALL_BARS=0,//bar-by-bar returns for all bars
   ENUM_RETURNS_POSITION_OPEN_BARS,//bar-by-bar returns for bars with open trades
   ENUM_RETURNS_TRADES//trade returns
  };
//+------------------------------------------------------------------+
//| Class for calculating Sharpe Ratio in the tester                 |
//+------------------------------------------------------------------+
class CReturns
  {
private:
   CArrayDouble*     m_all_bars_equity;
   CArrayDouble*     m_open_position_bars_equity;
   CArrayDouble*     m_trade_equity;
   string            m_error_string;
   CArrayDouble*     m_all_bars_returns;
   CArrayDouble*     m_open_position_bars_returns;
   CArrayDouble*     m_trade_returns;

   int               ProcessHistory(void);
   void              CalculateReturns(CArrayDouble &r,CArrayDouble &e);

public:
                     CReturns(void);
                    ~CReturns(void);

   void              OnNewTick(void);

   bool              GetEquityCurve(const ENUM_RETURNS_TYPE return_type,double &out_equity[]);
   bool              GetReturns(const ENUM_RETURNS_TYPE return_type,double &out_returns[]);
  };
//+------------------------------------------------------------------+
//|  constructor                                                     |
//+------------------------------------------------------------------+
CReturns::CReturns(void)
  {
   m_error_string = "";
//---
   m_all_bars_equity=new CArrayDouble();
//---
   m_open_position_bars_equity=new CArrayDouble();
//---
   m_trade_equity=new CArrayDouble();
//---
   m_all_bars_returns=new CArrayDouble();
//---
   m_open_position_bars_returns=new CArrayDouble();
//---
   m_trade_returns=new CArrayDouble();
  }
//+------------------------------------------------------------------+
//|destructor                                                        |
//+------------------------------------------------------------------+
CReturns::~CReturns(void)
  {
   delete m_all_bars_equity;
//---
   delete m_open_position_bars_equity;
//---
   delete m_trade_equity;
//---
   delete m_all_bars_returns;
//---
   delete m_open_position_bars_returns;
//---
   delete m_trade_returns;
  }
//+------------------------------------------------------------------+
//|Adds equity value on every tick enumerating bars accordingly      |
//+------------------------------------------------------------------+


void CReturns::OnNewTick(void)
  {
//---
   static datetime last_time = 0;
//---
   datetime open = iTime(_Symbol, PERIOD_CURRENT, 0);
//---
   if(open > last_time)
     {
      //---
      if(last_time==0)
        {
         //---
         m_all_bars_equity.Clear();
         m_open_position_bars_equity.Clear();
         //---
        }
      //---
      last_time = open;
      //---
      m_all_bars_equity.Add(AccountInfoDouble(ACCOUNT_EQUITY));
      //---
      int openorders=0;
      //---
#ifdef __MQL5__
      openorders=PositionsTotal();
#else
      for(int k=OrdersTotal(); k>=0; k--)
        {
         if(!OrderSelect(k, SELECT_BY_POS, MODE_TRADES))
            continue;
         //---
         if(OrderType()>1)
            continue;
         //---
         openorders++;
        }
#endif
      //---
      if(openorders)
         m_open_position_bars_equity.Add((last_time)?AccountInfoDouble(ACCOUNT_EQUITY):AccountInfoDouble(ACCOUNT_BALANCE));
     }
  }
//+------------------------------------------------------------------+
//| Process history trades and fills trades_equity array             |
//+------------------------------------------------------------------+
int CReturns::ProcessHistory(void)
  {
   int total_trades=0;
   double equity = TesterStatistics(STAT_INITIAL_DEPOSIT);
   m_trade_equity.Clear();
//---
#ifdef __MQL5__
   if(!HistorySelect(0,TimeCurrent()))
     {
      Print("Error selecting history deals ", GetLastError());
      return 0;
     }
//---
   total_trades=HistoryDealsTotal();
//---
   ulong ticket;
//---
   if(!m_trade_equity.Add(equity))
      return 0;
//---
   for(int i=0; i<total_trades; i++)
     {
      ticket=HistoryDealGetTicket(i);
      //---
      if(ticket<=0)
         continue;
      //---
      if(HistoryDealGetInteger(ticket,DEAL_ENTRY)!=DEAL_ENTRY_OUT)
         continue;
      //---
      equity+=HistoryDealGetDouble(ticket,DEAL_PROFIT)+HistoryDealGetDouble(ticket,DEAL_SWAP)+HistoryDealGetDouble(ticket,DEAL_COMMISSION);
      //---
      if(!m_trade_equity.Add(equity))
         return 0;
      //---
     }
#else
   total_trades=OrdersHistoryTotal();
//---
   if(!m_trade_equity.Add(equity))
      return 0;
//---
   for(int i=0; i<total_trades; i++)
     {
      if(!OrderSelect(i, SELECT_BY_POS, MODE_HISTORY))
         continue;
      //---
      equity+=OrderProfit()+OrderCommission()+OrderSwap();
      //---
      if(!m_trade_equity.Add(equity))
         return 0;
      //---
     }
#endif
//---
   return m_trade_equity.Total();
  }
//+------------------------------------------------------------------+
//| get equity curve                                                 |
//+------------------------------------------------------------------+
bool CReturns::GetEquityCurve(const ENUM_RETURNS_TYPE return_type,double &out_equity[])
  {
   int m_counter=0;
   CArrayDouble *equity;
   ZeroMemory(out_equity);
//---
   switch(return_type)
     {
      case ENUM_RETURNS_ALL_BARS:
         m_counter=m_all_bars_equity.Total();
         equity=m_all_bars_equity;
         break;
      case ENUM_RETURNS_POSITION_OPEN_BARS:
         m_counter=m_open_position_bars_equity.Total();
         equity=m_open_position_bars_equity;
         break;
      case ENUM_RETURNS_TRADES:
         m_counter=(m_trade_equity.Total()>1)?m_trade_equity.Total():ProcessHistory();
         equity=m_trade_equity;
         break;
      default:
         return false;
     }
//--- if there are no bars, return 0
   if(m_counter < 2)
      return false;
//---
   if(ArraySize(out_equity)!=m_counter)
      if(ArrayResize(out_equity,equity.Total()) < m_counter)
         return false;
//---
   for(int i=0; i<equity.Total(); i++)
      out_equity[i]=equity[i];
//---
   return(true);
//---
  }
//+------------------------------------------------------------------+
//|calculate the returns                                             |
//+------------------------------------------------------------------+
void CReturns::CalculateReturns(CArrayDouble &r,CArrayDouble &e)
  {
   int m_counter=e.Total();
//--- if there are no bars, return 0
   if(m_counter < 2)
     {
      Print(__FUNCTION__+" Not enough equity data points for calculate returns");
      return ;
     }
//--- calculate average returns
   double log_return=0, prev_equity=0,curr_equity=0;
   r.Clear();
//--- calculate the logarithms of increments using the equity array
   for(int i = 1; i < m_counter; i++)
     {
      //---
      curr_equity=e[i];
      prev_equity=e[i-1];
      //---
      log_return = MathLog(curr_equity / prev_equity); // increment logarithm
      //---
      if(!r.Add(log_return))                           // fill the array of log returns
        {
         Print("error ",__FUNCTION__," ",__LINE__," ",GetLastError());
         return ;
        }
      //---
     }
//---
   return;
  }
//+------------------------------------------------------------------+
//|Gets the returns into array                                       |
//+------------------------------------------------------------------+
bool CReturns::GetReturns(const ENUM_RETURNS_TYPE return_type,double &out_returns[])
  {
//---
   CArrayDouble *returns,*equity;
   ZeroMemory(out_returns);
//---
   switch(return_type)
     {
      case ENUM_RETURNS_ALL_BARS:
         returns=m_all_bars_returns;
         equity=m_all_bars_equity;
         break;
      case ENUM_RETURNS_POSITION_OPEN_BARS:
         returns=m_open_position_bars_returns;
         equity=m_open_position_bars_equity;
         break;
      case ENUM_RETURNS_TRADES:
         if(m_trade_equity.Total()<2)
            ProcessHistory();
         returns=m_trade_returns;
         equity=m_trade_equity;
         break;
      default:
         return false;
     }
//--- if there are no bars, return 0
   if(equity.Total() < 2)
       return false;
//--- calculate average returns
   CalculateReturns(returns,equity);
//--- return the mean return
   if(returns.Total()<=0)
      return false;
//---
   if(ArraySize(out_returns)!=returns.Total()) {
      if(ArrayResize(out_returns,returns.Total()) < returns.Total())
          return false;
    }      
//---
   for(int i=0; i<returns.Total(); i++)
      out_returns[i]=returns[i];
//---
   return(true);
//---
  }
//+------------------------------------------------------------------+
