//+------------------------------------------------------------------+
//|                                  PropFirmDrawdownMonitor.mq5      |
//|                                  Copyright 2025, Lorenzo          |
//|                          Prop Firm Drawdown Monitor - Free        |
//+------------------------------------------------------------------+
#property copyright   "Lorenzo"
#property link        "https://www.mql5.com/en/market/product/165214"
#property version     "1.10"
#property description "Real-time drawdown monitoring script for prop firm accounts."
#property description "Tracks daily and total drawdown with auto-close on breach."
#property description "Supports static and trailing total drawdown modes."
#property description "Full version: Prop Firm Shield on MQL5 Market."
#property script_show_inputs

//--- Input parameters
input group "=== Account Settings ==="
input double InpAccountSize      = 100000;  // Account Size ($) - Your funded amount
input double InpDailyDDPct       = 5.0;     // Daily Drawdown Limit (%)
input double InpTotalDDPct       = 10.0;    // Total Drawdown Limit (%)
input bool   InpTrailingMode     = false;   // Trailing Total DD (true) or Static (false)

input group "=== Daily Reset Settings ==="
input int    InpResetHour        = 0;       // Daily Reset Hour (0-23)
input int    InpResetTZOffset    = 1;       // Reset Timezone UTC Offset (FTMO=+1/+2)

input group "=== Safety Settings ==="
input double InpBufferPct        = 0.5;     // Safety Buffer (%) - Close before breach
input double InpWarningPct       = 80.0;    // Warning at % of limit reached
input bool   InpAutoClose        = true;    // Auto-Close on Breach
input bool   InpBlockNewTrades   = true;    // Block New Trades After Breach

input group "=== Display ==="
input int    InpDashX            = 20;      // Dashboard X Position
input int    InpDashY            = 30;      // Dashboard Y Position
input color  InpColorSafe        = clrLime;       // Safe Color
input color  InpColorWarning     = clrOrange;     // Warning Color
input color  InpColorBreach      = clrRed;        // Breach Color

//--- Global tracking variables
double g_dailyRefBalance   = 0;
double g_totalRefBalance   = 0;
double g_highEquity        = 0;
double g_limDaily          = 0;
double g_limTotal          = 0;
datetime g_lastResetDate   = 0;
bool   g_breached          = false;
bool   g_dailyWarning      = false;
bool   g_totalWarning      = false;
int    g_lastPosCount      = 0;  // For trade-blocking via polling

//--- GlobalVariable keys for state persistence
string g_gvPrefix;
string g_gvDailyRef;
string g_gvHighEquity;
string g_gvLastReset;
string g_gvBreached;

//+------------------------------------------------------------------+
//| Script program start function                                     |
//+------------------------------------------------------------------+
void OnStart()
{
   //--- Initialize
   if(!Initialize())
      return;

   //--- Store initial position count for trade blocking
   g_lastPosCount = PositionsTotal();

   //--- Main monitoring loop (500ms cycle, same as EA timer)
   while(!IsStopped())
   {
      //--- Check for daily reset
      CheckDailyReset();

      //--- Get current equity
      double equity = AccountInfoDouble(ACCOUNT_EQUITY);

      //--- Update high equity watermark (for trailing mode)
      if(equity > g_highEquity)
         g_highEquity = equity;

      //--- Calculate daily drawdown
      double dailyLoss = g_dailyRefBalance - equity;
      double dailyPct  = (InpAccountSize > 0) ? (dailyLoss / InpAccountSize) * 100.0 : 0;
      if(dailyPct < 0) dailyPct = 0;

      //--- Calculate total drawdown
      double totalLoss, totalPct;
      if(InpTrailingMode)
      {
         totalLoss = g_highEquity - equity;
         totalPct  = (InpAccountSize > 0) ? (totalLoss / InpAccountSize) * 100.0 : 0;
      }
      else
      {
         totalLoss = g_totalRefBalance - equity;
         totalPct  = (InpAccountSize > 0) ? (totalLoss / InpAccountSize) * 100.0 : 0;
      }
      if(totalPct < 0) totalPct = 0;

      //--- Check for warnings
      bool dailyWarn = (dailyPct >= InpDailyDDPct * InpWarningPct / 100.0);
      bool totalWarn = (totalPct >= InpTotalDDPct * InpWarningPct / 100.0);

      if(dailyWarn && !g_dailyWarning)
      {
         PrintFormat("[PFDM] WARNING: Daily DD at %.2f%% ($%.2f) - limit %.1f%%", dailyPct, dailyLoss, InpDailyDDPct);
         Alert("Prop Firm DD Monitor: Daily drawdown at ", DoubleToString(dailyPct, 2), "% - WARNING!");
         g_dailyWarning = true;
      }
      if(!dailyWarn) g_dailyWarning = false;

      if(totalWarn && !g_totalWarning)
      {
         PrintFormat("[PFDM] WARNING: Total DD at %.2f%% ($%.2f) - limit %.1f%%", totalPct, totalLoss, InpTotalDDPct);
         Alert("Prop Firm DD Monitor: Total drawdown at ", DoubleToString(totalPct, 2), "% - WARNING!");
         g_totalWarning = true;
      }
      if(!totalWarn) g_totalWarning = false;

      //--- Check for breach
      bool dailyBreach = (dailyPct >= (InpDailyDDPct - InpBufferPct));
      bool totalBreach = (totalPct >= (InpTotalDDPct - InpBufferPct));

      if((dailyBreach || totalBreach) && !g_breached)
      {
         g_breached = true;

         string reason = dailyBreach ? "DAILY" : "TOTAL";
         double breachPct = dailyBreach ? dailyPct : totalPct;

         PrintFormat("[PFDM] BREACH DETECTED: %s DD at %.2f%% - EMERGENCY CLOSE", reason, breachPct);
         Alert("PROP FIRM DD BREACH: ", reason, " drawdown at ", DoubleToString(breachPct, 2), "%!");

         if(InpAutoClose)
            CloseAllPositions();

         if(InpBlockNewTrades)
            GlobalVariableSet(g_gvPrefix + "TradeBlocked", 1.0);

         SaveState();
      }

      //--- Trade blocking via polling (replaces OnTradeTransaction)
      if(InpBlockNewTrades && g_breached)
         CheckAndBlockNewTrades();

      //--- Update dashboard
      UpdateDashboard(dailyPct, dailyLoss, totalPct, totalLoss, equity);

      //--- Persist state
      SaveState();

      //--- 500ms monitoring cycle
      Sleep(500);
   }

   //--- Cleanup on script stop
   Cleanup();
}

//+------------------------------------------------------------------+
//| Initialize all variables and dashboard                            |
//+------------------------------------------------------------------+
bool Initialize()
{
   //--- Build unique GV prefix using account + chart
   g_gvPrefix    = "PFDM_" + IntegerToString(AccountInfoInteger(ACCOUNT_LOGIN)) + "_";
   g_gvDailyRef  = g_gvPrefix + "DailyRef";
   g_gvHighEquity= g_gvPrefix + "HighEq";
   g_gvLastReset = g_gvPrefix + "LastReset";
   g_gvBreached  = g_gvPrefix + "Breached";

   //--- Calculate dollar limits
   g_limDaily = InpAccountSize * InpDailyDDPct / 100.0;
   g_limTotal = InpAccountSize * InpTotalDDPct / 100.0;

   //--- Total reference is always the account size (initial funded amount)
   g_totalRefBalance = InpAccountSize;

   //--- Try to restore persisted state
   if(!LoadState())
   {
      //--- Fresh start: set initial values
      g_dailyRefBalance = MathMax(AccountInfoDouble(ACCOUNT_BALANCE),
                                   AccountInfoDouble(ACCOUNT_EQUITY));
      g_highEquity      = g_dailyRefBalance;
      g_lastResetDate   = GetResetDate();
      g_breached        = false;
   }

   //--- Create dashboard
   CreateDashboard();

   PrintFormat("[PFDM] Initialized. Account: $%.0f | Daily: %.1f%% ($%.0f) | Total: %.1f%% ($%.0f) | Mode: %s",
               InpAccountSize, InpDailyDDPct, g_limDaily, InpTotalDDPct, g_limTotal,
               InpTrailingMode ? "Trailing" : "Static");
   PrintFormat("[PFDM] Daily Ref: $%.2f | High Equity: $%.2f | Breached: %s",
               g_dailyRefBalance, g_highEquity, g_breached ? "YES" : "NO");

   return true;
}

//+------------------------------------------------------------------+
//| Cleanup on script termination                                     |
//+------------------------------------------------------------------+
void Cleanup()
{
   SaveState();
   DeleteDashboard();
   Print("[PFDM] Script stopped. State saved.");
}

//+------------------------------------------------------------------+
//| Check for and block new trades opened during breach               |
//| Replaces OnTradeTransaction with polling approach                 |
//+------------------------------------------------------------------+
void CheckAndBlockNewTrades()
{
   int currentPosCount = PositionsTotal();

   //--- If position count increased, new trades were opened
   if(currentPosCount > g_lastPosCount)
   {
      PrintFormat("[PFDM] BLOCKED: %d new position(s) detected during breach. Closing...",
                  currentPosCount - g_lastPosCount);

      //--- Close all positions (any new ones will be included)
      CloseAllPositions();
   }

   //--- Update tracked count
   g_lastPosCount = PositionsTotal();
}

//+------------------------------------------------------------------+
//| Check for daily reset                                             |
//+------------------------------------------------------------------+
void CheckDailyReset()
{
   datetime today = GetResetDate();

   if(today > g_lastResetDate)
   {
      double oldRef = g_dailyRefBalance;
      g_dailyRefBalance = MathMax(AccountInfoDouble(ACCOUNT_BALANCE),
                                   AccountInfoDouble(ACCOUNT_EQUITY));
      g_lastResetDate = today;
      g_dailyWarning = false;

      PrintFormat("[PFDM] DAILY RESET: Old ref $%.2f -> New ref $%.2f", oldRef, g_dailyRefBalance);
   }
}

//+------------------------------------------------------------------+
//| Get current date in reset timezone                                |
//+------------------------------------------------------------------+
datetime GetResetDate()
{
   datetime serverTime = TimeCurrent();
   datetime gmtTime = TimeGMT();
   int serverOffset = (int)(serverTime - gmtTime);

   datetime resetTime = serverTime - serverOffset + InpResetTZOffset * 3600;

   MqlDateTime dt;
   TimeToStruct(resetTime, dt);

   if(dt.hour < InpResetHour)
   {
      resetTime -= 86400;
      TimeToStruct(resetTime, dt);
   }

   return StringToTime(StringFormat("%04d.%02d.%02d", dt.year, dt.mon, dt.day));
}

//+------------------------------------------------------------------+
//| Close all open positions and pending orders                       |
//+------------------------------------------------------------------+
void CloseAllPositions()
{
   int closed = 0;
   int failed = 0;

   for(int i = PositionsTotal() - 1; i >= 0; i--)
   {
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0) continue;

      if(ClosePosition(ticket))
         closed++;
      else
         failed++;
   }

   for(int i = OrdersTotal() - 1; i >= 0; i--)
   {
      ulong ticket = OrderGetTicket(i);
      if(ticket == 0) continue;

      MqlTradeRequest req = {};
      MqlTradeResult  res = {};
      req.action = TRADE_ACTION_REMOVE;
      req.order  = ticket;

      if(OrderSend(req, res))
         closed++;
      else
         failed++;
   }

   PrintFormat("[PFDM] Emergency close complete: %d closed, %d failed", closed, failed);
}

//+------------------------------------------------------------------+
//| Close a single position by ticket                                 |
//+------------------------------------------------------------------+
bool ClosePosition(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return false;

   MqlTradeRequest req = {};
   MqlTradeResult  res = {};

   req.action   = TRADE_ACTION_DEAL;
   req.position = ticket;
   req.symbol   = PositionGetString(POSITION_SYMBOL);
   req.volume   = PositionGetDouble(POSITION_VOLUME);
   req.deviation= 50;

   long type = PositionGetInteger(POSITION_TYPE);
   if(type == POSITION_TYPE_BUY)
   {
      req.type  = ORDER_TYPE_SELL;
      req.price = SymbolInfoDouble(req.symbol, SYMBOL_BID);
   }
   else
   {
      req.type  = ORDER_TYPE_BUY;
      req.price = SymbolInfoDouble(req.symbol, SYMBOL_ASK);
   }

   //--- Determine fill mode
   long fillMode = SymbolInfoInteger(req.symbol, SYMBOL_FILLING_MODE);
   if((fillMode & SYMBOL_FILLING_FOK) != 0)
      req.type_filling = ORDER_FILLING_FOK;
   else if((fillMode & SYMBOL_FILLING_IOC) != 0)
      req.type_filling = ORDER_FILLING_IOC;
   else
      req.type_filling = ORDER_FILLING_RETURN;

   if(!OrderSend(req, res))
   {
      PrintFormat("[PFDM] Failed to close position %d: error %d", ticket, GetLastError());
      return false;
   }

   return true;
}

//+------------------------------------------------------------------+
//| Save state to GlobalVariables                                     |
//+------------------------------------------------------------------+
void SaveState()
{
   GlobalVariableSet(g_gvDailyRef,   g_dailyRefBalance);
   GlobalVariableSet(g_gvHighEquity, g_highEquity);
   GlobalVariableSet(g_gvLastReset,  (double)g_lastResetDate);
   GlobalVariableSet(g_gvBreached,   g_breached ? 1.0 : 0.0);
}

//+------------------------------------------------------------------+
//| Load state from GlobalVariables                                   |
//+------------------------------------------------------------------+
bool LoadState()
{
   if(!GlobalVariableCheck(g_gvDailyRef)) return false;

   g_dailyRefBalance = GlobalVariableGet(g_gvDailyRef);

   if(GlobalVariableCheck(g_gvHighEquity))
      g_highEquity = GlobalVariableGet(g_gvHighEquity);
   else
      g_highEquity = g_dailyRefBalance;

   if(GlobalVariableCheck(g_gvLastReset))
      g_lastResetDate = (datetime)GlobalVariableGet(g_gvLastReset);

   if(GlobalVariableCheck(g_gvBreached))
      g_breached = (GlobalVariableGet(g_gvBreached) > 0);

   PrintFormat("[PFDM] State restored: DailyRef=$%.2f | HighEq=$%.2f | Breached=%s",
               g_dailyRefBalance, g_highEquity, g_breached ? "YES" : "NO");
   return true;
}

//+------------------------------------------------------------------+
//| Dashboard - Create objects                                        |
//+------------------------------------------------------------------+
void CreateDashboard()
{
   CreateLabel("PFDM_BG",     InpDashX, InpDashY, "");
   CreateLabel("PFDM_Title",  InpDashX + 10, InpDashY + 5, "PROP FIRM DD MONITOR");
   CreateLabel("PFDM_Sep1",   InpDashX + 10, InpDashY + 25, "--------------------------------");
   CreateLabel("PFDM_Daily",  InpDashX + 10, InpDashY + 42, "Daily DD:  ---");
   CreateLabel("PFDM_Total",  InpDashX + 10, InpDashY + 60, "Total DD:  ---");
   CreateLabel("PFDM_Sep2",   InpDashX + 10, InpDashY + 78, "--------------------------------");
   CreateLabel("PFDM_Equity", InpDashX + 10, InpDashY + 95, "Equity:    ---");
   CreateLabel("PFDM_Status", InpDashX + 10, InpDashY + 113, "Status:    INITIALIZING");
   CreateLabel("PFDM_Mode",   InpDashX + 10, InpDashY + 131, "Mode:      ---");
   CreateLabel("PFDM_Reset",  InpDashX + 10, InpDashY + 149, "Reset TZ:  ---");

   ObjectSetInteger(0, "PFDM_Title", OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, "PFDM_Sep1",  OBJPROP_COLOR, clrGray);
   ObjectSetInteger(0, "PFDM_Sep2",  OBJPROP_COLOR, clrGray);
   ObjectSetInteger(0, "PFDM_Mode",  OBJPROP_COLOR, clrGray);
   ObjectSetInteger(0, "PFDM_Reset", OBJPROP_COLOR, clrGray);

   ObjectSetString(0, "PFDM_Mode", OBJPROP_TEXT,
      "Mode:      " + (InpTrailingMode ? "TRAILING" : "STATIC"));
   ObjectSetString(0, "PFDM_Reset", OBJPROP_TEXT,
      "Reset TZ:  UTC" + (InpResetTZOffset >= 0 ? "+" : "") + IntegerToString(InpResetTZOffset) +
      " @" + IntegerToString(InpResetHour) + ":00");

   ChartRedraw();
}

//+------------------------------------------------------------------+
//| Dashboard - Update values                                         |
//+------------------------------------------------------------------+
void UpdateDashboard(double dailyPct, double dailyLoss, double totalPct, double totalLoss, double equity)
{
   color dailyColor = InpColorSafe;
   if(dailyPct >= InpDailyDDPct * InpWarningPct / 100.0) dailyColor = InpColorWarning;
   if(dailyPct >= InpDailyDDPct - InpBufferPct)           dailyColor = InpColorBreach;

   string dailyText = StringFormat("Daily DD:  %.2f%% / %.1f%%  ($%.0f / $%.0f)",
                                    dailyPct, InpDailyDDPct, dailyLoss, g_limDaily);
   ObjectSetString(0, "PFDM_Daily", OBJPROP_TEXT, dailyText);
   ObjectSetInteger(0, "PFDM_Daily", OBJPROP_COLOR, dailyColor);

   color totalColor = InpColorSafe;
   if(totalPct >= InpTotalDDPct * InpWarningPct / 100.0) totalColor = InpColorWarning;
   if(totalPct >= InpTotalDDPct - InpBufferPct)           totalColor = InpColorBreach;

   string totalText = StringFormat("Total DD:  %.2f%% / %.1f%%  ($%.0f / $%.0f)",
                                    totalPct, InpTotalDDPct, totalLoss, g_limTotal);
   ObjectSetString(0, "PFDM_Total", OBJPROP_TEXT, totalText);
   ObjectSetInteger(0, "PFDM_Total", OBJPROP_COLOR, totalColor);

   ObjectSetString(0, "PFDM_Equity", OBJPROP_TEXT,
      StringFormat("Equity:    $%.2f", equity));
   ObjectSetInteger(0, "PFDM_Equity", OBJPROP_COLOR, clrWhite);

   string status;
   color  statusColor;
   if(g_breached)
   {
      status = "BREACH - TRADING BLOCKED";
      statusColor = InpColorBreach;
   }
   else if(dailyPct >= InpDailyDDPct * InpWarningPct / 100.0 || 
           totalPct >= InpTotalDDPct * InpWarningPct / 100.0)
   {
      status = "WARNING - APPROACHING LIMIT";
      statusColor = InpColorWarning;
   }
   else
   {
      status = "SAFE";
      statusColor = InpColorSafe;
   }

   ObjectSetString(0, "PFDM_Status", OBJPROP_TEXT, "Status:    " + status);
   ObjectSetInteger(0, "PFDM_Status", OBJPROP_COLOR, statusColor);

   ChartRedraw();
}

//+------------------------------------------------------------------+
//| Dashboard - Delete all objects                                    |
//+------------------------------------------------------------------+
void DeleteDashboard()
{
   ObjectDelete(0, "PFDM_BG");
   ObjectDelete(0, "PFDM_Title");
   ObjectDelete(0, "PFDM_Sep1");
   ObjectDelete(0, "PFDM_Daily");
   ObjectDelete(0, "PFDM_Total");
   ObjectDelete(0, "PFDM_Sep2");
   ObjectDelete(0, "PFDM_Equity");
   ObjectDelete(0, "PFDM_Status");
   ObjectDelete(0, "PFDM_Mode");
   ObjectDelete(0, "PFDM_Reset");
   ChartRedraw();
}

//+------------------------------------------------------------------+
//| Helper - Create chart label                                       |
//+------------------------------------------------------------------+
void CreateLabel(string name, int x, int y, string text)
{
   ObjectCreate(0, name, OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, name, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(0, name, OBJPROP_YDISTANCE, y);
   ObjectSetString(0, name, OBJPROP_TEXT, text);
   ObjectSetString(0, name, OBJPROP_FONT, "Consolas");
   ObjectSetInteger(0, name, OBJPROP_FONTSIZE, 10);
   ObjectSetInteger(0, name, OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, name, OBJPROP_SELECTABLE, false);
}
//+------------------------------------------------------------------+
