//+------------------------------------------------------------------------------------------------------+
//|  MACD-v for MT4                                                                                      |
//|  Complete version with auto-detect subwindow and levels                                              |
//|  https://www.tradingview.com/script/mMMHCZUx-AS-MACD-v-Hist-Alex-Spiroglou-S-M-A-R-T-TRADER-SYSTEMS/ |
//+------------------------------------------------------------------------------------------------------+
#property copyright ""
#property link      ""
#property version   "1.00"
#property strict
#property indicator_separate_window
#property indicator_buffers 3

//--- indicator buffers
double macdvBuffer[];
double signalBuffer[];
double histBuffer[];

//--- user inputs
extern int    FastLen       = 12;
extern int    SlowLen       = 26;
extern int    SignalLen     = 9;
extern int    ATRPeriod     = 26;     // ATR denominator used in Pine script
extern int    MaxBarsBack   = 5000;   // Maximum bars to calculate (0 = all bars)
extern bool   ShowHistogram = true;
extern bool   ShowLevels    = true;

// Level values
#define LV_0    0.0
#define LV_50   50.0
#define LV_M50  -50.0
#define LV_100  100.0
#define LV_M100 -100.0
#define LV_150  150.0
#define LV_M150 -150.0

// Subwindow index for object creation, auto-detected in init()
int ThisSubWindow = 1;

// Unique object prefix to avoid namespace conflicts
// Use StringConcatenate to build object prefix reliably (avoid '+' numeric coercion under strict mode)
string ObjectPrefix = StringConcatenate("MACDv_", IntegerToString(ChartID()), "_");

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int init()
  {
   // Set indicator short name for ChartWindowFind-style search fallback
   IndicatorShortName("MACD-v (converted)");

   IndicatorBuffers(3);
   SetIndexBuffer(0, macdvBuffer);
   SetIndexBuffer(1, signalBuffer);
   SetIndexBuffer(2, histBuffer);

   SetIndexStyle(0, DRAW_LINE, STYLE_SOLID, 2, clrDodgerBlue);
   SetIndexStyle(1, DRAW_LINE, STYLE_SOLID, 2, clrSlateBlue);
   SetIndexStyle(2, DRAW_HISTOGRAM, STYLE_SOLID, 2, clrGray);
   
   SetIndexLabel(0, "MACD-v");
   SetIndexLabel(1, "Signal");
   SetIndexLabel(2, "Histogram");

   // In MT4, WindowOnDropped() returns the subwindow index where the indicator was dropped
   ThisSubWindow = WindowOnDropped();
   if(ThisSubWindow < 0) ThisSubWindow = 1; // fallback to first indicator window

   return(0);
  }

//+------------------------------------------------------------------+
//| Create horizontal level object helper                            |
//+------------------------------------------------------------------+
void CreateLevel(string name, double price, color clr)
  {
   // Get the current subwindow dynamically using ChartWindowFind
   int currentWindow = ChartWindowFind();
   if(currentWindow < 0) currentWindow = ThisSubWindow;
   
   // If object does not exist, create in the detected subwindow; otherwise update.
   if(ObjectFind(name) < 0)
     {
      // Create the horizontal line in the intended subwindow.
      // ObjectCreate(name, type, sub_window, time, price)
      // time parameter is arbitrary for HLINE; set to Time[0].
      if(!ObjectCreate(name, OBJ_HLINE, currentWindow, Time[0], price))
         Print("Failed to create level ", name);
      else
        {
         ObjectSetInteger(0, name, OBJPROP_COLOR, clr);
         ObjectSetInteger(0, name, OBJPROP_WIDTH, 1);
         ObjectSetInteger(0, name, OBJPROP_RAY, false);
         // Keep lines non-selectable to reduce accidental movement
         ObjectSetInteger(0, name, OBJPROP_SELECTABLE, false);
         ObjectSetInteger(0, name, OBJPROP_SELECTED, false);
        }
     }
   else
     {
      // Update existing object's price and color
      ObjectSetDouble(0, name, OBJPROP_PRICE, price);
      ObjectSetInteger(0, name, OBJPROP_COLOR, clr);
     }
  }

//+------------------------------------------------------------------+
//| Clean up objects on deinit                                       |
//+------------------------------------------------------------------+
int deinit()
  {
   // Remove created level objects on deinit if requested
   if(ShowLevels)
     {
      ObjectDelete(ObjectPrefix + "Lv_150");
      ObjectDelete(ObjectPrefix + "Lv_100");
      ObjectDelete(ObjectPrefix + "Lv_50");
      ObjectDelete(ObjectPrefix + "Lv_0");
      ObjectDelete(ObjectPrefix + "Lv_m50");
      ObjectDelete(ObjectPrefix + "Lv_m100");
      ObjectDelete(ObjectPrefix + "Lv_m150");
     }
   return(0);
  }

//+------------------------------------------------------------------+
//| main calculation loop (classic MQL4 start)                       |
//+------------------------------------------------------------------+
int start()
  {
   // Ensure levels are created/updated after timeframe changes
   if(ShowLevels)
     {
      CreateLevel(ObjectPrefix + "Lv_150", LV_150, clrRed);
      CreateLevel(ObjectPrefix + "Lv_100", LV_100, clrSilver);
      CreateLevel(ObjectPrefix + "Lv_50",  LV_50,  clrLime);
      CreateLevel(ObjectPrefix + "Lv_0",   LV_0,   clrGray);
      CreateLevel(ObjectPrefix + "Lv_m50", LV_M50, clrRed);
      CreateLevel(ObjectPrefix + "Lv_m100",LV_M100,clrSilver);
      CreateLevel(ObjectPrefix + "Lv_m150",LV_M150,clrLime);
     }
   
   // Determine number of bars to process (respect MaxBarsBack but ensure enough history)
   int totalBars = Bars;
   int minBars   = MathMax(SlowLen, ATRPeriod) + SignalLen + 5;
   if(totalBars < minBars) return(0);

   int counted_bars = IndicatorCounted();
   if(counted_bars < 0) return(-1);
   if(counted_bars > 0) counted_bars--;
   
   int barsToProcess = totalBars - counted_bars;
   if(MaxBarsBack > 0 && barsToProcess > MaxBarsBack)
      barsToProcess = MaxBarsBack;

   // Process bars from oldest to newest
   int limit = barsToProcess - 1;
   
   // 1) Compute raw MACD-v values
   for(int i = limit; i >= 0; i--)
     {
      double fast = iMA(NULL, 0, FastLen, 0, MODE_EMA, PRICE_CLOSE, i);
      double slow = iMA(NULL, 0, SlowLen, 0, MODE_EMA, PRICE_CLOSE, i);
      double vola = iATR(NULL, 0, ATRPeriod, i);
      
      if(MathAbs(vola) > 1e-12)
         macdvBuffer[i] = (fast - slow) / vola * 100.0;
      else
         macdvBuffer[i] = 0.0;
     }

   // 2) Compute signal EMA
   double alpha = 2.0 / (SignalLen + 1.0);
   for(int j = limit; j >= 0; j--)
     {
      // Initialize signal only if we're processing the oldest bar in history
      // or if signalBuffer[j+1] is not yet calculated (first run scenario)
      if(j == totalBars - 1 || (j < totalBars - 1 && signalBuffer[j+1] == 0.0 && macdvBuffer[j+1] != 0.0))
         signalBuffer[j] = macdvBuffer[j];
      else
         signalBuffer[j] = alpha * macdvBuffer[j] + (1.0 - alpha) * signalBuffer[j+1];
     }

   // 3) Compute histogram
   for(int k = limit; k >= 0; k--)
     {
      if(ShowHistogram)
         histBuffer[k] = macdvBuffer[k] - signalBuffer[k];
      else
         histBuffer[k] = EMPTY_VALUE;
     }

   return(0);
  }
//+------------------------------------------------------------------+
