//+------------------------------------------------------------------+
//|                                        ZigZagExtremaOnBuffer.mqh |
//|                           Copyright 2021, Tobias Johannes Zimmer |
//|                                 https://www.mql5.com/pennyhunter |
//+------------------------------------------------------------------+
#property copyright "Copyright 2021, Tobias Johannes Zimmer"
#property link      "https://www.mql5.com/pennyhunter"

//--- Enum search mode
enum EnSearchMode {
   Extremum = 0,  // ZigZag
   Peak = 1,      // Peak
   Bottom = -1    // Bottom
};

//--- input parameters
int         ExtRecalc = 3;                  // number of last extremes for recalculation
double      var;
double      HighMapBuffer[], LowMapBuffer[];
//+------------------------------------------------------------------+
//| ZigZag calculation                                               |
//+------------------------------------------------------------------+
int ZZOnBuffer(const int rates,              //On a non indicator array, Arraysize can be used as 'rates_total'
               const int calculated,
               const double &src[],         //source buffer for calculation
               double &dst[],               //destination buffer for calculation buffer
               EnSearchMode mode_search,   //Set output buffer to High only, Low only or both in
               const int &depth,
               const int &deviation,
               const int &backstep          //ZZ input parameters
              ) {

   if(rates < 100) {
      Print("Not enough data, bars < 100");
      return(0);
   }
//---resizing buffers
   ArrayResize(HighMapBuffer, rates + 1);
   ArrayResize(LowMapBuffer, rates + 1);
//--- local auxilliary variables initialisation
   int    j = 0;
   int    start = 0, extreme_counter = 0, extreme_search = Extremum;
   int    shift = 0, back = 0, last_high_pos = 0, last_low_pos = 0;
   double val = 0, res = 0;
   double curlow = 0, curhigh = 0, last_high = 0, last_low = 0;

//--- initializing buffers
   if(calculated == 0) {
      ArrayInitialize(dst, 0.0);
      ArrayInitialize(HighMapBuffer, 0.0);
      ArrayInitialize(LowMapBuffer, 0.0);
      start = depth;
   }

//--- ZigZag was already calculated before
   if(calculated > 0) {
      j = rates - 1;
      //--- searching for the third extremum from the last uncompleted bar
      while(extreme_counter < ExtRecalc && j > rates - 100) {
         res = dst[j];
         if(res != 0.0)
            extreme_counter++;
         j--;
      }
      j++;
      start = j;

      //--- what type of exremum we search for
      if(LowMapBuffer[j] != 0.0) {
         curlow = LowMapBuffer[j];
         extreme_search = Peak;
      } else {
         curhigh = HighMapBuffer[j];
         extreme_search = Bottom;
      }
      //--- clear indicator values
      for(j = start + 1; j < rates && !IsStopped(); j++) {
         dst[j] = 0.0;
         LowMapBuffer[j] = 0.0;
         HighMapBuffer[j] = 0.0;
      }
   }

//--- searching for high and low extremes
   for(shift = start; shift < rates && !IsStopped(); shift++) {
      //--- low search
      val = src[Lowest(src, depth, shift)];
      if(val == last_low)
         val = 0.0;
      else {
         last_low = val;
         if((src[shift] - val) > deviation * _Point)
            val = 0.0;
         else {
            for(back = 1; back <= backstep; back++) {
               res = LowMapBuffer[shift - back];
               if((res != 0) && (res > val))
                  LowMapBuffer[shift - back] = 0.0;
            }
         }
      }
      if(src[shift] == val)
         LowMapBuffer[shift] = val;
      else
         LowMapBuffer[shift] = 0.0;
      //--- high search
      val = src[Highest(src, depth, shift)];
      if(val == last_high)
         val = 0.0;
      else {
         last_high = val;
         if((val - src[shift]) > deviation * _Point)
            val = 0.0;
         else {
            for(back = 1; back <= backstep; back++) {
               res = HighMapBuffer[shift - back];
               if((res != 0) && (res < val))
                  HighMapBuffer[shift - back] = 0.0;
            }
         }
      }
      if(src[shift] == val)
         HighMapBuffer[shift] = val;
      else
         HighMapBuffer[shift] = 0.0;
   }

//--- set last values
   if(extreme_search == 0) { // undefined values
      last_low = 0.0;
      last_high = 0.0;
   } else {
      last_low = curlow;
      last_high = curhigh;
   }

//--- final selection of extreme points for ZigZag
   for(shift = start; shift < rates && !IsStopped(); shift++) {
      res = 0.0;
      switch(extreme_search) {
      case Extremum:
         if(last_low == 0.0 && last_high == 0.0) {
            if(HighMapBuffer[shift] != 0) {
               last_high = src[shift];
               last_high_pos = shift;
               extreme_search = Bottom;
               dst[shift] = last_high * (mode_search == Extremum) + last_high * (mode_search == Peak);
               res = 1;
            }
            if(LowMapBuffer[shift] != 0.0) {
               last_low = src[shift];
               last_low_pos = shift;
               extreme_search = Peak;
               dst[shift] = last_low * (mode_search == Extremum) + last_low * (mode_search == Bottom);
               res = 1;
            }
         }
         break;
      case Peak:
         if(LowMapBuffer[shift] != 0.0 && LowMapBuffer[shift] < last_low && HighMapBuffer[shift] == 0.0) {
            dst[last_low_pos] = 0.0;
            last_low_pos = shift;
            last_low = LowMapBuffer[shift];
            dst[shift] = last_low * (mode_search == Extremum) + last_low * (mode_search == Bottom);
            res = 1;
         }
         if(HighMapBuffer[shift] != 0.0 && LowMapBuffer[shift] == 0.0) {
            last_high = HighMapBuffer[shift];
            last_high_pos = shift;
            dst[shift] = last_high * (mode_search == Extremum) + last_high * (mode_search == Peak);
            extreme_search = Bottom;
            res = 1;
         }
         break;
      case Bottom:
         if(HighMapBuffer[shift] != 0.0 && HighMapBuffer[shift] > last_high && LowMapBuffer[shift] == 0.0) {
            dst[last_high_pos] = 0.0;
            last_high_pos = shift;
            last_high = HighMapBuffer[shift];
            dst[shift] = last_high * (mode_search == Extremum) + last_high * (mode_search == Peak);;
         }
         if(LowMapBuffer[shift] != 0.0 && HighMapBuffer[shift] == 0.0) {
            last_low = LowMapBuffer[shift];
            last_low_pos = shift;
            dst[shift] = last_low * (mode_search == Extremum) + last_low * (mode_search == Bottom);
            extreme_search = Peak;
         }
         break;
      default:
         return(rates);
      }
   }

//--- return value of calculated for next call
   return(rates);
}
//+------------------------------------------------------------------+
//|  Search for the index of the highest bar                         |
//+------------------------------------------------------------------+
int Highest(const double &array[], const int depth, const int start) {
   if(start < 0)
      return(0);

   double max = array[start];
   int    index = start;
//--- start searching
   for(int i = start - 1; i > start - depth && i >= 0; i--) {
      if(array[i] > max) {
         index = i;
         max = array[i];
      }
   }
//--- return index of the highest bar
   return(index);
}
//+------------------------------------------------------------------+
//|  Search for the index of the lowest bar                          |
//+------------------------------------------------------------------+
int Lowest(const double &array[], const int depth, const int start) {
   if(start < 0)
      return(0);

   double min = array[start];
   int    index = start;
//--- start searching
   for(int i = start - 1; i > start - depth && i >= 0; i--) {
      if(array[i] < min) {
         index = i;
         min = array[i];
      }
   }
//--- return index of the lowest bar
   return(index);
}

//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
