//+------------------------------------------------------------------+
//| FILE:    Include/DeeFX/CLatencyMonitor.mqh                       |
//| BRAND:   DeeFX Precision Labs                                    |
//| PRODUCT: LagShield Pro                                           |
//| VERSION: v1.2                                                    |
//| DATE:    2026-02-24                                              |
//| TYPE:    Include — Latency Tracking / ATR Volatility / GV IPC   |
//| DESC:    Measures inter-tick delta, detects high-volatility lag, |
//|          manages GlobalVariable IPC flag for cross-EA signaling  |
//+------------------------------------------------------------------+
// ChangeLog:
//   v1.0 — Initial release.
//   v1.1 — Removed dead members m_hAtrSma + m_smaBuf (never used — manual
//           SMA calculated from m_atrBuf only). Removed associated
//           IndicatorRelease(m_hAtrSma) guard in Deinit().
//           Guard macro renamed CLATENCYMONITOR_MQH -> CLAT_MONITOR_MQH.
//           %I64u format specifiers replaced with (string) casts.
//   v1.2 — Destructor cleared (was calling Deinit() — violates "destructor does nothing" standard).
//           Init() now resets all timing/lag state — prevents stale values on EA reinit.
//+------------------------------------------------------------------+
#ifndef CLAT_MONITOR_MQH
#define CLAT_MONITOR_MQH

#define GV_LAG_FLAG "Terminal_Lag_Detected"

class CLatencyMonitor
  {
private:
   //--- Config
   ulong             m_lagThresholdMs;
   int               m_persistenceSec;
   int               m_atrPeriod;
   double            m_atrSmaMultiplier;

   //--- ATR
   int               m_hAtr;
   bool              m_atrValid;
   double            m_atrBuf[];            // Manual SMA computed from this buffer

   //--- Timing state
   ulong             m_lastTickTime;        // GetTickCount64() at last tick
   bool              m_firstTick;           // Skip delta measurement on first tick

   //--- Lag state
   bool              m_lagActive;           // True = lag condition currently active
   ulong             m_lagStartTime;        // Wall clock when lag first triggered
   bool              m_gvOwnedByUs;         // True = we set the GV (cleanup guard)

public:
                     CLatencyMonitor()
     : m_lagThresholdMs(500),
       m_persistenceSec(3),
       m_atrPeriod(14),
       m_atrSmaMultiplier(1.5),
       m_hAtr(INVALID_HANDLE),
       m_atrValid(false),
       m_lastTickTime(0),
       m_firstTick(true),
       m_lagActive(false),
       m_lagStartTime(0),
       m_gvOwnedByUs(false)
     {}

                    ~CLatencyMonitor() {}   // Destructor does nothing — Deinit() is explicit

   //--- Init — call from OnInit()
   bool              Init(ulong lagThresholdMs, int persistenceSec,
                          int atrPeriod, double atrSmaMultiplier,
                          string symbol, ENUM_TIMEFRAMES tf)
     {
      //--- Reset all timing and lag state — values persist across OnDeinit/OnInit cycles
      m_firstTick    = true;
      m_lastTickTime = 0;
      m_lagActive    = false;
      m_lagStartTime = 0;
      m_atrValid     = false;
      // m_gvOwnedByUs: Deinit() already cleared this and deleted the GV — no double-delete

      m_lagThresholdMs   = lagThresholdMs;
      m_persistenceSec   = persistenceSec;
      m_atrPeriod        = atrPeriod;
      m_atrSmaMultiplier = atrSmaMultiplier;

      m_hAtr = iATR(symbol, tf, atrPeriod);
      if(m_hAtr == INVALID_HANDLE)
        {
         Alert("DeeFX | CLatencyMonitor: ATR handle failed. Volatility gate DISABLED.");
         m_atrValid = false;
         return true;   // Non-fatal — utility continues without volatility gate
        }

      m_atrValid = true;
      ArraySetAsSeries(m_atrBuf, true);
      return true;
     }

   //--- Called from OnTick() — measure inter-tick delta, evaluate ATR gate
   //    Returns true if lag is newly triggered this tick (caller drives immediate UI)
   bool              OnTick()
     {
      ulong now = GetTickCount64();

      //--- First tick: initialise timestamp only, skip delta
      if(m_firstTick)
        {
         m_lastTickTime = now;
         m_firstTick    = false;
         return false;
        }

      ulong delta    = now - m_lastTickTime;
      m_lastTickTime = now;

      if(delta > m_lagThresholdMs && IsHighVolatility())
        {
         if(!m_lagActive)
           {
            m_lagActive    = true;
            m_lagStartTime = now;
            PrintFormat("DeeFX | LAG DETECTED | Delta: %s ms | ATR gate: HIGH", (string)delta);
           }
         return true;
        }

      //--- Lag cleared this tick
      if(m_lagActive)
        {
         m_lagActive = false;
         PrintFormat("DeeFX | LAG CLEARED | Duration: %s ms",
                     (string)(now - m_lagStartTime));
        }
      return false;
     }

   //--- Called from OnTimer() — persistence check, GlobalVariable management
   //    Returns true if lag has persisted beyond PersistenceSec threshold
   bool              CheckPersistence()
     {
      if(!m_lagActive)
        {
         //--- Lag cleared — release GV if we own it
         if(m_gvOwnedByUs)
           {
            GlobalVariableSet(GV_LAG_FLAG, 0.0);
            m_gvOwnedByUs = false;
            PrintFormat("DeeFX | GV [%s] cleared (lag resolved)", GV_LAG_FLAG);
           }
         return false;
        }

      ulong elapsedMs = GetTickCount64() - m_lagStartTime;
      bool  panic     = (elapsedMs >= (ulong)(m_persistenceSec * 1000));

      if(panic && !m_gvOwnedByUs)
        {
         //--- Read-before-write: never overwrite another EA's flag
         if(GlobalVariableGet(GV_LAG_FLAG) < 1.0)
           {
            GlobalVariableSet(GV_LAG_FLAG, 1.0);
            m_gvOwnedByUs = true;
            PrintFormat("DeeFX | GV [%s] set — lag > %d s", GV_LAG_FLAG, m_persistenceSec);
           }
        }

      return panic;
     }

   //--- Accessors
   ulong             LastDeltaMs() const { return (m_firstTick ? 0 : GetTickCount64() - m_lastTickTime); }
   bool              IsLagActive() const { return m_lagActive; }
   bool              IsAtrValid()  const { return m_atrValid;  }

   //--- Cleanup — release ATR handle, delete GV only if owned
   void              Deinit()
     {
      if(m_gvOwnedByUs)
        {
         GlobalVariableDel(GV_LAG_FLAG);
         m_gvOwnedByUs = false;
        }
      if(m_hAtr != INVALID_HANDLE)
        {
         IndicatorRelease(m_hAtr);
         m_hAtr = INVALID_HANDLE;
        }
     }

private:
   //--- Self-normalising ATR gate: ATR > multiplier x SMA(ATR, period)
   //    Instrument-agnostic — adapts across all pairs and timeframes automatically
   bool              IsHighVolatility()
     {
      if(!m_atrValid || m_hAtr == INVALID_HANDLE)
         return true;   // Gate disabled — fail open (allow lag detection)

      if(CopyBuffer(m_hAtr, 0, 0, m_atrPeriod + 1, m_atrBuf) < m_atrPeriod + 1)
         return true;   // Insufficient history — fail open

      double currentAtr = m_atrBuf[0];

      //--- Manual SMA over m_atrPeriod bars (index 1..period, index 0 = current)
      double sum = 0.0;
      for(int i = 1; i <= m_atrPeriod; i++)
         sum += m_atrBuf[i];
      double atrSma = sum / m_atrPeriod;

      if(atrSma <= 0.0) return true;   // Guard divide-by-zero

      return (currentAtr > m_atrSmaMultiplier * atrSma);
     }
  };

#endif // CLAT_MONITOR_MQH
//+------------------------------------------------------------------+
