//+------------------------------------------------------------------+
//|                                               ShowTradeLines.mq5 |
//|                                    Copyright (c) 2026, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//+------------------------------------------------------------------+
#property copyright   "Copyright (c) 2026 Marketeer"
#property link        "https://www.mql5.com/en/users/marketeer"
#property description "Show entry/exit points of existing positions/deals as trend lines and/or arrows on charts."
#property version     "1.0"
#property service

#define PUSH(A,V) (A[ArrayResize(A, ArrayRange(A, 0) + 1, ArrayRange(A, 0) * 2) - 1] = V)

//+------------------------------------------------------------------+
//| I N P U T S                                                      |
//+------------------------------------------------------------------+

enum LINE_TYPE
{
   OBJ_TREND_ = OBJ_TREND, // Trend Line
   OBJ_VLINE_ = OBJ_VLINE  // Vertical Line
};

input int TimeInterval = 5; // TimeInterval (seconds)
input string LinePrefix = "TradeTag";
input color LineColorProfit = clrDodgerBlue;
input color LineColorLoss = clrCoral;
input ENUM_LINE_STYLE LineStyle = STYLE_DASHDOT;
input int LineWidth = 1;
input LINE_TYPE LineType = OBJ_TREND_;
input bool ShowDeals = true;
input bool KeepLinesOnCharts = false;
input bool Logging = true;

//+------------------------------------------------------------------+
//| G L O B A L S                                                    |
//+------------------------------------------------------------------+

int OwnedVars = 1;
long LastCID = 0;
const string DealTag = "D_";

//+------------------------------------------------------------------+
//| M A I N                                                          |
//+------------------------------------------------------------------+

void OnStart()
{
   if(Logging) Print(MQLInfoString(MQL_PROGRAM_NAME) + " started");
   while(!IsStopped())
   {
      if(!CheckPositions())
      {
         Print("Critical failure: stopped");
         return; // exit on error
      }
      Sleep(TimeInterval * 1000);
   }
   if(Logging) Print(MQLInfoString(MQL_PROGRAM_NAME) + " stopped");
}

//+------------------------------------------------------------------+
//| Display positions as objects on corresponding charts             |
//+------------------------------------------------------------------+

bool CheckPositions()
{
   if(OwnedVars)
   {
      // check for outdated objects for removed positions
      OwnedVars = RemoveOwnedGlobalVariables();
   }

   if(!PositionsTotal()) return true;

   static string symbols[];
   static ulong tickets[];
   for(int i = 0; i < PositionsTotal(); i++)
   {
      PUSH(symbols, PositionGetSymbol(i));
      PUSH(tickets, PositionGetTicket(i));
   }
   
   if(ArraySize(symbols) != ArraySize(tickets)) return false;
   
   // check last active chart or browse them for new active one
   long cid = LastCID && (ChartGetInteger(LastCID, CHART_BRING_TO_TOP) || ChartGetInteger(LastCID, CHART_IS_MAXIMIZED)) ? LastCID : ChartFirst();
   LastCID = 0; // need to re-assign in the loop below, otherwise will start from first next time
   int i = 0;
   while(cid > -1)
   {
      const bool t = (bool)ChartGetInteger(cid, CHART_BRING_TO_TOP);
      const bool m = (bool)ChartGetInteger(cid, CHART_IS_MAXIMIZED);
      if(t || m) // only one chart is on top
      {
         int indices[];
         const string symbol = ChartSymbol(cid);
         const int n = ArrayContains(symbols, symbol, indices);
         for(int i = 0; i < n; i++)
         {
            const int p = indices[i];
            if(PositionSelectByTicket(tickets[p]))
            {
               const datetime open = (datetime)PositionGetInteger(POSITION_TIME);
               const double profit = PositionGetDouble(POSITION_PROFIT); // NB! If a partial close made for position, it's profit/loss is NOT included into this FLOATING profit
               const double swap = PositionGetDouble(POSITION_SWAP);
               
               const string name = LinePrefix + "_" + (string)tickets[p] + "_" + (string)cid;
               int w = ObjectFind(cid, name);
               if(w < 0)
               {
                  if(Logging) PrintFormat("Adding position: #%lld %s -> chart %lld %s %s",
                     tickets[p], TimeToString(open, TIME_DATE | TIME_SECONDS),
                     cid, symbol, EnumToString(ChartPeriod(cid)));
                  if(ObjectCreate(cid, name, (ENUM_OBJECT)LineType, 0, open, PositionGetDouble(POSITION_PRICE_OPEN),
                     TimeCurrent(), PositionGetDouble(POSITION_PRICE_CURRENT)))
                  {
                     w = 1; // flag to assign additional properties below
                     ObjectSetInteger(cid, name, OBJPROP_STYLE, LineStyle);
                     ObjectSetInteger(cid, name, OBJPROP_WIDTH, LineWidth);
                     
                     if(!GlobalVariableCheck(name) && GlobalVariableSet(name, tickets[p]))
                     {
                        OwnedVars++;
                     }
                  }
               }
               
               if(w >= 0)
               {
                  ObjectSetInteger(cid, name, OBJPROP_COLOR, profit + swap >= 0 ? LineColorProfit : LineColorLoss);
                  if(LineType == OBJ_TREND_)
                  {
                     ObjectSetInteger(cid, name, OBJPROP_TIME, 1, TimeCurrent());
                     ObjectSetDouble(cid, name, OBJPROP_PRICE, 1, PositionGetDouble(POSITION_PRICE_CURRENT));
                  }
                  ObjectSetString(cid, name, OBJPROP_TEXT,
                     StringFormat("%s %s Profit:%+.2f%s", (PositionGetInteger(POSITION_TYPE) ? "Sell" : "Buy"),
                     (string)PositionGetDouble(POSITION_VOLUME), profit, StringFormatNonZero(" Swap:%+.2f", swap)));
               }
               
               if(ShowDeals && HistorySelectByPosition(PositionGetInteger(POSITION_IDENTIFIER)))
               {
                  for(int i = HistoryDealsTotal() - 1; i >= 0; i--)
                  {
                     const ulong t = HistoryDealGetTicket(i);
                     const ENUM_DEAL_ENTRY e = (ENUM_DEAL_ENTRY)HistoryDealGetInteger(t, DEAL_ENTRY);
                     // if(e == DEAL_ENTRY_IN || e == DEAL_ENTRY_INOUT) // show all type of deals, including partial closure
                     {
                        if(!CreateDealArrow(symbol, t, cid, tickets[p])) break; // exit loop on first already existing object
                        
                        if(e == DEAL_ENTRY_INOUT) break; // stop lookup at most recent flip of position
                     }
                  }
               }
            }
         }
         LastCID = cid;
         return true; // no need to review other charts if active one is just processed
      }
      cid = ChartNext(cid);
   }
   return true;
}

//+------------------------------------------------------------------+
//| Helper for deal visualization                                    |
//+------------------------------------------------------------------+

bool CreateDealArrow(const string symbol, const ulong t, const long cid, const ulong position, const bool createGV = true)
{
   const string arrow = LinePrefix + DealTag + (string)t + "_" + (string)cid;
   if(ObjectFind(cid, arrow) < 0)
   {
      const ENUM_DEAL_ENTRY e = (ENUM_DEAL_ENTRY)HistoryDealGetInteger(t, DEAL_ENTRY);
      const ENUM_OBJECT sign = HistoryDealGetInteger(t, DEAL_TYPE) == DEAL_TYPE_BUY ? OBJ_ARROW_BUY : (HistoryDealGetInteger(t, DEAL_TYPE) == DEAL_TYPE_SELL ? OBJ_ARROW_SELL : OBJ_ARROW_CHECK);
      if(Logging) PrintFormat("Adding deal: #%lld %s -> chart %lld %s %s",
         t, TimeToString(HistoryDealGetInteger(t, DEAL_TIME), TIME_DATE | TIME_SECONDS),
         cid, symbol, EnumToString(ChartPeriod(cid)));
      if(ObjectCreate(cid, arrow, sign, 0, HistoryDealGetInteger(t, DEAL_TIME), HistoryDealGetDouble(t, DEAL_PRICE)))
      {
         const string info = StringFormat("%s %s @ %.*f %s", EnumToStringShort((ENUM_DEAL_TYPE)HistoryDealGetInteger(t, DEAL_TYPE)),
            (string)HistoryDealGetDouble(t, DEAL_VOLUME), SymbolInfoInteger(symbol, SYMBOL_DIGITS), HistoryDealGetDouble(t, DEAL_PRICE), EnumToStringShort(e));
         const string added = (e != DEAL_ENTRY_IN) ? StringFormat(" Profit:%+.2f%s%s",
               HistoryDealGetDouble(t, DEAL_PROFIT),
               StringFormatNonZero(" Swap:%+.2f", HistoryDealGetDouble(t, DEAL_SWAP)),
               StringFormatNonZero(" Comm:%+.2f", HistoryDealGetDouble(t, DEAL_COMMISSION)))
               : StringFormatNonZero(" Comm:%+.2f", HistoryDealGetDouble(t, DEAL_COMMISSION));
         ObjectSetString(cid, arrow, OBJPROP_TEXT, info + added);
         if(createGV && !GlobalVariableCheck(arrow) && GlobalVariableSet(arrow, /*tickets[p]*/position))
         {
            OwnedVars++;
         }
         return true;
      }
   }
   return false;
}

//+------------------------------------------------------------------+
//| Cleanup of controlled Global Variables that are not used anymore |
//+------------------------------------------------------------------+

int RemoveOwnedGlobalVariables()
{
   const int n = GlobalVariablesTotal();
   if(!n) return 0;
   int owned = 0; // counter of variables under control
   for(int i = n - 1; i >= 0; i--)
   {
      const string s = GlobalVariableName(i);
      if(StringFind(s, LinePrefix) == 0)
      {
         owned++;
         const ulong ticket = (ulong)GlobalVariableGet(s);
         static string words[];
         const int m = StringSplit(s, '_', words);
         if(m > 2)
         {
            const long cid = StringToInteger(words[m - 1]);
            if(!PositionSelectByTicket(ticket))
            {
               if(ShowDeals && KeepLinesOnCharts) // try to add OUT deal for closed position
               {
                  if(StringFind(s, DealTag) > -1) // this is an IN deal object with deal ticket in its name
                  {
                     const long deal = StringToInteger(words[m - 2]);
                     const string symbol = HistoryDealGetString(deal, DEAL_SYMBOL);
                     ResetLastError();
                     if(HistorySelectByPosition(HistoryDealGetInteger(deal, DEAL_POSITION_ID)))
                     {
                        for(int j = HistoryDealsTotal() - 1; j >= 0; j--)
                        {
                           // create object if it's not existed yet, otherwise stop lookup of deals
                           if(!CreateDealArrow(symbol, HistoryDealGetTicket(j), cid, ticket, false)) break;
                        }
                     }
                     else
                     {
                        PrintFormat("Can't select history by deal %lld, error: %d", deal, _LastError);
                     }
                  }
               }
               if(!KeepLinesOnCharts)
               {
                  if(Logging) PrintFormat("Removing line '%s' on chart %lld (position #%lld)", s, cid, ticket);
                  ObjectDelete(cid, s);
               }
               if(Logging) PrintFormat("Removing gvariable '%s' for chart %lld (position #%lld)", s, cid, ticket);
               if(GlobalVariableDel(s)) owned--;
            }
            else
            {
               if(!ChartGetInteger(cid, CHART_WINDOW_HANDLE))
               {
                  if(Logging) PrintFormat("Removing gvariable '%s' for closed chart %lld (position #%lld)", s, cid, ticket);
                  if(GlobalVariableDel(s)) owned--;
               }
               else
               {
                  if(ObjectFind(cid, s) < 0)
                  {
                     if(Logging) PrintFormat("Removing gvariable '%s' for missing object on chart %lld (position #%lld)", s, cid, ticket);
                     if(GlobalVariableDel(s)) owned--;
                  }
               }
            }
         }
         else
         {
            // if(Logging) PrintFormat("Unconventional variable name '%s' with prefix %s", s, LinePrefix);
            // could be a collision or improper prefix setup
         }
      }
   }
   return owned;
}

//+------------------------------------------------------------------+
//| Auxiliary stuff                                                  |
//+------------------------------------------------------------------+

template<typename T>
int ArrayContains(const T &a[], const T v, int &indices[])
{
   ArrayResize(indices, 0);
   for(int i = 0; i < ArraySize(a); i++)
   {
      if(a[i] == v)
      {
         PUSH(indices, i);
      }
   }
   return ArraySize(indices);
}

template<typename E>
string EnumToStringShort(const E e)
{
   string element = EnumToString(e);
   string type = typename(e);
   if(StringReplace(type, "enum ENUM_", "") == 1)
   {
      StringReplace(element, type + "_", "");
   }
   return element;
}

template<typename T>
string StringFormatNonZero(const string format, const T value)
{
   if(value != 0)
   {
      return StringFormat(format, value);
   }
   return "";
}

//+------------------------------------------------------------------+
