//+------------------------------------------------------------------+
//|                                                   EADetector.mqh |
//|                                       Copyright 2024-2026, hini. |
//|                               https://www.mql5.com/en/users/hini |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024-2026, hini."
#property link      "https://www.mql5.com/en/users/hini"
#property strict
#include "Expert.mqh"

//+------------------------------------------------------------------+
//| Enum for EA Duplicate Detection Policies                         |
//+------------------------------------------------------------------+
enum ENUM_DUPLICATE_POLICY {
  POLICY_ALLOW_ALL,                    // Allow all (No detection)
  POLICY_ONE_PER_TERMINAL,             // Only one instance allowed globally in the terminal
  POLICY_ONE_PER_SYMBOL,               // Only one instance allowed per symbol (regardless of period)
  POLICY_ONE_PER_SYMBOL_PERIOD,        // Only one instance allowed per symbol and period
  POLICY_UNIQUE_PARAMS_ONLY            // Only allow running if parameters are different (regardless of symbol/period)
};

//+------------------------------------------------------------------+
//| Enum for actions to take when a duplicate is detected            |
//+------------------------------------------------------------------+
enum ENUM_DUPLICATE_ACTION {
  ACTION_DENY_WITH_MESSAGE,            // Deny and show an alert box (Default)
  ACTION_ASK_USER                      // Ask the user for a decision
};

//+------------------------------------------------------------------+
//| Structure for Duplicate Detector Configuration (Simplified)      |
//+------------------------------------------------------------------+
struct SDuplicateDetectorConfig {
  // Basic Configuration
  ENUM_DUPLICATE_POLICY policy;        // Detection policy
  ENUM_DUPLICATE_ACTION action;        // Handling action

  // Message Configuration
  string msgContent;                   // Popup content ({info} will be replaced)

  // Behavioral Control
  bool skipInTester;                   // Skip detection in Strategy Tester mode

  // Constructor - Default values
  SDuplicateDetectorConfig() {
    policy = POLICY_ONE_PER_SYMBOL_PERIOD;
    action = ACTION_DENY_WITH_MESSAGE;
    msgContent = "The same EA is already running:\n{info}\n\nExecution cannot continue!";
    skipInTester = true;
  }
};

//+------------------------------------------------------------------+
//| Structure for Detection Results                                  |
//+------------------------------------------------------------------+
struct SDuplicateDetectResult {
  bool isDuplicate;                    // Whether a duplicate was found
  long duplicateChartId;               // Chart ID where the duplicate EA is located
  string duplicateInfo;                // Description of the duplicate EA instance
  int userDecision;                    // User decision (INIT_SUCCEEDED/INIT_FAILED)

  SDuplicateDetectResult() {
    isDuplicate = false;
    duplicateChartId = 0;
    duplicateInfo = "";
    userDecision = INIT_SUCCEEDED;
  }
};

//+------------------------------------------------------------------+
//| Static Class for EA Duplicate Detection                          |
//+------------------------------------------------------------------+
class CEADuplicateDetector {
private:
  // Get EA information string (returns different formats based on policy)
  static string GetExpertInfo(const long chartId, const ENUM_DUPLICATE_POLICY policy) {
    if (!EXPERT::Is(chartId)) {
      return "";
    }

    MqlParam parameters[];
    string names[];

    if (!EXPERT::Parameters(chartId, parameters, names)) {
      return "";
    }

    string info = "";

    switch (policy) {
      case POLICY_ONE_PER_TERMINAL:
        // Global policy: Show EA name
        info = parameters[0].string_value;
        break;

      case POLICY_ONE_PER_SYMBOL:
        // Symbol policy: Show Symbol + EA name
        info = ::ChartSymbol(chartId) + " - " + parameters[0].string_value;
        break;

      case POLICY_ONE_PER_SYMBOL_PERIOD:
        // Symbol/Period policy: Show Symbol + Period + EA name
        info = ::ChartSymbol(chartId) + " " +
               ::EnumToString(::ChartPeriod(chartId)) + " - " +
               parameters[0].string_value;
        break;

      case POLICY_UNIQUE_PARAMS_ONLY: {
        // Unique parameters policy: Show full parameter information
        info = ::ChartSymbol(chartId) + " " +
               ::EnumToString(::ChartPeriod(chartId)) + " - " +
               parameters[0].string_value + "\nParameters:\n";
        const int amount = ::ArraySize(names);
        for (int i = 0; i < amount; i++) {
          info += "  " + names[i] + " = " + parameters[i + 1].string_value + "\n";
        }
        break;
      }

      default:
        info = "";
    }

    return info;
  }

  // Compare input parameters of two EAs
  static bool CompareParameters(const long chartId1, const long chartId2) {
    MqlParam params1[], params2[];
    string names1[], names2[];

    if (!EXPERT::Parameters(chartId1, params1, names1) ||
        !EXPERT::Parameters(chartId2, params2, names2)) {
      return false;
    }

    // Check number of parameters
    if (::ArraySize(params1) != ::ArraySize(params2)) {
      return false;
    }

    // Compare each parameter
    const int count = ::ArraySize(names1);
    for (int i = 0; i < count; i++) {
      // Compare parameter names
      if (names1[i] != names2[i]) {
        return false;
      }

      // Compare parameter values
      if (params1[i + 1].string_value != params2[i + 1].string_value) {
        return false;
      }
    }

    return true;
  }

  // Get the name of the Expert Advisor
  static string GetExpertName(const long chartId) {
    if (!EXPERT::Is(chartId)) {
      return "";
    }

    MqlParam parameters[];
    string names[];

    if (!EXPERT::Parameters(chartId, parameters, names)) {
      return "";
    }

    // parameters[0].string_value is the EA name
    return parameters[0].string_value;
  }

  // Compare EAs on two charts based on the specified policy
  static bool CompareByPolicy(const long chartId1, const long chartId2,
                              const ENUM_DUPLICATE_POLICY policy) {
    // Check if both charts have an EA attached
    if (!EXPERT::Is(chartId1) || !EXPERT::Is(chartId2)) {
      return false;
    }

    switch (policy) {
      case POLICY_ALLOW_ALL:
        // Allow all, no detection logic
        return false;

      case POLICY_ONE_PER_TERMINAL:
        // Global limit: Compare EA names
        return GetExpertName(chartId1) == GetExpertName(chartId2);

      case POLICY_ONE_PER_SYMBOL:
        // Per symbol limit: Compare Symbol and EA name
        return ::ChartSymbol(chartId1) == ::ChartSymbol(chartId2) &&
               GetExpertName(chartId1) == GetExpertName(chartId2);

      case POLICY_ONE_PER_SYMBOL_PERIOD:
        // Per symbol and period limit: Compare Symbol, Period, and EA name
        return ::ChartSymbol(chartId1) == ::ChartSymbol(chartId2) &&
               ::ChartPeriod(chartId1) == ::ChartPeriod(chartId2) &&
               GetExpertName(chartId1) == GetExpertName(chartId2);

      case POLICY_UNIQUE_PARAMS_ONLY:
        // Unique parameters: Compare all input parameters
        return CompareParameters(chartId1, chartId2);

      default:
        return false;
    }
  }

  // Handle a detected duplicate
  static int HandleDuplicate(const long duplicateChartId, const string info,
                            const SDuplicateDetectorConfig &config) {
    // Construct the message
    string message = config.msgContent;
    ::StringReplace(message, "{info}", info);

    // Dialog title
    string title = "Duplicate EA Detected";

    // Action based on configuration
    if (config.action == ACTION_DENY_WITH_MESSAGE) {
      // Deny and show Warning box (MB_OK)
      ::MessageBox(message, title, MB_OK | MB_ICONWARNING);
      return INIT_FAILED;
    }

    // ACTION_ASK_USER: Ask for user choice (MB_YESNO)
    int result = ::MessageBox(message, title, MB_YESNO | MB_ICONQUESTION);

    // Yes: Allow execution, No: Deny execution
    return (result == IDYES) ? INIT_SUCCEEDED : INIT_FAILED;
  }

public:
  //+------------------------------------------------------------------+
  //| Main detection method: Check if the EA on current chart is a dup |
  //+------------------------------------------------------------------+
  static SDuplicateDetectResult Detect(const SDuplicateDetectorConfig &config) {
    SDuplicateDetectResult result;

    // Skip detection if in Strategy Tester
    if (config.skipInTester && ::MQLInfoInteger(MQL_TESTER)) {
      result.userDecision = INIT_SUCCEEDED;
      return result;
    }

    // Return immediately if policy allows all
    if (config.policy == POLICY_ALLOW_ALL) {
      result.userDecision = INIT_SUCCEEDED;
      return result;
    }

    const long currentChartId = ::ChartID();
    long chartId = ::ChartFirst();

    // Iterate through all open charts
    while (chartId != -1) {
      // Skip the current chart itself
      if (chartId != currentChartId) {
        // Compare based on policy
        if (CompareByPolicy(currentChartId, chartId, config.policy)) {
          result.isDuplicate = true;
          result.duplicateChartId = chartId;

          // Get details of the duplicate instance
          result.duplicateInfo = GetExpertInfo(chartId, config.policy);

          // Handle the duplication
          result.userDecision = HandleDuplicate(chartId, result.duplicateInfo, config);

          return result;
        }
      }

      chartId = ::ChartNext(chartId);
    }

    // No duplicates found
    result.userDecision = INIT_SUCCEEDED;
    return result;
  }

  //+------------------------------------------------------------------+
  //| Simplified version: Detect using default configuration           |
  //+------------------------------------------------------------------+
  static int DetectSimple() {
    SDuplicateDetectorConfig config;
    SDuplicateDetectResult result = Detect(config);
    return result.userDecision;
  }

  //+------------------------------------------------------------------+
  //| Quick check: Returns true if duplicate is running, no prompts    |
  //+------------------------------------------------------------------+
  static bool IsRunning(const ENUM_DUPLICATE_POLICY policy = POLICY_ONE_PER_SYMBOL_PERIOD) {
    const long currentChartId = ::ChartID();
    long chartId = ::ChartFirst();

    while (chartId != -1) {
      if (chartId != currentChartId) {
        if (CompareByPolicy(currentChartId, chartId, policy)) {
          return true;
        }
      }
      chartId = ::ChartNext(chartId);
    }

    return false;
  }

  //+------------------------------------------------------------------+
  //| Get a list of Chart IDs running the same EA                      |
  //+------------------------------------------------------------------+
  static void GetDuplicateCharts(long &chartIds[],
                                 const ENUM_DUPLICATE_POLICY policy = POLICY_ONE_PER_SYMBOL_PERIOD) {
    ::ArrayResize(chartIds, 0);

    const long currentChartId = ::ChartID();
    long chartId = ::ChartFirst();

    while (chartId != -1) {
      if (chartId != currentChartId && CompareByPolicy(currentChartId, chartId, policy)) {
        const int size = ::ArraySize(chartIds);
        ::ArrayResize(chartIds, size + 1);
        chartIds[size] = chartId;
      }

      chartId = ::ChartNext(chartId);
    }
  }
};

//+------------------------------------------------------------------+
//| Global Convenience Functions: Direct calls returning bool        |
//+------------------------------------------------------------------+

// Check: Only one instance allowed in the terminal globally
// Returns: true = allow running, false = should stop
bool CheckDuplicate_OnePerTerminal() {
  SDuplicateDetectorConfig config;
  config.policy = POLICY_ONE_PER_TERMINAL;
  config.action = ACTION_DENY_WITH_MESSAGE;
  config.msgContent = "This EA is already running in this terminal!\n{info}\n\nOnly one instance is allowed at a time.";

  SDuplicateDetectResult result = CEADuplicateDetector::Detect(config);
  return (result.userDecision == INIT_SUCCEEDED);
}

// Check: Only one instance per symbol (regardless of timeframe)
// Returns: true = allow running, false = should stop
bool CheckDuplicate_OnePerSymbol() {
  SDuplicateDetectorConfig config;
  config.policy = POLICY_ONE_PER_SYMBOL;
  config.action = ACTION_DENY_WITH_MESSAGE;
  config.msgContent = "An instance is already running on symbol {info}!\n\nExecution cannot continue!";

  SDuplicateDetectResult result = CEADuplicateDetector::Detect(config);
  return (result.userDecision == INIT_SUCCEEDED);
}

// Check: Only one instance per symbol and timeframe (Recommended default)
// Returns: true = allow running, false = should stop
bool CheckDuplicate_OnePerSymbolPeriod() {
  SDuplicateDetectorConfig config;
  config.policy = POLICY_ONE_PER_SYMBOL_PERIOD;
  config.action = ACTION_DENY_WITH_MESSAGE;
  config.msgContent = "An instance is already running on this symbol and period!\n{info}\n\nExecution cannot continue!";

  SDuplicateDetectResult result = CEADuplicateDetector::Detect(config);
  return (result.userDecision == INIT_SUCCEEDED);
}

// Check: Only allow running if parameters are unique
// Returns: true = allow running, false = should stop
bool CheckDuplicate_UniqueParams() {
  SDuplicateDetectorConfig config;
  config.policy = POLICY_UNIQUE_PARAMS_ONLY;
  config.action = ACTION_DENY_WITH_MESSAGE;
  config.msgContent = "An EA with identical parameters was detected:\n{info}\n\nExecution cannot continue!";

  SDuplicateDetectResult result = CEADuplicateDetector::Detect(config);
  return (result.userDecision == INIT_SUCCEEDED);
}

// Check: Ask the user whether to continue (Per symbol and timeframe)
// Returns: true = user chose to continue, false = user chose to stop
bool CheckDuplicate_AskUser() {
  SDuplicateDetectorConfig config;
  config.policy = POLICY_ONE_PER_SYMBOL_PERIOD;
  config.action = ACTION_ASK_USER;
  config.msgContent = "Duplicate EA detected:\n{info}\n\nDo you want to continue running?";

  SDuplicateDetectResult result = CEADuplicateDetector::Detect(config);
  return (result.userDecision == INIT_SUCCEEDED);
}