//+------------------------------------------------------------------+
//|                                       TradingStatsExporter.mq5   |
//|                                                                  |
//+------------------------------------------------------------------+
#property copyright "Trading Stats Exporter"
#property version   "1.00"
#property script_show_inputs

input string   CSVFileName = "TradingStats"; // CSV file name (without extension)

//+------------------------------------------------------------------+
//| Structure to hold statistics                                     |
//+------------------------------------------------------------------+
struct StatData
{
   string   symbol;
   long     magic;
   double   totalPL;
   double   totalVolume;
   int      totalTrades;
   int      winTrades;
   int      lossTrades;
   double   grossProfit;
   double   grossLoss;
   double   netPosition;
   double   exposure;
};

//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
{
   Print("Starting Trading Statistics Export...");
   
   // Request entire account history
   datetime from = 0; // From the beginning
   datetime to = TimeCurrent();
   
   if(!HistorySelect(from, to))
   {
      Print("Failed to retrieve history!");
      return;
   }
   
   int totalDeals = HistoryDealsTotal();
   Print("Total deals in history: ", totalDeals);
   
   if(totalDeals == 0)
   {
      Print("No deals found in account history.");
      return;
   }
   
   // Arrays to store statistics
   StatData stats[];
   
   // Process all deals
   for(int i = 0; i < totalDeals; i++)
   {
      ulong ticket = HistoryDealGetTicket(i);
      if(ticket == 0) continue;
      
      // Get deal properties
      long magic = HistoryDealGetInteger(ticket, DEAL_MAGIC);
      string symbol = HistoryDealGetString(ticket, DEAL_SYMBOL);
      double profit = HistoryDealGetDouble(ticket, DEAL_PROFIT);
      double volume = HistoryDealGetDouble(ticket, DEAL_VOLUME);
      double swap = HistoryDealGetDouble(ticket, DEAL_SWAP);
      double commission = HistoryDealGetDouble(ticket, DEAL_COMMISSION);
      long entry = HistoryDealGetInteger(ticket, DEAL_ENTRY);
      long type = HistoryDealGetInteger(ticket, DEAL_TYPE);
      
      // Skip non-trading deals (deposits, withdrawals, etc.)
      if(type != DEAL_TYPE_BUY && type != DEAL_TYPE_SELL)
         continue;
      
      // Calculate total P&L including swap and commission
      double totalPL = profit + swap + commission;
      
      // Find or create stat entry
      int idx = FindStatIndex(stats, symbol, magic);
      if(idx == -1)
      {
         // Create new entry
         idx = ArraySize(stats);
         ArrayResize(stats, idx + 1);
         stats[idx].symbol = symbol;
         stats[idx].magic = magic;
         stats[idx].totalPL = 0;
         stats[idx].totalVolume = 0;
         stats[idx].totalTrades = 0;
         stats[idx].winTrades = 0;
         stats[idx].lossTrades = 0;
         stats[idx].grossProfit = 0;
         stats[idx].grossLoss = 0;
         stats[idx].netPosition = 0;
         stats[idx].exposure = 0;
      }
      
      // Update statistics
      stats[idx].totalPL += totalPL;
      stats[idx].totalVolume += volume;
      
      // Count trades (only on exit)
      if(entry == DEAL_ENTRY_OUT || entry == DEAL_ENTRY_INOUT || entry == DEAL_ENTRY_OUT_BY)
      {
         stats[idx].totalTrades++;
         if(totalPL > 0)
         {
            stats[idx].winTrades++;
            stats[idx].grossProfit += totalPL;
         }
         else if(totalPL < 0)
         {
            stats[idx].lossTrades++;
            stats[idx].grossLoss += totalPL;
         }
      }
      
      // Update net position (entry adds, exit subtracts)
      if(entry == DEAL_ENTRY_IN)
      {
         if(type == DEAL_TYPE_BUY)
            stats[idx].netPosition += volume;
         else
            stats[idx].netPosition -= volume;
      }
      else if(entry == DEAL_ENTRY_OUT || entry == DEAL_ENTRY_OUT_BY)
      {
         if(type == DEAL_TYPE_SELL)
            stats[idx].netPosition += volume;
         else
            stats[idx].netPosition -= volume;
      }
   }
   
   // Calculate current open positions for exposure
   CalculateOpenPositions(stats);
   
   // Export to single CSV file
   ExportToCSV(stats);
   
   Print("Export completed successfully!");
}

//+------------------------------------------------------------------+
//| Find stat index for symbol/magic combination                     |
//+------------------------------------------------------------------+
int FindStatIndex(StatData &stats[], string symbol, long magic)
{
   for(int i = 0; i < ArraySize(stats); i++)
   {
      if(stats[i].symbol == symbol && stats[i].magic == magic)
         return i;
   }
   return -1;
}

//+------------------------------------------------------------------+
//| Calculate current open positions for exposure                    |
//+------------------------------------------------------------------+
void CalculateOpenPositions(StatData &stats[])
{
   for(int i = 0; i < PositionsTotal(); i++)
   {
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0) continue;
      
      string symbol = PositionGetString(POSITION_SYMBOL);
      long magic = PositionGetInteger(POSITION_MAGIC);
      double volume = PositionGetDouble(POSITION_VOLUME);
      long type = PositionGetInteger(POSITION_TYPE);
      
      int idx = FindStatIndex(stats, symbol, magic);
      if(idx != -1)
      {
         // Calculate exposure (considering direction)
         if(type == POSITION_TYPE_BUY)
            stats[idx].exposure += volume;
         else
            stats[idx].exposure -= volume;
      }
   }
}

//+------------------------------------------------------------------+
//| Export all statistics to single CSV file                         |
//+------------------------------------------------------------------+
void ExportToCSV(StatData &stats[])
{
   string filename = CSVFileName + ".csv";
   int handle = FileOpen(filename, FILE_WRITE|FILE_CSV|FILE_ANSI, ',');
   
   if(handle == INVALID_HANDLE)
   {
      Print("Failed to create CSV file: ", filename);
      return;
   }
   
   // Write header
   FileWrite(handle, "");
   FileWrite(handle, "=== STATISTICS BY MAGIC NUMBER ===");
   FileWrite(handle, "Magic Number", "Symbol", "Total P&L", "Total Trades", 
             "Win Rate %", "Gross Profit", "Gross Loss", "Profit Factor", 
             "Net Position", "Current Exposure", "Total Volume");
   
   // Group by magic number
   long magicNumbers[];
   
   // Collect unique magic numbers
   for(int i = 0; i < ArraySize(stats); i++)
   {
      if(FindMagicIndex(magicNumbers, stats[i].magic) == -1)
      {
         int idx = ArraySize(magicNumbers);
         ArrayResize(magicNumbers, idx + 1);
         magicNumbers[idx] = stats[i].magic;
      }
   }
   
   // Write magic number statistics (aggregated across all symbols)
   for(int m = 0; m < ArraySize(magicNumbers); m++)
   {
      long magic = magicNumbers[m];
      double totalPL = 0;
      int totalTrades = 0;
      int winTrades = 0;
      double grossProfit = 0;
      double grossLoss = 0;
      double netPosition = 0;
      double exposure = 0;
      double totalVolume = 0;
      string symbols = "";
      
      // Aggregate all symbols for this magic
      for(int i = 0; i < ArraySize(stats); i++)
      {
         if(stats[i].magic == magic)
         {
            totalPL += stats[i].totalPL;
            totalTrades += stats[i].totalTrades;
            winTrades += stats[i].winTrades;
            grossProfit += stats[i].grossProfit;
            grossLoss += stats[i].grossLoss;
            netPosition += stats[i].netPosition;
            exposure += stats[i].exposure;
            totalVolume += stats[i].totalVolume;
            
            if(symbols != "") symbols += "+";
            symbols += stats[i].symbol;
         }
      }
      
      double winRate = totalTrades > 0 ? (winTrades * 100.0 / totalTrades) : 0;
      double profitFactor = grossLoss != 0 ? -grossProfit / grossLoss : 0;
      
      string magicLabel = magic == 0 ? "0 (Manual)" : IntegerToString(magic);
      
      FileWrite(handle, 
                magicLabel,
                symbols,
                DoubleToString(totalPL, 2),
                IntegerToString(totalTrades),
                DoubleToString(winRate, 2),
                DoubleToString(grossProfit, 2),
                DoubleToString(grossLoss, 2),
                DoubleToString(profitFactor, 2),
                DoubleToString(netPosition, 2),
                DoubleToString(exposure, 2),
                DoubleToString(totalVolume, 2));
   }
   
   // Write separator
   FileWrite(handle, "");
   FileWrite(handle, "=== STATISTICS BY CURRENCY ===");
   FileWrite(handle, "Currency", "Total P&L", "Total Trades", "Win Rate %", 
             "Gross Profit", "Gross Loss", "Profit Factor", "Net Position", 
             "Current Exposure", "Total Volume");
   
   // Group by currency
   string currencies[];
   double currencyPL[];
   int currencyTrades[];
   int currencyWins[];
   double currencyGrossProfit[];
   double currencyGrossLoss[];
   double currencyNetPos[];
   double currencyExposure[];
   double currencyVolume[];
   
   for(int i = 0; i < ArraySize(stats); i++)
   {
      // Extract base currency from symbol (first 3 characters typically)
      string currency = StringSubstr(stats[i].symbol, 0, 3);
      
      int idx = FindCurrencyIndex(currencies, currency);
      if(idx == -1)
      {
         idx = ArraySize(currencies);
         ArrayResize(currencies, idx + 1);
         ArrayResize(currencyPL, idx + 1);
         ArrayResize(currencyTrades, idx + 1);
         ArrayResize(currencyWins, idx + 1);
         ArrayResize(currencyGrossProfit, idx + 1);
         ArrayResize(currencyGrossLoss, idx + 1);
         ArrayResize(currencyNetPos, idx + 1);
         ArrayResize(currencyExposure, idx + 1);
         ArrayResize(currencyVolume, idx + 1);
         
         currencies[idx] = currency;
         currencyPL[idx] = 0;
         currencyTrades[idx] = 0;
         currencyWins[idx] = 0;
         currencyGrossProfit[idx] = 0;
         currencyGrossLoss[idx] = 0;
         currencyNetPos[idx] = 0;
         currencyExposure[idx] = 0;
         currencyVolume[idx] = 0;
      }
      
      currencyPL[idx] += stats[i].totalPL;
      currencyTrades[idx] += stats[i].totalTrades;
      currencyWins[idx] += stats[i].winTrades;
      currencyGrossProfit[idx] += stats[i].grossProfit;
      currencyGrossLoss[idx] += stats[i].grossLoss;
      currencyNetPos[idx] += stats[i].netPosition;
      currencyExposure[idx] += stats[i].exposure;
      currencyVolume[idx] += stats[i].totalVolume;
   }
   
   // Write currency data
   for(int i = 0; i < ArraySize(currencies); i++)
   {
      double winRate = currencyTrades[i] > 0 ? (currencyWins[i] * 100.0 / currencyTrades[i]) : 0;
      double profitFactor = currencyGrossLoss[i] != 0 ? -currencyGrossProfit[i] / currencyGrossLoss[i] : 0;
      
      FileWrite(handle, currencies[i], 
                DoubleToString(currencyPL[i], 2),
                IntegerToString(currencyTrades[i]),
                DoubleToString(winRate, 2),
                DoubleToString(currencyGrossProfit[i], 2),
                DoubleToString(currencyGrossLoss[i], 2),
                DoubleToString(profitFactor, 2),
                DoubleToString(currencyNetPos[i], 2),
                DoubleToString(currencyExposure[i], 2),
                DoubleToString(currencyVolume[i], 2));
   }
   
   FileClose(handle);
   Print("All statistics exported to: ", filename);
}

//+------------------------------------------------------------------+
//| Find magic number index in array                                 |
//+------------------------------------------------------------------+
int FindMagicIndex(long &magics[], long magic)
{
   for(int i = 0; i < ArraySize(magics); i++)
   {
      if(magics[i] == magic)
         return i;
   }
   return -1;
}

//+------------------------------------------------------------------+
//| Find currency index in array                                     |
//+------------------------------------------------------------------+
int FindCurrencyIndex(string &currencies[], string currency)
{
   for(int i = 0; i < ArraySize(currencies); i++)
   {
      if(currencies[i] == currency)
         return i;
   }
   return -1;
}
//+------------------------------------------------------------------+