//+------------------------------------------------------------------+
//|                                                    Singleton.mqh |
//|                                    2019-2020, dimitri pecheritsa |
//|                                                 792112@gmail.com |
//+------------------------------------------------------------------+
//| singleton > creational design pattern                            |
//+------------------------------------------------------------------+
//   design patterns: elements of reusable object-oriented software
//   gof > erich gamma, richard helm, ralph johnson, john vlissides
//   published in 1994
//+------------------------------------------------------------------+
//| intent > one instance of a class with a global point of access   |
//+------------------------------------------------------------------+
//| benefits > variable aspect > the sole instance of a class        |
//+------------------------------------------------------------------+
//| applicability                                                    |
//+------------------------------------------------------------------+
//   one instance of a class
//      accessible to clients from a well-known access point
//      extensible by subclassing
//         extended instance should be good without changing the code
//+------------------------------------------------------------------+
//| structure                                                        |
//+------------------------------------------------------------------+
//
//                       |       Singleton      |
//                       |----------------------|
//                       |static Instance()     |
//                       | return uniqueInstance|
//                       |SingletonOperation()  |
//                       |GetSingletonData()    |
//                       |----------------------|
//                       |static uniqueInstance |
//                       |singletonData         |
//
#include <SRC\Patterns\PatternOrganizer.mqh>
namespace Singleton
{
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Singleton
//   instance method
//      lets clients access its unique instance
//      class operation > static member function in c++
//   own unique instance may be created
  {
public:
   static Singleton* Instance(void);
   void              SingletonOperation(void);
   string            GetSingletonData(void);
protected:
                     Singleton(void);
   static Singleton* uniqueInstance;
   string            singletonData;
  };
Singleton* Singleton::uniqueInstance=NULL;
//+------------------------------------------------------------------+
//| participants > singleton                                         |
//+------------------------------------------------------------------+
Singleton::Singleton(void)
  {
   Print("singleton ",&this," created");
  }
//+------------------------------------------------------------------+
//| participants > singleton                                         |
//+------------------------------------------------------------------+
void Singleton::SingletonOperation(void) //operation
  {
   Print("running singleton operation > setting singleton data");
   singletonData="singleton data";
  }
//+------------------------------------------------------------------+
//| participants > singleton                                         |
//+------------------------------------------------------------------+
string Singleton::GetSingletonData(void) //get data
  {
   Print("reading and returning singleton data");
   return singletonData;
  }
//+------------------------------------------------------------------+
//| participants > singleton                                         |
//+------------------------------------------------------------------+
Singleton* Singleton::Instance(void) //get instance
  {
   Print("singleton instance method is running");
   if(!CheckPointer(uniqueInstance))
     {
      Print("unique instance of the singleton is empty");
      uniqueInstance=new Singleton;
      Print("singleton assigned to unique instance");
     }
   Print("unique instance contains singleton: ",uniqueInstance);
   Print("returning the unique instance ",uniqueInstance," of the singleton");
   return uniqueInstance;
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Client:public ClientExample
  {
public:
   string            Output(void);
   void              Run(void);
  };
string Client::Output(void) {return __FUNCTION__;}
//+------------------------------------------------------------------+
//| collaborations                                                   |
//+------------------------------------------------------------------+
void Client::Run(void) //clients access a singleton solely through instance()
  {
   Print("requesting singleton instance 1");
   Singleton* instance1=Singleton::Instance();
   Print("requesting singleton instance 2");
   Singleton* instance2=Singleton::Instance();
   string compareInstances=
      (instance1==instance2)?
      "instances 1 and instance 2 are same objects":
      "instances are different objects";
   Print(compareInstances);
//---
   Print("requesting singleton operation on instance 1");
   instance1.SingletonOperation();
   Print("requesting singleton data via singleton instance 2");
   string singletonData=instance2.GetSingletonData();
   Print(singletonData);
//---finish
   delete instance1;
  }
}
//+------------------------------------------------------------------+
//| output                                                           |
//+------------------------------------------------------------------+
//   Creational::Singleton::Client::Output
//   requesting singleton instance 1
//   singleton instance method is running
//   unique instance of the singleton is empty
//   singleton 25165824 created
//   singleton assigned to unique instance
//   unique instance contains singleton: 25165824
//   returning the unique instance 25165824 of the singleton
//   requesting singleton instance 2
//   singleton instance method is running
//   unique instance contains singleton: 25165824
//   returning the unique instance 25165824 of the singleton
//   instances 1 and instance 2 are same objects
//   requesting singleton operation on instance 1
//   running singleton operation > setting singleton data
//   requesting singleton data via singleton instance 2
//   reading and returning singleton data
//   singleton data
//+------------------------------------------------------------------+
//| consequences                                                     |
//+------------------------------------------------------------------+
//   controlled access to sole instance
//   reduced name space
//      better than global variables
//   operations and representation
//      can be changed
//      may be subclassed
//   variable number of instances is ok
//   static methods
//      more flexible than class operations
//      but non-virtual and non-polymorphic
//+------------------------------------------------------------------+
//| implementation                                                   |
//+------------------------------------------------------------------+
//   ensuring a unique instance > in c++
//      static member _instance > pointer to its unique instance
//      static instance()
//         lazy initialization > create and store on first access
//         singleton subclass pointer to _instance can be assigned
//      constructor is protected
//   subclassing the singleton
//      singleton is determined in instance()
//         possible singletons are hard-wired
//      instance() implementation
//         in subclass, not in parent class
//            choice of singletons at link-time is fixed
//      registry of singletons
//         register themselves by name in constructor
//            static singleton is defined in the implementation file
//            all possible singletons must be created
//               else they won't get registered
//         map between string names and singletons
//         instance() asks for the singleton by name
//         looks up the singleton
//         returns if it exists
//+------------------------------------------------------------------+
//| related patterns                                                 |
//+------------------------------------------------------------------+
//   abstract factory, builder, prototype
//      can be implemented using the singleton
//+------------------------------------------------------------------+
