//+------------------------------------------------------------------+
//|                                               FactoryMethods.mqh |
//|                                    2019-2020, dimitri pecheritsa |
//|                                                 792112@gmail.com |
//+------------------------------------------------------------------+
//| factory method > creational design pattern                       |
//+------------------------------------------------------------------+
//   design patterns: elements of reusable object-oriented software
//   gof > erich gamma, richard helm, ralph johnson, john vlissides 
//   published in 1994
//+------------------------------------------------------------------+
//| intent                                                           |
//+------------------------------------------------------------------+
//   interface for creating an object > defers instantiation to subclasses
//   subclasses decide which class to instantiate
//+------------------------------------------------------------------+
//| benefits                                                         |
//+------------------------------------------------------------------+
//   variable aspect > subclass of object that is instantiated
//   refactoring problem > creating an object by specifying a class explicitly
//   refactoring solution > create objects indirectly
//      also > abstract factory, prototype
//+------------------------------------------------------------------+
//| applicability                                                    |
//+------------------------------------------------------------------+
//   class can't anticipate the class of objects it must create
//   class wants its subclasses to specify the objects it creates
//   class delegates responsibility to one of several helper subclasses
//      localize the knowledge of which helper subclass is the delegate
//+------------------------------------------------------------------+
//| structure                                                        |
//+------------------------------------------------------------------+
//
//                                 |         Creator        |
//                                 |------------------------|
//                                 |FactoryMethod()         |
//                                 |AnOperation()           |
//                                 | ...                    |
//                                 | product=FactoryMethod()|
//        |Product|                | ...                    |
//            ^                                ^
//            |                                |
//    |ConcreteProduct|<- - - - -|      ConcreteCreator      |
//                               |---------------------------|
//                               |FactoryMethod()            |
//                               | return new ConcreteProduct|
//
#include <SRC\Patterns\PatternOrganizer.mqh>
namespace FactoryMethod
{
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
interface Product //interface of objects > created by factory method 
  {
  };
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class ConcreteProduct:public Product //implements the product interface
  {
public:
                     ConcreteProduct(void);
  };
ConcreteProduct::ConcreteProduct(void)
  {
   Print("concrete product: ",&this," created");
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Creator
//   factory method 
//      declaration > returns an object of type product
//      default implementation > returns a default concrete product
//   may call the factory method > create a product object
  {
public:
   virtual Product*  FactoryMethod(void)=0;
   void              AnOperation(void);
                    ~Creator(void);
protected:
   Product*          product;
  };
Creator::~Creator(void) {delete product;}
void Creator::AnOperation(void)
  {
   Print("creator is running its operation");
   delete product;
   product=FactoryMethod();
   Print("creator has saved the product received from a virtual factory method");
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class ConcreteCreator:public Creator
//   overrides the factory method, returns a concrete product
  {
public:
   Product*          FactoryMethod(void);
  };
Product* ConcreteCreator::FactoryMethod(void)
  {
   Print("creator is running a factory method");
   Print("concrete creator is creating and returning new concrete product");
   return new ConcreteProduct;
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Client:public ClientExample
  {
public:
   string            Output(void);
   void              Run(void);
  };
string Client::Output(void) {return __FUNCTION__;}
//+------------------------------------------------------------------+
//| collaborations                                                   |
//+------------------------------------------------------------------+
void Client::Run(void)
//   creator > determined by subclasses 
//   subclasses define factory method > returns concrete product 
  {
   Print("requesting to make a creator");
   ConcreteCreator creator;
   Print("requesting creator to run its factory method to return the product");
   Product* product=creator.FactoryMethod();
   Print("requesting creator to run its operation");
   creator.AnOperation();
//---
   delete product;
  }
}
//+------------------------------------------------------------------+
//| output                                                           |
//+------------------------------------------------------------------+
//   Creational::FactoryMethod::Client::Output
//   requesting to make a creator
//   requesting creator to run its factory method to return the product
//   creator is running a factory method
//   concrete creator is creating and returning new concrete product
//   concrete product: 16777216 created
//   requesting creator to run its operation
//   creator is running its operation
//   creator is running a factory method
//   concrete creator is creating and returning new concrete product
//   concrete product: 17825792 created
//   creator has saved the product received from a virtual factory method
//+------------------------------------------------------------------+
//| consequences                                                     |
//+------------------------------------------------------------------+
//   code only deals with the product interface
//      can work with any user-defined concrete product classes
//      but clients might have to subclass the creator class
//         to create a particular concrete product object
//   provides hooks for subclasses
//      creating objects inside a class 
//         more flexible than creating an object directly
//      the factory method 
//         not abstract
//         provides a reasonable default implementation
//   connects parallel class hierarchies when
//      class delegates some responsibilities to a separate class
//      consider graphical figures > can be manipulated interactively
//         manipulator object
//            implements the interaction
//            keeps track of any manipulation-specific state that's needed
//         figure class > provides a factory method
//            lets clients create a corresponding manipulator
//         figure subclasses > override this method to return 
//            instance of the manipulator subclass that's right for them
//+------------------------------------------------------------------+
//| implementation                                                   |
//+------------------------------------------------------------------+
//   creator
//      abstract
//      concrete
//         default implementation for the factory method
//         class of objects > parent instantiates, can be changed
//   parameterized factory methods
//      new identifiers for new kinds of products
//      associate existing identifiers with different products
//   language-specific variants and issues
//      factory methods in c++
//         always virtual functions, often pure virtual
//         lazy initialization
//            creator's constructor doesn't call factory methods
//   using templates to avoid subclassing
//      template subclass of creator > parameterized by the product class
//   naming conventions > make use of factory methods clear
//+------------------------------------------------------------------+
//| related patterns                                                 |
//+------------------------------------------------------------------+
//   abstract factory > is often implemented with factory methods
//   template methods > call factory methods
//   prototypes
//      don't require subclassing creator
//      often require an initialize operation on the product class
//         creator uses initialize() to initialize the object
//         factory method doesn't require such an operation
//+------------------------------------------------------------------+
