//+------------------------------------------------------------------+
//|                                                      Builder.mqh |
//|                                    2019-2020, dimitri pecheritsa |
//|                                                 792112@gmail.com |
//+------------------------------------------------------------------+
//| builder > creational design pattern                              |
//+------------------------------------------------------------------+
//   design patterns: elements of reusable object-oriented software
//   gof > erich gamma, richard helm, ralph johnson, john vlissides 
//   published in 1994
//+------------------------------------------------------------------+
//| intent                                                           |
//+------------------------------------------------------------------+
//   for a complex object > separate construction from representation
//   same construction process makes different representations
//+------------------------------------------------------------------+
//| benefits                                                         |
//+------------------------------------------------------------------+
//   variable aspect > the way composite object gets created
//   refactoring problem > algorithmic dependencies
//   refactoring solution > isolate algorithms that may change
//      also > iterator, strategy, template method, visitor
//+------------------------------------------------------------------+
//| applicability                                                    |
//+------------------------------------------------------------------+
//   algorithm for creating a complex object should be
//      independent of the parts
//         that make up the object
//         how they're assembled
//   construction > must allow different representations
//+------------------------------------------------------------------+
//| structure                                                        |
//+------------------------------------------------------------------+
//
//                                  builder
//  |           Director          |o--------->|  Builder  |
//  |-----------------------------|           |-----------|
//  |Construct()                  |           |BuildPart()|
//  | for all objects in structure|                 ^
//  |  builder.BuildPart()        |                 |
//                                          |ConcreteBuilder|- - ->|Product|
//                                          |---------------|
//                                          |BuildPart()    |
//                                          |GetResult()    |
//
#include <SRC\Patterns\PatternOrganizer.mqh>
namespace Builder
{
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Product
//   represents the complex object under construction
//      concrete builder
//         builds the product's internal representation
//         defines the process by which it's assembled
//   includes
//      classes that define the constituent parts
//      interfaces for assembling the parts into the final result
  {
public:
   void              Add(string);
   void              Show();
protected:
   string            parts[];
  };
//+------------------------------------------------------------------+
//| participants > product                                           |
//+------------------------------------------------------------------+
void Product::Add(string part)
  {
   int size=ArraySize(parts);
   ArrayResize(parts,size+1);
   parts[size]=part;
   Print("product added ",part," to itself");
  }
//+------------------------------------------------------------------+
//| participants > product                                           |
//+------------------------------------------------------------------+
void Product::Show(void)
  {
   Print("product is showing all parts that it is made of");
   int total=ArraySize(parts);
   for(int i=0; i<total; i++)
      Print(parts[i]);
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
interface Builder //abstract interface for creating product parts
  {
   void BuildPartA();
   void BuildPartB();
   void BuildPartC();
   Product* GetResult();
  };
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Director //constructs an object with the builder interface
  {
public:
   void              Construct();
                     Director(Builder*);
                    ~Director();
protected:
   Builder*          builder;
  };
//+------------------------------------------------------------------+
//| participants > director                                          |
//+------------------------------------------------------------------+
void Director::Director(Builder *b)
  {
   builder=b;
   Print("director created and received builder ",b);
  }
//+------------------------------------------------------------------+
//| participants > director                                          |
//+------------------------------------------------------------------+
void Director::~Director(void)
  {
   delete builder;
  }
//+------------------------------------------------------------------+
//| participants > director                                          |
//+------------------------------------------------------------------+
void Director::Construct(void)
  {
   Print("director has started the construction");
   Print("director is requesting its builder to build the product parts");
   builder.BuildPartA();
   builder.BuildPartB();
   builder.BuildPartC();
   Print("director's builder constructed the product from parts");
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class ConcreteBuilder:public Builder
//   constructs and assembles product parts
//      by implementing the builder interface
//   defines and keeps track of the representation it creates
//   provides an interface for retrieving the product
  {
public:
   void              BuildPartA();
   void              BuildPartB();
   void              BuildPartC();
   Product*          GetResult();
protected:
   Product           product;
  };
//+------------------------------------------------------------------+
//| participants > concrete builder                                  |
//+------------------------------------------------------------------+
void ConcreteBuilder::BuildPartA(void)
  {
   Print("builder is requesting the product to add part a to itself");
   product.Add("part a");
   Print("builder has made part a and added it to the product");
  }
//+------------------------------------------------------------------+
//| participants > concrete builder                                  |
//+------------------------------------------------------------------+
void ConcreteBuilder::BuildPartB(void)
  {
   Print("builder is requesting the product to add part b to itself");
   product.Add("part b");
   Print("builder has made part b and added it to the product");
  }
//+------------------------------------------------------------------+
//| participants > concrete builder                                  |
//+------------------------------------------------------------------+
void ConcreteBuilder::BuildPartC(void)
  {
   Print("builder is requesting the product to add part c to itself");
   product.Add("part c");
   Print("builder has made part c and added it to the product");
  }
//+------------------------------------------------------------------+
//| participants > concrete builder                                  |
//+------------------------------------------------------------------+
Product* ConcreteBuilder::GetResult(void)
  {
   Print("builder is returning the product");
   return &product;
  }
//+------------------------------------------------------------------+
//| participants                                                     |
//+------------------------------------------------------------------+
class Client:public ClientExample
  {
public:
   string            Output();
   void              Run();
  };
string Client::Output() {return __FUNCTION__;}
//+------------------------------------------------------------------+
//| collaborations                                                   |
//+------------------------------------------------------------------+
void Client::Run()
//   client
//      creates the director object
//      configures it with the desired builder object
//   director
//      notifies the builder whenever a part of the product should be built
//   builder > handles requests from the director, adds parts to the product
//   client > retrieves the product from the builder
//
//  aClient                           aDirector     aConcreteBuilder
//     |                                  |                |
//    | |new ConcreteBuilder                               |
//    | |- - - - - - - - - - - - - - - - -|- - - - - - - >| |
//    | |new Director(aConcreteBuilder)                    |
//    | |- - - - - - - - - - - - - - - ->| |               |
//    | |Construct()                      |                |
//    | |------------------------------->| |BuildPartA()   |
//    | |                                | |------------->| |
//    | |                                | |BuildPartB()   |
//    | |                                | |------------->| |
//    | |                                | |BuildPartC()   |
//    | |                                | |------------->| |
//    | |GetResult()                      |                |
//    | |---------------------------------|-------------->| |
//     |                                  |                |
  {
   Print("client is requesting to create a new concrete builder");
   Builder* builder=new ConcreteBuilder;
   Print("client is requesting to create a director and give him the builder");
   Director director(builder);
   Print("client is requesting the director to perform the construction");
   director.Construct();
   Print("client is requesting the builder to return the result product");
   Product* product=builder.GetResult();
   Print("client is requesting the product the describe itself");
   product.Show();
  }
}
//+------------------------------------------------------------------+
//| output                                                           |
//+------------------------------------------------------------------+
//   Creational::Builder::Client::Output
//   client is requesting to create a new concrete builder
//   client is requesting to create a director and give him the builder
//   director created and received builder 10485760
//   client is requesting the director to perform the construction
//   director has started the construction
//   director is requesting its builder to build the product parts
//   builder is requesting the product to add part a to itself
//   product added part a to itself
//   builder has made part a and added it to the product
//   builder is requesting the product to add part b to itself
//   product added part b to itself
//   builder has made part b and added it to the product
//   builder is requesting the product to add part c to itself
//   product added part c to itself
//   builder has made part c and added it to the product
//   director's builder constructed the product from parts
//   client is requesting the builder to return the result product
//   builder is returning the product
//   client is requesting the product the describe itself
//   product is showing all parts that it is made of
//   part a
//   part b
//   part c
//+------------------------------------------------------------------+
//| consequences                                                     |
//+------------------------------------------------------------------+
//   it lets you vary product's internal representation
//      define a new kind of builder
//   isolates code for construction and representation
//      concretebuilder
//         contains all the code
//            to create and assemble a product
//      code is written once
//         directors can reuse it
//            same set of parts to build product variants
//   finer control over the construction
//      step by step
//         construction of the product under the director's control
//      finished product
//         retrieved by the director from the builder
//+------------------------------------------------------------------+
//| implementation                                                   |
//+------------------------------------------------------------------+
//   operation for each component defined by builder
//      does nothing by default
//   assembly and construction interface
//      step-by-step fashion
//      general interface for all kinds of concrete builders
//         a model where
//            construction results are appended to the product
//               is usually sufficient
//         if parts constructed earlier must be accessed
//            director receives child nodes from the builder
//               passes them back to the builder to build the parent nodes
//   no abstract class for products
//      products differ greatly
//      client is in a position to know
//         concrete subclass of builder in use
//         can handle its products accordingly
//   empty methods as default in builder
//      clients override operations they're interested in
//+------------------------------------------------------------------+
//| related patterns                                                 |
//+------------------------------------------------------------------+
//   abstract factory
//      may also construct complex objects
//      emphasis is on families of simple/complex products
//      builder > focuses on constructing a complex object step by step
//      product gets returned immediately
//      builder > returns the product as a final step
//   composite > is often made by builder
//+------------------------------------------------------------------+
