//+------------------------------------------------------------------+
//|                                     C_Trade_TrailingStopLoss.mqh |
//|                                     Copyright 2017, Mark Wilson. |
//|                          https://www.mql5.com/en/users/balrog100 |
//+------------------------------------------------------------------+
#include <C_Trade.mqh>

#property copyright "Copyright 2017, Mark Wilson"
#property link      "https://www.mql5.com/en/users/balrog100"
#property strict
//+------------------------------------------------------------------+
//| Class C_Trade                                                    |
//+------------------------------------------------------------------+
class C_Trade_TrailingStopLoss : public C_Trade
  {
public:
   //Constructor/Destructor
                     C_Trade_TrailingStopLoss();
                     C_Trade_TrailingStopLoss(const string strSymbol,const E_BS eType,const double dblVolume,const double dblOpenPrice,const double dblTrailAmount,const double dblHiddenTakeProfit,const string strComment,const int intMagicNumber,const double dblLiveSpread,const bool boolDrawLines=False,const int intTimeFrame=0);
                     C_Trade_TrailingStopLoss(C_Trade &objOriginalTrade,const double dblTrailAmount,const int intTimeFrame=0);
                    ~C_Trade_TrailingStopLoss();
   //Public Accessor Functions
   double GetTrailAmount(){return this.m_dblTrailAmount;};
   int GetTimeFrame(){return this.m_intTimeFrame;};
   //Public Functions
   void              Replicate(C_Trade_TrailingStopLoss &objTrade);
   virtual string    Description();
   virtual int       OrderSend(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE);
   virtual bool      ScanWithEveryTick(const int intSlippage,const color stoploss_color=clrNONE,const color takeprofit_color=clrNONE,const StopLossCheckMethod eCheckMethod=BidAsk,const bool boolUpdateLiveTrade=False);
protected:
   double            m_dblTrailAmount;
   int               m_intTimeFrame;     //for trailing stoploss when using previous close.
private:
   //Private Functions
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
C_Trade_TrailingStopLoss::C_Trade_TrailingStopLoss()
  {
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade_TrailingStopLoss::C_Trade_TrailingStopLoss(const string strSymbol,const E_BS eType,const double dblVolume,const double dblOpenPrice,const double dblTrailAmount,const double dblHiddenTakeProfit,const string strComment,const int intMagicNumber,const double dblLiveSpread,const bool boolDrawLines=False,const int intTimeFrame=0):C_Trade(strSymbol,eType,dblVolume,dblOpenPrice,0,dblHiddenTakeProfit,strComment,intMagicNumber,dblLiveSpread,boolDrawLines)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Set the initial hidden stoploss.
   this.m_dblHiddenStopLoss=dblOpenPrice-dblTrailAmount;
   if(eType==SELL) this.m_dblHiddenStopLoss=dblOpenPrice+dblTrailAmount;

//Set the trail amount
   this.m_dblTrailAmount=dblTrailAmount;
   this.m_intTimeFrame=intTimeFrame;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade_TrailingStopLoss::C_Trade_TrailingStopLoss(C_Trade &objOriginalTrade,const double dblTrailAmount,const int intTimeFrame=0)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//We are initiating this trade with the data from objOriginalTrade and then adding the trail amount and timeframe onto the class
   C_Trade::Replicate(objOriginalTrade);

   this.m_dblTrailAmount=dblTrailAmount;
   this.m_intTimeFrame=intTimeFrame;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
C_Trade_TrailingStopLoss::~C_Trade_TrailingStopLoss()
  {
  }
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Public Functions                                                 |
//+------------------------------------------------------------------+
void C_Trade_TrailingStopLoss::Replicate(C_Trade_TrailingStopLoss &objTrade)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Call base class function to replicate data
   C_Trade::Replicate(objTrade);

//Process any class specific functions.
   this.m_dblTrailAmount=objTrade.m_dblTrailAmount;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string C_Trade_TrailingStopLoss::Description()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   string strRet=C_Trade::Description();
   strRet+=", TrailAmt: "+DoubleToString(this.m_dblTrailAmount,5);

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return strRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int C_Trade_TrailingStopLoss::OrderSend(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Call the base class OrderSend, if successful, update the trailing stoploss so that it is relative to the open price
   int intRet=C_Trade::OrderSend(dblPrice,intSlippage,arrow_color);
   if(intRet>=0)
     {
      if(this.m_eType==BUY)
        {
         this.m_dblHiddenStopLoss=this.m_dblOpenPrice-this.m_dblTrailAmount;
        }
      else
        {
         this.m_dblHiddenStopLoss=this.m_dblOpenPrice+this.m_dblTrailAmount;
        }

#ifdef D_PRINT_TRADE_UPDATES
      //OrderSend should print out description, but will miss the hidden stoploss update that uses the open price.
      Print(__FUNCTION__," ExtraUpdateInfo: "+this.Description());
#endif
     }

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return intRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade_TrailingStopLoss::ScanWithEveryTick(const int intSlippage,const color stoploss_color=clrNONE,const color takeprofit_color=clrNONE,const StopLossCheckMethod eCheckMethod=BidAsk,const bool boolUpdateLiveTrade=False)
  {
//This function should be called within the OnTick function and looks to update of the trailing stoploss.   It also checks the base class to see
//if the hidden stoploss or take profit has been breached.
//You can set it to look at the previous candle's close.   Alterntively you can use live market data, Bid/Ask High/Low to trail the stoploss.
//You can set it to update only the hidden stoploss by setting boolUpdateLiveTrades to false.
//You can also specify that it doesn't re-draw the stoploss and takeprofit lines on the chart when it updates by setting boolUpdateChart to false.

//Function should return True if the trade has been closed, not if a stoploss has been updated.   If it has been closed return it here.
   bool boolRet=C_Trade::ScanWithEveryTick(intSlippage,stoploss_color,takeprofit_color);
   if(boolRet) return boolRet;


//Assuming that there are not stoploss or takeprofit breaches, then we scan for a stoploss update.
//Depending upon the Check Method (PreviousClose, BidAsk or HighLow), then calculate the appropriate Trail Level, which is where to move the stoploss
//should it exceed the previous stoploss and assuming we are above the TrailStart level.

   double dblTrailLevel,dblSpot;
   bool boolStopLossUpdate=False;

   switch(this.m_eType)
     {
      case BUY:
         //Calculate the Trail Level based upon the check type.
         switch(eCheckMethod)
           {
            case BidAsk:
               dblTrailLevel=MarketInfo(this.m_strSymbol,MODE_BID)-this.m_dblTrailAmount;
               break;
            case HighLow:
               dblTrailLevel=MarketInfo(this.m_strSymbol,MODE_HIGH)-this.m_dblTrailAmount;
               break;
            default: //PrevClose
               dblTrailLevel=iClose(this.m_strSymbol,this.m_intTimeFrame,1)-this.m_dblTrailAmount;
               break;
           }
         //If the trail level is suitable, update the stoploss.
         if(dblTrailLevel>0 && dblTrailLevel>this.m_dblHiddenStopLoss)
           {
            dblSpot=MarketInfo(this.m_strSymbol,MODE_BID);
            boolStopLossUpdate=C_Trade::OrderModify(dblSpot,dblTrailLevel,this.m_dblHiddenTakeProfit,clrNONE,boolUpdateLiveTrade);
           }
         break;
      case SELL:
         //Calculate the Trail Level based upon the check type.
         switch(eCheckMethod)
           {
            case BidAsk:
               dblTrailLevel=MarketInfo(this.m_strSymbol,MODE_ASK)+this.m_dblTrailAmount;
               break;
            case HighLow:
               dblTrailLevel=MarketInfo(this.m_strSymbol,MODE_LOW)+this.m_dblTrailAmount;
               break;
            default: //PrevClose
               dblTrailLevel=iClose(this.m_strSymbol,this.m_intTimeFrame,1)+this.m_dblTrailAmount;
               break;
           }
         //If the trail level is suitable, update the stoploss.
         if(dblTrailLevel>0 && dblTrailLevel<this.m_dblHiddenStopLoss)
           {
            dblSpot=MarketInfo(this.m_strSymbol,MODE_ASK);
            boolStopLossUpdate=C_Trade::OrderModify(dblSpot,dblTrailLevel,this.m_dblHiddenTakeProfit,clrNONE,boolUpdateLiveTrade);
           }
         break;
     }

   return False;
  }
//+------------------------------------------------------------------+
