//+------------------------------------------------------------------+
//|                                      C_Trade_SteppedStopLoss.mqh |
//|                                     Copyright 2017, Mark Wilson. |
//|                          https://www.mql5.com/en/users/balrog100 |
//+------------------------------------------------------------------+
#include <C_Trade.mqh>

#property copyright "Copyright 2017, Mark Wilson"
#property link      "https://www.mql5.com/en/users/balrog100"
#property version   "1.00"
#property strict
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class C_Trade_SteppedStopLoss : public C_Trade
  {
public:
   //Constructor/Destructor
                     C_Trade_SteppedStopLoss();
                     C_Trade_SteppedStopLoss(const string strSymbol,const E_BS eType,const double dblVolume,const double dblOpenPrice,const double &dblHiddenStopLossSteps[],const int intStepBack,const double dblSpreadAdjust,const double dblHiddenTakeProfit,const string strComment,const int intMagicNumber,const double dblLiveSpread,const bool boolDrawLines=False,const color eStepColor=clrNONE,const int intTimeFrame=0);
                     C_Trade_SteppedStopLoss(C_Trade &objOriginalTrade,const double &dblHiddenStopLossSteps[],const int intStepBack,const double dblSpreadAdjust,const int intTimeFrame=0);
                    ~C_Trade_SteppedStopLoss();
   //Public Accessor Functions
   int GetNummberOfHiddenStopLossSteps(){return ArraySize(this.m_dblHiddenStopLossSteps);};
   double GetHiddenStopLossStep(const int intIndex=0){return this.m_dblHiddenStopLossSteps[intIndex];};
   double GetSpreadAdjust(){return this.m_dblSpreadAdjust;};
   int GetStepBack(){return this.m_intStepBack;};
   color GetStepColor(){return this.m_eStepColor;};
   int GetTimeFrame(){return this.m_intTimeFrame;};
   //Public Functions
   void              Replicate(C_Trade_SteppedStopLoss &objTrade);
   virtual string    Description();
   virtual int       OrderSend(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE);
   virtual bool      ScanWithEveryTick(const int intSlippage,const color stoploss_color=clrNONE,const color takeprofit_color=clrNONE,const StopLossCheckMethod eCheckMethod=BidAsk,const bool boolUpdateLiveTrade=False);
protected:
   //Protected Variables
   double            m_dblHiddenStopLossSteps[];
   double            m_dblSpreadAdjust;
   int               m_intStepBack;
   color             m_eStepColor;
   int               m_intTimeFrame;  //Used for stoploss tracking.
private:
   //Private Function
   bool              AddLines();
   bool              DeleteLines();
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade_SteppedStopLoss::C_Trade_SteppedStopLoss()
  {
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade_SteppedStopLoss::C_Trade_SteppedStopLoss(const string strSymbol,const E_BS eType,const double dblVolume,const double dblOpenPrice,const double &dblHiddenStopLossSteps[],const int intStepBack,const double dblSpreadAdjust,const double dblHiddenTakeProfit,const string strComment,const int intMagicNumber,const double dblLiveSpread,const bool boolDrawLines=False,const color eStepColor=clrNONE,const int intTimeFrame=0):C_Trade(strSymbol,eType,dblVolume,dblOpenPrice,dblHiddenStopLossSteps[0],dblHiddenTakeProfit,strComment,intMagicNumber,dblLiveSpread,boolDrawLines)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Previous Initiation of C_Trade, initiates the StopLoss with value dblHiddenStopLossSteps[0]

//Copy dblHiddenStopLossSteps[]
   if(ArraySize(this.m_dblHiddenStopLossSteps)>0) ArrayFree(this.m_dblHiddenStopLossSteps);
   ArrayCopy(this.m_dblHiddenStopLossSteps,dblHiddenStopLossSteps);

//If the spot goes past dblHiddenStopLossSteps[3] and intStepBack is 2 then the stoploss is set to the following:
//Buy:   dblHiddenStopLoss[1]-dblSpreadAdjust
//Sell:  dblHiddenStopLoss[1]+dblSpreadAdjust

//Set the initial hidden stoploss taking into account the SpreadAdjustment
   if(this.m_eType==BUY)
     {
      this.m_dblHiddenStopLoss=dblHiddenStopLossSteps[0]-dblSpreadAdjust;
     }
   else
     {
      this.m_dblHiddenStopLoss=dblHiddenStopLossSteps[0]+dblSpreadAdjust;
     }

   this.m_dblSpreadAdjust=dblSpreadAdjust;
   this.m_intStepBack=intStepBack;
   this.m_eStepColor=eStepColor;
   this.m_intTimeFrame=intTimeFrame;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade_SteppedStopLoss::C_Trade_SteppedStopLoss(C_Trade &objOriginalTrade,const double &dblHiddenStopLossSteps[],const int intStepBack,const double dblSpreadAdjust,const int intTimeFrame=0)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//We are initiating this trade with the data from objOriginalTrade and then adding the trail amount and timeframe onto the class
   C_Trade::Replicate(objOriginalTrade);

//Copy dblHiddenStopLossSteps[]
   if(ArraySize(this.m_dblHiddenStopLossSteps)>0) ArrayFree(this.m_dblHiddenStopLossSteps);
   ArrayCopy(this.m_dblHiddenStopLossSteps,dblHiddenStopLossSteps);

   this.m_intStepBack=intStepBack;
   this.m_dblSpreadAdjust=dblSpreadAdjust;
   this.m_intTimeFrame=intTimeFrame;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade_SteppedStopLoss::~C_Trade_SteppedStopLoss()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//This must go before the array is deleted to ensure the array size can be used for deleting lines.
   this.DeleteLines();

//Destroy dblHiddenStopLossSteps[], MUST BE LAST STEP IN DESTRUCTOR.
   if(ArraySize(this.m_dblHiddenStopLossSteps)>0) ArrayFree(this.m_dblHiddenStopLossSteps);

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

  }
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Public Functions                                                 |
//+------------------------------------------------------------------+
void C_Trade_SteppedStopLoss::Replicate(C_Trade_SteppedStopLoss &objTrade)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Call base class function to replicate data
   C_Trade::Replicate(objTrade);

//Process any class specific functions.
   ArrayFree(this.m_dblHiddenStopLossSteps);
   ArrayCopy(this.m_dblHiddenStopLossSteps,objTrade.m_dblHiddenStopLossSteps);
   this.m_dblSpreadAdjust=objTrade.m_dblSpreadAdjust;
   this.m_intStepBack=objTrade.m_intStepBack;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string C_Trade_SteppedStopLoss::Description()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   string strRet=C_Trade::Description();
   strRet+=", Steps: {";

   int intSize=ArraySize(this.m_dblHiddenStopLossSteps);
   for(int i=0;i<intSize;i++)
     {
      strRet+=DoubleToString(this.m_dblHiddenStopLossSteps[i],5)+",";
     }
   strRet+="}, SprdAdj: "+DoubleToString(this.m_dblSpreadAdjust,5)+", StpBack: "+DoubleToString(this.m_dblSpreadAdjust,5);

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return strRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int C_Trade_SteppedStopLoss::OrderSend(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Call the base class OrderSend, if successful, update the lines on the chart to shop the stoploss steps.
   int intRet=C_Trade::OrderSend(dblPrice,intSlippage,arrow_color);
   if(intRet>=0)
     {
      //If successful, draw the lines
      this.AddLines();
     }

   return intRet;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade_SteppedStopLoss::ScanWithEveryTick(const int intSlippage,const color stoploss_color=clrNONE,const color takeprofit_color=clrNONE,const StopLossCheckMethod eCheckMethod=BidAsk,const bool boolUpdateLiveTrade=False)
  {
//This function should be called within the OnTick function and looks for stoploss and takeprofit breaches, and to update the stoploss in steps.   You can
//set it to look at the previous candle's close.   Alterntively you can use live market data, Bid/Ask High/Low to trail the stoploss.
//You can set it to update only the hidden stoploss by setting boolUpdateLiveTrades to false.
//You can also specify that it doesn't re-draw the stoploss and takeprofit lines on the chart when it updates by setting boolUpdateChart to false.

//As an example, if the spot goes past dblHiddenStopLossSteps[3] and intStepBack is 2 then the stoploss is set to the following:
//Buy:   dblHiddenStopLoss[1]-dblSpreadAdjust
//Sell:  dblHiddenStopLoss[1]+dblSpreadAdjust

//Function should return True if the trade has been closed, not if a stoploss has been updated.   If it has been closed return it here.
   bool boolRet=C_Trade::ScanWithEveryTick(intSlippage,stoploss_color,takeprofit_color);
   if(boolRet) return boolRet;

//Depending upon the Check Method (PreviousClose, BidAsk or HighLow), then calculate the appropriate Trail Level, which is where to move the stoploss
//should it exceed the previous stoploss and assuming we are above the TrailStart level

   double dblSpot,dblLevel;
   bool boolStopLossUpdate=False;
   int intCount=ArraySize(this.m_dblHiddenStopLossSteps);

   switch(this.m_eType)
     {
      case BUY:
         //Calculate the Trail Level based upon the check type.
         switch(eCheckMethod)
           {
            case BidAsk:
               dblLevel=MarketInfo(this.m_strSymbol,MODE_BID);
               break;
            case HighLow:
               dblLevel=MarketInfo(this.m_strSymbol,MODE_HIGH);
               break;
            default: //PrevClose
               dblLevel=iClose(this.m_strSymbol,this.m_intTimeFrame,1);
               break;
           }
         for(int i=intCount-1;i>=this.m_intStepBack;i--)
           {
            if(dblLevel>this.m_dblHiddenStopLossSteps[i])
              {
               dblSpot=MarketInfo(this.m_strSymbol,MODE_BID);
               int intIndex=MathMax(0,i-this.m_intStepBack);
               double dblSL= this.m_dblHiddenStopLossSteps[intIndex]-this.m_dblSpreadAdjust;
               //Only update the stoploss if we are increasing the value
               if(dblSL>this.m_dblHiddenStopLoss)
                 {
                  boolStopLossUpdate=C_Trade::OrderModify(dblSpot,dblSL,this.m_dblHiddenTakeProfit,clrNONE,boolUpdateLiveTrade);
                 }
               break;
              }
           }
         break;
      case SELL:
         //Calculate the Trail Level based upon the check type.
         switch(eCheckMethod)
           {
            case BidAsk:
               dblLevel=MarketInfo(this.m_strSymbol,MODE_ASK);
               break;
            case HighLow:
               dblLevel=MarketInfo(this.m_strSymbol,MODE_LOW);
               break;
            default: //PrevClose
               dblLevel=iClose(this.m_strSymbol,this.m_intTimeFrame,1);
               break;
           }
         for(int i=intCount-1;i>=this.m_intStepBack;i--)
           {
            if(dblLevel<this.m_dblHiddenStopLossSteps[i])
              {
               dblSpot=MarketInfo(this.m_strSymbol,MODE_ASK);
               int intIndex=MathMax(0,i-this.m_intStepBack);
               double dblSL= this.m_dblHiddenStopLossSteps[intIndex]+this.m_dblSpreadAdjust;
               //Only update the stoploss if we are decreasing the value.
               if(dblSL<this.m_dblHiddenStopLoss)
                 {
                  boolStopLossUpdate=C_Trade::OrderModify(dblSpot,dblSL,this.m_dblHiddenTakeProfit,clrNONE,boolUpdateLiveTrade);
                 }
               break;
              }
           }
         break;
     }

   return False;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade_SteppedStopLoss::AddLines()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   bool boolRet=True;

//Do not draw the lines if the step color is not present
   if(this.m_eStepColor==clrNONE)   return True;

//Delete the lines
   this.DeleteLines();

//Add the hidden stoploss lines
   string strNameStart="HSSSL_"+IntegerToString(this.m_intTicket)+"_";
   int intSize=ArraySize(this.m_dblHiddenStopLossSteps);
   for(int i=0;i<intSize;i++)
     {
      string strName=strNameStart+IntegerToString(i);
      boolRet=(boolRet && CreateHorizontalLine(0,strName,0,this.m_dblHiddenStopLossSteps[i],this.m_eStepColor,STYLE_DOT,1,True));
     }

   ChartRedraw();

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return boolRet;

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade_SteppedStopLoss::DeleteLines()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   bool boolRet=True;

//Delete the StopLoss Lines
   string strNameStart="HSSSL_"+IntegerToString(this.m_intTicket)+"_";
   int intSize=ArraySize(this.m_dblHiddenStopLossSteps);

   for(int i=0;i<intSize;i++)
     {
      string strName=strNameStart+IntegerToString(i);
      boolRet=(boolRet && DeleteLine(0,strName,True));
     }

   ChartRedraw();

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return boolRet;
  }
//+------------------------------------------------------------------+
