//+------------------------------------------------------------------+
//|                                            C_Trade_Pending_4.mq4 |
//|                                     Copyright 2017, Mark Wilson. |
//|                          https://www.mql5.com/en/users/balrog100 |
//+------------------------------------------------------------------+
#include <C_Trade.mqh>
#include <C_Trade_Pending.mqh>


#property copyright "Copyright 2017, Mark Wilson."
#property link      "https://www.mql5.com/en/users/balrog100"
#property version   "1.00"
#property strict

//+------------------------------------------------------------------+
//| Inputs                                                           |
//+------------------------------------------------------------------+
input E_BS  I_BuyOrSell=BUY;                                               //Buy or Sell
input int I_HiddenStopLossInPoints=5;                                      //Stoploss (points)
input int I_LowerPendingOffsetInPoints=15;                                 //Start of Pending Range (points)
input int I_UpperPendingOffsetInPoints=35;                                 //End of Pending Range (points)
input int I_HiddenTakeProfitInPoints=60;                                   //TakeProfit (points)
input int I_LiveSpreadInPoints=40;                                         //Extra spread for live sl/tp.
input int I_ExpiryInPeriods=6;                                             //Expiry in Periods.
input double I_Volume=0.1;                                                 //Volume/Lots
input int I_Slippage=5;                                                    //Slippage
input bool I_DrawLines=True;                                               //Draw Hidden SL/TP on chart

//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
C_Trade_Pending *g_objPendingTrade;
C_Trade *g_objTrade;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+


int OnInit()
  {
//This uses a class to represent pending orders in the hidden stoploss system.   However, instead of having stop/limit orders, the system
//just uses a single C_Trade_Pending class where you enter a LowerLimit, UpperLimit and Expiry.   A zero LowerLimit or
//large upper limit could be used if something similar to a traditional limit/stop order is required.

   Print(__FILE__+" "+__FUNCTION__,"---START-------------------------------------------------");

//First, validate the input data
   if(I_LowerPendingOffsetInPoints<=I_HiddenStopLossInPoints)
     {
      Print(__FILE__+" "+__FUNCTION__," I_LowerPendingOffsetInPoints must be greater than I_HiddenStopLossInPoints");
      return(INIT_PARAMETERS_INCORRECT);
     }
   if(I_UpperPendingOffsetInPoints<=I_LowerPendingOffsetInPoints)
     {
      Print(__FILE__+" "+__FUNCTION__," I_UpperPendingOffsetInPoints must be greater than I_LowerPendingOffsetInPoints");
      return(INIT_PARAMETERS_INCORRECT);
     }
   if(I_HiddenTakeProfitInPoints<=I_UpperPendingOffsetInPoints)
     {
      Print(__FILE__+" "+__FUNCTION__," I_HiddenTakeProfitInPoints must be greater than I_UpperPendingOffsetInPoints");
      return(INIT_PARAMETERS_INCORRECT);
     }

//Get the actual live spread.
   double dblLiveSpread=I_LiveSpreadInPoints*Point();

//Get a suitable spot, sl, tp and pending range
   RefreshRates();
   double dblSpot,dblHiddenStopLoss,dblHiddenTakeProfit,dblLowerStart,dblUpperStart;
   if(I_BuyOrSell==BUY)
     {
      dblSpot=Ask;
      dblHiddenStopLoss=dblSpot+I_HiddenStopLossInPoints*Point();
      dblHiddenTakeProfit=dblSpot+I_HiddenTakeProfitInPoints*Point();
      dblLowerStart=dblSpot+I_LowerPendingOffsetInPoints*Point();
      dblUpperStart=dblSpot+I_UpperPendingOffsetInPoints*Point();
     }
   else
     {
      dblSpot=Bid;
      dblHiddenStopLoss=dblSpot-I_HiddenStopLossInPoints*Point();
      dblHiddenTakeProfit=dblSpot-I_HiddenTakeProfitInPoints*Point();
      dblUpperStart=dblSpot-I_LowerPendingOffsetInPoints*Point();
      dblLowerStart=dblSpot-I_UpperPendingOffsetInPoints*Point();
     }

//Create a new trade starting at the lower start, which we will pass into the pending order, HOWEVER, DO NOT OPEN THE TRADE.
   g_objTrade=NULL;
   g_objTrade= new C_Trade(Symbol(),I_BuyOrSell,I_Volume,dblLowerStart,dblHiddenStopLoss,dblHiddenTakeProfit,"",0,dblLiveSpread,I_DrawLines);

//Create a new pending object.
   datetime dtExpiry=TimeCurrent()+I_ExpiryInPeriods*PeriodSeconds();
   g_objPendingTrade= new C_Trade_Pending(dblLowerStart,dblUpperStart,dtExpiry);

//Print out the details
   string strPrint="New pending order created";
   Comment(strPrint);
   Print(__FILE__+" "+__FUNCTION__," ",strPrint);

   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {

//If the EA is shut down, destroy pending order.
   if(g_objPendingTrade!=NULL)
     {
      delete g_objPendingTrade;
     }

//If the EA has been shut down, destroy g_objTrade.
   if(g_objTrade!=NULL)
     {
      if(g_objTrade.TradeIsLive())
        {  //Annoying, I wanted to destroy the trade in the destructor, but it kept making MetaTrader hang.
         RefreshRates();
         double dblSpot=Bid;
         if(g_objTrade.GetType()==SELL) dblSpot=Ask;
         g_objTrade.OrderClose(dblSpot,I_Slippage,clrRed);

         //Trade Close Error Processing could be added here.
        }
      delete g_objTrade;
     }

   Print(__FILE__+" "+__FUNCTION__,"---END---------------------------------------------------");

  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {

//Check with every tick to see if we need to open the pending order
   if(g_objPendingTrade!=NULL)
     {
      if(g_objPendingTrade.ScanForTrade(g_objTrade))
        {
         //Pending order initiated, so open the trade.
         double dblSpot=Ask;
         if(g_objTrade.GetType()==SELL) dblSpot=Bid;
         if(!g_objTrade.OrderSend(dblSpot,I_Slippage))
           {
            //Trade Entry Error Processing could be added here.

            //Destroy the class
            if(g_objTrade!=NULL) delete g_objTrade;
            g_objTrade=NULL;
           }

         //Destroy the pending object - could leave open and incorporate into error processing.
         if(g_objPendingTrade!=NULL)   delete g_objPendingTrade;
         g_objPendingTrade=NULL;

        }
      else if(g_objPendingTrade.HasExpired())
        {
         //Destroy the pending object and the trade object as neither are being used.
         if(g_objPendingTrade!=NULL)   delete g_objPendingTrade;
         g_objPendingTrade=NULL;
         if(g_objTrade!=NULL) delete g_objTrade;
         g_objTrade=NULL;
        }
     }

//If the trade is live, scan for stoploss/takeprofit closure etc.
   if(g_objTrade!=NULL)
     {
      if(g_objTrade.TradeIsLive())
        {
         if(g_objTrade.ScanWithEveryTick(I_Slippage))
           {  //Ensure trade is closed.
            if(g_objTrade!=NULL) delete g_objTrade;
            g_objTrade=NULL;
           }
        }
     }

  }
//+------------------------------------------------------------------+
