//+------------------------------------------------------------------+
//|                                                  SimpleTable.mqh |
//|                        Copyright 2012, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2012, MetaQuotes Software Corp."
#property link      "http://www.mql5.com"
#property version   "1.00"
#include<ChartObjects\ChartObjectsTxtControls.mqh>
//+------------------------------------------------------------------+
//|  A class to implement a column                                   |
//+------------------------------------------------------------------+
class CColumn
  {
private:
   //--- The coordinates of the upper left corner of the first cell
   int               m_x;
   int               m_y;
   //--- The width and height of each cell
   int               m_width;
   int               m_height;
   //--- The color of the background and frame of the cell
   color             m_backcolor;
   color             m_bordercolor;
   //--- The number of cells and an array to store cells
   int               m_total;
   CChartObjectEdit  m_items[];
   //--- Type of text alignment
   ENUM_ALIGN_MODE   m_align;
   //---
   color             m_textcolor;
public:
   //--- Constructor and destructor
                     CColumn(void):m_total(0){ };
                    ~CColumn(void){};
   //--- Creating a column
   void              Create(const int X,const int Y,const int w,const int h);
   //--- Set the background and frame color
   void              BackColor(const color clr);
   void              BorderColor(const color clr);
   //--- Text alignment
   void              TextAlign(const ENUM_ALIGN_MODE align);
   //---
   void              Color(const color clr);
   //--- Add a line to the column
   void              AddItem(string name,string value);
   //--- setting value
   void              SetValue(const int index,const string svalue);
   //--- Delete all
   void              DeleteAll(void);
  };
//+------------------------------------------------------------------+
//|  Constructor                                                     |
//+------------------------------------------------------------------+
void CColumn::Create(const int X,const int Y,const int w,const int h)
  {
   m_x=X;
   m_y=Y;
   m_width=w;
   m_height=h;
  }
//+------------------------------------------------------------------+
//|  Text alignment                                                  |
//+------------------------------------------------------------------+
void CColumn::TextAlign(const ENUM_ALIGN_MODE align)
  {
//--- Change the alignment type
   m_align=align;
//--- Set alignment for already existing objects
   for(int i=0;i<m_total;i++)
     {
      m_items[i].TextAlign(m_align);
     }
  }
//+------------------------------------------------------------------+
//|  Set the text color                                              |
//+------------------------------------------------------------------+
void CColumn::Color(const color clr)
  {
   m_textcolor=clr;
//--- Change the color for existing objects
   for(int i=0;i<m_total;i++) m_items[i].Color(m_textcolor);
//---
  }
//+------------------------------------------------------------------+
//| Set the background color for the cell                            |
//+------------------------------------------------------------------+
void CColumn::BackColor(const color clr)
  {
   m_backcolor=clr;
   for(int i=0;i<m_total;i++) m_items[i].BackColor(m_backcolor);
//---
  }
//+------------------------------------------------------------------+
//| Set the frame color                                              |
//+------------------------------------------------------------------+
void CColumn::BorderColor(const color clr)
  {
   m_bordercolor=clr;
   for(int i=0;i<m_total;i++) m_items[i].BorderColor(m_bordercolor);
  }
//+------------------------------------------------------------------+
//| Add one more cell at the bottom of the column                    |
//+------------------------------------------------------------------+
void CColumn::AddItem(string name,string value)
  {
//--- Prepare an array 
   int size=ArraySize(m_items);
   int new_size;
   if((new_size=ArrayResize(m_items,size+1))!=-1) m_total=new_size;
//--- Calculate the Y coordinate for the graphical object to be created
   int curr_y=m_y+m_height*size;
//--- Create a cell
   if(m_items[size].Create(0,name,0,m_x,curr_y,m_width,m_height))
     {
      m_items[size].BorderColor(m_bordercolor);
      m_items[size].BackColor(m_backcolor);
      m_items[size].Color(m_textcolor);
      m_items[size].Description(value);
      m_items[size].TextAlign(m_align);
     }
   else  Comment(__FUNCTION__," Error",GetLastError());
//---
  }
//+------------------------------------------------------------------+
//| Update text in the cell                                          |
//+------------------------------------------------------------------+
void CColumn::SetValue(const int index,const string svalue)
  {
   if(index<m_total) m_items[index].Description(svalue);
//---  
  }
//+------------------------------------------------------------------+
//|  Deleting all objects                                            |
//+------------------------------------------------------------------+
void CColumn::DeleteAll(void)
  {
//--- If there are elements to be deleted
   if(m_total>0)
     {
      //--- Pass in the loop and remove
      for(int i=m_total-1;i>=0;i--)
        {
         string n=m_items[i].Name();
         if(!m_items[i].Delete()) Print(__FUNCTION__,"Failed to delete the object. Error ",GetLastError());
         ResetLastError();
        }
     }
//---    
  }
//+------------------------------------------------------------------+
//|  A class of a simple table with two columns                      |
//+------------------------------------------------------------------+
class CSimpleTable
  {
private:
   string            m_name;           // The name of the table
   int               m_x;              // The X coordinate if the upper left corner
   int               m_y;              // The Y coordinate if the upper left corner
   color             m_backcolor;      // Background color
   color             m_bordercolor;    // Frame color
   color             m_textcolor;      // Text color
   CColumn           m_columns[2];     // Rows in the color
   int               m_rows;           // The number of rows
public:
   //--- Constructor/destructor
                     CSimpleTable();
                    ~CSimpleTable();
   //--- initializing
   void              Create(const string name,const int X,const int Y,const int w1,const int w2,const int h);
   //--- Set the color
   void              BackColor(const color clr);
   void              BorderColor(const color clr);
   void              TextColor(const color clr);
   //--- Add a row to the table
   void              AddRow(const string left_cell,const string right_cell);
   //--- The number of rows in the table
   int               Rows(void) {return (m_rows);};
   //--- Update a cell by the number of the column and the row
   void              SetValue(const int col,const int row,const string sval);
  };
//+------------------------------------------------------------------+
//| Initialization                                                   |
//+------------------------------------------------------------------+
CSimpleTable::Create(const string name,const int X,const int Y,const int w1,const int w2,const int h)
  {
//---
   m_name=name;
   m_x=X;
   m_y=Y;
//--- Create and align the first column
   m_columns[0].Create(X,Y,w1,h);
   m_columns[0].TextAlign(ALIGN_LEFT);
//--- Create and align the second column
   m_columns[1].Create(X+w1,Y,w2,h);
   m_columns[1].TextAlign(ALIGN_RIGHT);
//---
  }
//+------------------------------------------------------------------+
//|  Constructor                                                     |
//+------------------------------------------------------------------+
CSimpleTable::CSimpleTable(): m_rows(0)
  {
//--- The names of the table cells are generated randomly, recharge the generator
   MathSrand(GetTickCount());
  }
//+------------------------------------------------------------------+
//|  Destructor                                                      |
//+------------------------------------------------------------------+
CSimpleTable::~CSimpleTable()
  {
  }
//+------------------------------------------------------------------+
//|  Set the background color                                        |
//+------------------------------------------------------------------+
void CSimpleTable::BackColor(const color clr)
  {
   m_backcolor=clr;
   m_columns[0].BackColor(m_backcolor);
   m_columns[1].BackColor(m_backcolor);
  }
//+------------------------------------------------------------------+
//|  Set the border color                                            |
//+------------------------------------------------------------------+
void CSimpleTable::BorderColor(const color clr)
  {
   m_bordercolor=clr;
   m_columns[0].BorderColor(m_bordercolor);
   m_columns[1].BorderColor(m_bordercolor);
  }
//+------------------------------------------------------------------+
//|  Set the text color                                              |
//+------------------------------------------------------------------+
void CSimpleTable::TextColor(const color clr)
  {
   m_textcolor=clr;
   m_columns[0].Color(m_textcolor);
   m_columns[1].Color(m_textcolor);
  }
//+------------------------------------------------------------------+
//|  Add a row                                                       |
//+------------------------------------------------------------------+
void CSimpleTable::AddRow(const string left_cell,const string right_cell)
  {
   string cell_name=m_name+"_c1_"+(string)MathRand();
   //PrintFormat("1. Add a cell %s",cell_name);
   m_columns[0].AddItem(cell_name,left_cell);
   cell_name=m_name+"_c2_"+(string)MathRand();
   //PrintFormat("2. Add a cell %s",cell_name);
   m_columns[1].AddItem(cell_name,right_cell);
//--- Counter of rows
   m_rows++;
  }
//+------------------------------------------------------------------+
//| Set/update a value in the cell                                   |
//+------------------------------------------------------------------+
void CSimpleTable::SetValue(const int col,const int row,const string sval)
  {
   if(col>1 || col<0) return;
   if(row>m_rows) return;
   m_columns[col].SetValue(row,sval);
  }
//+------------------------------------------------------------------+
