//+------------------------------------------------------------------+
//|                                              Moving Averages.mq5 |
//|                        Copyright 2010, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2010, MetaQuotes Software Corp."
#property link      "http://www.mql5.com"
#property version   "1.00"

#include <Trade\Trade.mqh>

input double MaximumRisk        = 2;       // Maximum Risk in percentage
input double DecreaseFactor     = 3;       // Descrease factor

input int    MovingPeriodEURUSD = 13;      // Moving Average period on EURUSD
input int    MovingShiftEURUSD  = 0;       // Moving Average shift on EURUSD

input int    MovingPeriodGBPUSD = 13;      // Moving Average period on GBPUSD
input int    MovingShiftGBPUSD  = 0;       // Moving Average shift on GBPUSD

input int    MovingPeriodUSDCHF = 13;      // Moving Average period on USDCHF
input int    MovingShiftUSDCHF  = 0;       // Moving Average shift on USDCHF

input int    MovingPeriodUSDJPY = 13;      // Moving Average period on USDJPY
input int    MovingShiftUSDJPY  = 0;       // Moving Average shift on USDJPY

//--- iMA handlers
int          ExtHandleEURUSD=0;
int          ExtHandleGBPUSD=0;
int          ExtHandleUSDCHF=0;
int          ExtHandleUSDJPY=0;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- we can trade just on demo account
   int check=CheckAccount();
   if(check<0) // trading is prohibited
      return(check);
//--- create MA on EURUSD
   int res=CreateMA(ExtHandleEURUSD,MovingPeriodEURUSD,MovingShiftEURUSD,"EURUSD");
   if(res<0) return(res);
//--- create MA on GBPUSD
   res=CreateMA(ExtHandleGBPUSD,MovingPeriodGBPUSD,MovingShiftGBPUSD,"GBPUSD");
   if(res<0) return(res);

//--- create MA on USDCHF
   res=CreateMA(ExtHandleUSDCHF,MovingPeriodUSDCHF,MovingShiftUSDCHF,"USDCHF");
   if(res<0) return(res);
//--- create MA on USDJPY
   res=CreateMA(ExtHandleUSDJPY,MovingPeriodUSDJPY,MovingShiftUSDJPY,"USDJPY");
   if(res<0) return(res);
//---
   return(0);
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//--- check for EURUSD
   if(ExtHandleEURUSD)
     {
      if(PositionSelect("EURUSD")) CheckForClose("EURUSD",ExtHandleEURUSD);
      else                         CheckForOpen("EURUSD",ExtHandleEURUSD);
     }
//--- check for GBPUSD
   if(ExtHandleGBPUSD)
     {
      if(PositionSelect("GBPUSD")) CheckForClose("GBPUSD",ExtHandleGBPUSD);
      else                         CheckForOpen("GBPUSD",ExtHandleGBPUSD);
     }
//--- check for USDCHF
   if(ExtHandleUSDCHF)
     {
      if(PositionSelect("USDCHF")) CheckForClose("USDCHF",ExtHandleUSDCHF);
      else                         CheckForOpen("USDCHF",ExtHandleUSDCHF);
     }
//--- check for USDJPY
   if(ExtHandleUSDJPY)
     {
      if(PositionSelect("USDJPY")) CheckForClose("USDJPY",ExtHandleUSDJPY);
      else                         CheckForOpen("USDJPY",ExtHandleUSDJPY);
     }
//---
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {

//---
  }
//+------------------------------------------------------------------+
//|  Cretae Moving Average for the symbol                            |
//+------------------------------------------------------------------+
int CreateMA(int &handle,const int period,const int shift,const string symbol)
  {
   if(period>5)
     {
      //--- check symbol
      if(SymbolInMarketWatch(symbol))
        {
         ExtHandleUSDJPY=iMA(symbol,_Period,period,shift,MODE_SMA,PRICE_CLOSE);
         if(ExtHandleUSDJPY==INVALID_HANDLE)
           {
            printf("Error creating MA indicator for ",symbol);
            return(-4);
           }
        }
      else
        {
         Print(__FUNCTION__,"Failed to add ",symbol);
         return(-5);
        }
     }
//--- Indicator was successfully created
   return(0);
  }
//+------------------------------------------------------------------+
//|  Check if Symbol is in MarketWatch                               |
//+------------------------------------------------------------------+
bool SymbolInMarketWatch(const string symbol)
  {
//---
   bool found=false;
//--- The number of symbols available in MarketWatch
   int total=SymbolsTotal(true);
//--- Go through all symbols
   for(int i=0;i<total;i++)
     {
      if(symbol==SymbolName(i,true))
        {
         found=true;
         break;
        }
     }
//--- If the symbol is not found in Market Watch
   if(!found)
     {
      //--- Try to add it there 
      if(!SymbolSelect(symbol,true))
        {
         //--- Unsuccessful execution
         Print(__FUNCTION__,": Failed to add ",symbol);
         return(false); // No symbol in MarketWatch
        }
     }
//--- The symbol exists in MarketWatch
   return(true);
  }
//+------------------------------------------------------------------+
//|  Trading is allowed just on demo account                         |
//+------------------------------------------------------------------+
int CheckAccount()
  {
//--- Check if autotrading is allowed
   if(!TerminalInfoInteger(TERMINAL_TRADE_ALLOWED))
     {
      Alert("Autotrading is disabled in the terminal, the EA will be removed");
      return(-1);
     }
//--- Trading on a real account is prohibited
   if(AccountInfoInteger(ACCOUNT_TRADE_MODE)==ACCOUNT_TRADE_MODE_REAL)
     {
      Alert("The EA is not allowed to trade on a real account!");
      return(-2);
     }
//--- Can we trade on the current account (prohibited when using an investor password)
   if(!AccountInfoInteger(ACCOUNT_TRADE_ALLOWED))
     {
      Alert("Trading on this account is prohibited");
      return(-3);
     }
//--- Checks have been passed successfully
   return(0);
  }
//+------------------------------------------------------------------+
//| Calculate optimal lot size                                       |
//+------------------------------------------------------------------+
double TradeSizeOptimized(string symbol)
  {
   double price=0.0;
   double margin=0.0;
//--- select lot size
   if(!SymbolInfoDouble(symbol,SYMBOL_ASK,price))               return(0.0);
   if(!OrderCalcMargin(ORDER_TYPE_BUY,symbol,1.0,price,margin)) return(0.0);
   if(margin<=0.0)                                              return(0.0);

   double lot=NormalizeDouble(AccountInfoDouble(ACCOUNT_FREEMARGIN)*MaximumRisk/100.0/margin,2);
//--- calculate number of losses orders without a break
   if(DecreaseFactor>0)
     {
      //--- select history for access
      HistorySelect(0,TimeCurrent());
      //---
      int    orders=HistoryDealsTotal();  // total history deals
      int    losses=0;                    // number of losses orders without a break

      for(int i=orders-1;i>=0;i--)
        {
         ulong ticket=HistoryDealGetTicket(i);
         if(ticket==0)
           {
            Print("HistoryDealGetTicket failed, no trade history");
            break;
           }
         //--- check symbol
         if(HistoryDealGetString(ticket,DEAL_SYMBOL)!=symbol) continue;
         //--- check profit
         double profit=HistoryDealGetDouble(ticket,DEAL_PROFIT);
         if(profit>0.0) break;
         if(profit<0.0) losses++;
        }
      //---
      if(losses>1)
         lot=NormalizeDouble(lot-lot*losses/DecreaseFactor,1);
     }
//--- normalize and check limits
   double stepvol=SymbolInfoDouble(symbol,SYMBOL_VOLUME_STEP);
   lot=stepvol*NormalizeDouble(lot/stepvol,0);

   double minvol=SymbolInfoDouble(symbol,SYMBOL_VOLUME_MIN);
   if(lot<minvol) lot=minvol;

   double maxvol=SymbolInfoDouble(symbol,SYMBOL_VOLUME_MAX);
   if(lot>maxvol) lot=maxvol;
//--- return trading volume
   return(lot);
  }
//+------------------------------------------------------------------+
//| Check for open position conditions                               |
//+------------------------------------------------------------------+
void CheckForOpen(string symbol,int handle)
  {
   MqlRates rt[2];
//--- go trading only for first ticks of new bar
   if(CopyRates(symbol,_Period,0,2,rt)!=2)
     {
      Print("CopyRates of ",symbol," failed, no history");
      return;
     }
   if(rt[1].tick_volume>1) return;
//--- get current Moving Average 
   double   ma[2];
   if(CopyBuffer(handle,0,0,2,ma)!=2)
     {
      Print("CopyBuffer from iMA failed, no data");
      return;
     }
//--- check signals
   ENUM_ORDER_TYPE signal=WRONG_VALUE;

   if(rt[0].open>ma[0] && rt[0].close<ma[0]) signal=ORDER_TYPE_SELL;    // sell conditions
   else
      if(rt[0].open<ma[0] && rt[0].close>ma[0]) signal=ORDER_TYPE_BUY;  // buy conditions
//--- additional checking
   if(signal!=WRONG_VALUE)
      if(TerminalInfoInteger(TERMINAL_TRADE_ALLOWED))
         if(Bars(symbol,_Period)>100)
           {
            CTrade trade;
            trade.PositionOpen(symbol,signal,TradeSizeOptimized(symbol),
                               SymbolInfoDouble(symbol,signal==ORDER_TYPE_SELL ? SYMBOL_BID:SYMBOL_ASK),
                               0,0);
           }
//---
  }
//+------------------------------------------------------------------+
//| Check for close position conditions                              |
//+------------------------------------------------------------------+
void CheckForClose(string symbol,int handle)
  {
   MqlRates rt[2];
//--- go trading only for first ticks of new bar
   if(CopyRates(symbol,_Period,0,2,rt)!=2)
     {
      Print("CopyRates of ",symbol," failed, no history");
      return;
     }
   if(rt[1].tick_volume>1) return;
//--- get current Moving Average 
   double   ma[1];
   if(CopyBuffer(handle,0,0,1,ma)!=1)
     {
      Print("CopyBuffer from iMA failed, no data");
      return;
     }
//--- positions already selected before
   bool signal=false;
   long type=PositionGetInteger(POSITION_TYPE);

   if(type==(long)POSITION_TYPE_BUY   && rt[0].open>ma[0] && rt[0].close<ma[0]) signal=true;
   if(type==(long)POSITION_TYPE_SELL  && rt[0].open<ma[0] && rt[0].close>ma[0]) signal=true;
//--- additional checking
   if(signal)
      if(TerminalInfoInteger(TERMINAL_TRADE_ALLOWED))
         if(Bars(symbol,_Period)>100)
           {
            CTrade trade;
            trade.PositionClose(symbol,3);
           }
//---
  }
//+------------------------------------------------------------------+
