//+------------------------------------------------------------------+
//|                                  PatternProbabilityExpertAdvisor |
//|                                Copyright 2023, Evgeniy Koshtenko |
//|                          https://www.mql5.com/en/users/koshtenko |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Evgeniy Koshtenko"
#property link      "https://www.mql5.com/en/users/koshtenko"
#property version   "1.00"
#property copyright "Copyright 2023, Evgeniy Koshtenko"
#property link      "https://www.mql5.com/en/users/koshtenko"
#property version   "2.00"

#include <Trade\Trade.mqh>
#include <Arrays\ArrayString.mqh>

//--- input parameters
input int      InpPatternLength = 5;    // Pattern Length (3-10)
input int      InpLookback     = 1000;  // Lookback Period (100-5000)
input int      InpForecastHorizon = 6;  // Forecast Horizon (1-20)
input double   InpLotSize = 0.1;        // Lot Size
input int      InpMinOccurrences = 30;  // Minimum Pattern Occurrences

//--- global variables
int            g_pattern_length;
int            g_lookback;
int            g_forecast_horizon;
string         g_patterns[];
int            g_pattern_count;
CArrayString   g_symbols;
CTrade         trade;

struct SymbolData
{
   int pattern_occurrences[];
   int pattern_successes[];
   string best_buy_pattern;
   string best_sell_pattern;
};

SymbolData g_symbol_data[];

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   //--- validate inputs
   if(InpPatternLength < 3 || InpPatternLength > 10)
   {
      Print("Invalid Pattern Length. Must be between 3 and 10.");
      return INIT_PARAMETERS_INCORRECT;
   }
   
   if(InpLookback < 100 || InpLookback > 5000)
   {
      Print("Invalid Lookback Period. Must be between 100 and 5000.");
      return INIT_PARAMETERS_INCORRECT;
   }

   if(InpForecastHorizon < 1 || InpForecastHorizon > 20)
   {
      Print("Invalid Forecast Horizon. Must be between 1 and 20.");
      return INIT_PARAMETERS_INCORRECT;
   }

   //--- set global variables
   g_pattern_length = InpPatternLength;
   g_lookback = InpLookback;
   g_forecast_horizon = InpForecastHorizon;
   
   //--- generate all possible patterns
   if(!GeneratePatterns())
   {
      Print("Failed to generate patterns.");
      return INIT_FAILED;
   }
   
   //--- initialize symbols array with 10 most liquid forex pairs
   g_symbols.Add("EURUSD");
   g_symbols.Add("USDJPY");
   g_symbols.Add("GBPUSD");
   g_symbols.Add("AUDUSD");
   g_symbols.Add("USDCAD");
   g_symbols.Add("USDCHF");
   g_symbols.Add("NZDUSD");
   g_symbols.Add("EURJPY");
   g_symbols.Add("GBPJPY");
   g_symbols.Add("EURGBP");

   //--- initialize symbol data
   ArrayResize(g_symbol_data, g_symbols.Total());
   for(int i = 0; i < g_symbols.Total(); i++)
   {
      ArrayResize(g_symbol_data[i].pattern_occurrences, g_pattern_count);
      ArrayResize(g_symbol_data[i].pattern_successes, g_pattern_count);
   }
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   if(!IsNewBar()) return;
   
   for(int i = 0; i < g_symbols.Total(); i++)
   {
      string symbol = g_symbols.At(i);
      UpdatePatternStatistics(symbol, i);
      
      string current_pattern = GetCurrentPattern(symbol);
      
      if(current_pattern == g_symbol_data[i].best_buy_pattern)
      {
         if(PositionSelect(symbol) && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
         {
            trade.PositionClose(symbol);
         }
         if(!PositionSelect(symbol))
         {
            trade.Buy(InpLotSize, symbol, 0, 0, 0, "Buy Pattern: " + current_pattern);
         }
      }
      else if(current_pattern == g_symbol_data[i].best_sell_pattern)
      {
         if(PositionSelect(symbol) && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
         {
            trade.PositionClose(symbol);
         }
         if(!PositionSelect(symbol))
         {
            trade.Sell(InpLotSize, symbol, 0, 0, 0, "Sell Pattern: " + current_pattern);
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Generate all possible patterns                                   |
//+------------------------------------------------------------------+
bool GeneratePatterns()
{
   g_pattern_count = (int)MathPow(2, g_pattern_length);
   if(!ArrayResize(g_patterns, g_pattern_count))
   {
      Print("Failed to resize g_patterns array.");
      return false;
   }
   
   for(int i = 0; i < g_pattern_count; i++)
   {
      string pattern = "";
      for(int j = 0; j < g_pattern_length; j++)
      {
         pattern += ((i >> j) & 1) ? "U" : "D";
      }
      g_patterns[i] = pattern;
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Update pattern statistics and find best patterns for a symbol    |
//+------------------------------------------------------------------+
void UpdatePatternStatistics(const string symbol, const int symbol_index)
{
   ArrayInitialize(g_symbol_data[symbol_index].pattern_occurrences, 0);
   ArrayInitialize(g_symbol_data[symbol_index].pattern_successes, 0);
   
   int total_bars = iBars(symbol, PERIOD_CURRENT);
   int start = total_bars - g_lookback;
   if(start < g_pattern_length + g_forecast_horizon) start = g_pattern_length + g_forecast_horizon;
   
   double close[];
   ArraySetAsSeries(close, true);
   CopyClose(symbol, PERIOD_CURRENT, 0, total_bars, close);
   
   string patterns[];
   ArrayResize(patterns, total_bars);
   ArraySetAsSeries(patterns, true);
   
   for(int i = 0; i < total_bars - g_pattern_length; i++)
   {
      patterns[i] = "";
      for(int j = 0; j < g_pattern_length; j++)
      {
         patterns[i] += (close[i+j] > close[i+j+1]) ? "U" : "D";
      }
   }
   
   for(int i = start; i >= g_pattern_length + g_forecast_horizon; i--)
   {
      string current_pattern = patterns[i];
      int pattern_index = ArraySearch(g_patterns, current_pattern);
      
      if(pattern_index != -1)
      {
         g_symbol_data[symbol_index].pattern_occurrences[pattern_index]++;
         if(close[i-g_forecast_horizon] > close[i])
         {
            g_symbol_data[symbol_index].pattern_successes[pattern_index]++;
         }
      }
   }
   
   double best_buy_win_rate = 0;
   double best_sell_win_rate = 0;
   
   for(int i = 0; i < g_pattern_count; i++)
   {
      if(g_symbol_data[symbol_index].pattern_occurrences[i] >= InpMinOccurrences)
      {
         double win_rate = (double)g_symbol_data[symbol_index].pattern_successes[i] / g_symbol_data[symbol_index].pattern_occurrences[i];
         if(win_rate > best_buy_win_rate)
         {
            best_buy_win_rate = win_rate;
            g_symbol_data[symbol_index].best_buy_pattern = g_patterns[i];
         }
         if((1 - win_rate) > best_sell_win_rate)
         {
            best_sell_win_rate = 1 - win_rate;
            g_symbol_data[symbol_index].best_sell_pattern = g_patterns[i];
         }
      }
   }
   
   Print(symbol, " - Best Buy Pattern: ", g_symbol_data[symbol_index].best_buy_pattern, " (Win Rate: ", DoubleToString(best_buy_win_rate * 100, 2), "%)");
   Print(symbol, " - Best Sell Pattern: ", g_symbol_data[symbol_index].best_sell_pattern, " (Win Rate: ", DoubleToString(best_sell_win_rate * 100, 2), "%)");
}

//+------------------------------------------------------------------+
//| Get current price pattern for a symbol                           |
//+------------------------------------------------------------------+
string GetCurrentPattern(const string symbol)
{
   double close[];
   ArraySetAsSeries(close, true);
   CopyClose(symbol, PERIOD_CURRENT, 0, g_pattern_length + 1, close);
   
   string pattern = "";
   for(int i = 0; i < g_pattern_length; i++)
   {
      pattern += (close[i] > close[i+1]) ? "U" : "D";
   }
   
   return pattern;
}

//+------------------------------------------------------------------+
//| Custom function to search for a string in an array               |
//+------------------------------------------------------------------+
int ArraySearch(const string &arr[], string value)
{
   for(int i = 0; i < ArraySize(arr); i++)
   {
      if(arr[i] == value) return i;
   }
   return -1;
}

//+------------------------------------------------------------------+
//| Check if it's a new bar                                          |
//+------------------------------------------------------------------+
bool IsNewBar()
{
   static datetime last_time = 0;
   datetime current_time = iTime(_Symbol, PERIOD_CURRENT, 0);
   if(current_time != last_time)
   {
      last_time = current_time;
      return true;
   }
   return false;
}