//+------------------------------------------------------------------+
//|                                                      Refresh.mqh |
//|                   Copyright 2009-2019, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+

#define BUFFER_SIZE 100

int CheckLoadHistory(string symbol, ENUM_TIMEFRAMES period, datetime start_date)
{
  datetime first_date = 0;
  datetime times[BUFFER_SIZE];
  if(symbol == NULL || symbol == "") symbol = Symbol();
  if(period == PERIOD_CURRENT) period = Period();
  // check if symbol is selected in the MarketWatch
  if(!SymbolInfoInteger(symbol, SYMBOL_SELECT))
  {
    if(GetLastError() == ERR_MARKET_UNKNOWN_SYMBOL)
    {
      return(-1);
    }
    SymbolSelect(symbol, true);
  }
  // check if data is present
  SeriesInfoInteger(symbol, period, SERIES_FIRSTDATE, first_date);
  if(first_date > 0 && first_date <= start_date)
  {
    return(1);
  }
  // don't ask for load of its own data if it is an indicator
  if(MQL5InfoInteger(MQL5_PROGRAM_TYPE) == PROGRAM_INDICATOR && Period() == period && Symbol() == symbol)
  {
    return(-4);
  }
  // second attempt
  if(SeriesInfoInteger(symbol, period, SERIES_TERMINAL_FIRSTDATE, first_date))
  {
    // there is loaded data to build timeseries
    if(first_date > 0)
    {
      // force timeseries build
      CopyTime(symbol, period, first_date + PeriodSeconds(period), 1, times);
      // check date
      if(SeriesInfoInteger(symbol, period, SERIES_FIRSTDATE, first_date))
      if(first_date > 0 && first_date <= start_date)
      {
        return(2);
      }
    }
  }
  // max bars in chart from terminal options
  int max_bars = TerminalInfoInteger(TERMINAL_MAXBARS);
  // load symbol history info
  datetime first_server_date = 0;
  while(!SeriesInfoInteger(symbol, period, SERIES_SERVER_FIRSTDATE, first_server_date) && !IsStopped())
    Sleep(250);
  // fix start date for loading
  if(first_server_date > start_date) start_date = first_server_date;
  if(first_date > 0 && first_date < first_server_date)
    Print("Warning: first server date ", first_server_date, " for ", symbol,
          " does not match to first series date ", first_date);
  // load data step by step
  int fail_cnt = 0;
  while(!IsStopped())
  {
    // wait for timeseries build
    while(!SeriesInfoInteger(symbol, period, SERIES_SYNCHRONIZED) && !IsStopped())
       Sleep(250);
    // ask for built bars
    int bars = Bars(symbol, period);
    string lastmark = "";
    if(bars > 0)
    {
      // ask for first date
      if(SeriesInfoInteger(symbol, period, SERIES_FIRSTDATE, first_date))
      {
        if(first_date > 0 && (first_date <= start_date || (first_date - start_date)/PeriodSeconds() < BUFFER_SIZE))
        {
          return(0);
        }
        if(bars >= max_bars)
        {
          return(0);
        }
      }
      lastmark = TimeToString(iTime(symbol, period, bars - 1));
    }
    else
    {
      // surprisingly can be 0 bars and synchronized!
    }
    // copying of next part forces data loading
    int copied = CopyTime(symbol, period, bars, BUFFER_SIZE, times);
    if(copied > 0)
    {
      // check for data
      if(times[0] <= start_date) return(0);
      if(bars + copied >= max_bars) return(-2);
      fail_cnt = 0;
    }
    else
    {
      if(GetLastError() == ERR_HISTORY_NOT_FOUND)
      {
        return(3);
      }
      // no more than 100 failed attempts
      fail_cnt++;
      if(fail_cnt >= 100) return(-5);
      Sleep(250);
    }
  }
  return(-3);
}

int RefreshHistory(const string symbol, datetime start)
{
  if(start == 0)
  {
    start = (datetime)SeriesInfoInteger(symbol, PERIOD_CURRENT, SERIES_SERVER_FIRSTDATE);
  }
  if(start == 0)
  {
    Print("Error reading SERIES_SERVER_FIRSTDATE ", symbol, " ", GetLastError());
    MqlTick tick;
    SymbolInfoTick(symbol, tick);
    ResetLastError();
  }
  int res = CheckLoadHistory(symbol, PERIOD_CURRENT, start);
  switch(res)
  {
    case -1 : Print("Unknown symbol ", symbol);                             break;
    case -2 : Print("Requested bars more than max bars in chart ", symbol); break;
    case -3 : Print("Program was stopped ");                                break;
    case -4 : Print("Indicator shouldn't load its own data ", symbol);      break;
    case -5 : Print("Load failed ", symbol, ", refreshing started, please wait or check settings"); break;
    case  0 : Print("Loaded OK ", symbol);                              break;
    case  1 : Print("Loaded previously ", symbol);                      break;
    case  2 : Print("Loaded previously and built ", symbol);            break;
    case  3 : Print("History not found (not loaded yet) ", symbol); break;
    default : Print("Unknown result ", res, " for ", symbol);
  }
  return(res);
}
