//+------------------------------------------------------------------+
//|                                            ChartReporterCore.mqh |
//|                                    Copyright (c) 2019, Marketeer |
//|                            https://www.mql5.com/ru/articles/5913 |
//+------------------------------------------------------------------+

class Incrementer: public TypeContainer<int>
{
  public:
    Incrementer()
    {
      v = 1;
    }
    void inc()
    {
      ++v;
    }
};

class IndexMapInc: public IndexMap
{
  public:
    void inc(const string key)
    {
      Incrementer *c = dynamic_cast<Incrementer *>(this[key]);
      if(c == NULL)
      {
        add(key, new Incrementer());
      }
      else
      {
        c.inc();
      }
    }
};


class Processor
{
  protected:
    string symbol;
    ulong timestamp;
    string prefix;
    string pressed;
    IndexMap *data;
    string realsymbol;

    void createTrend(const long dealIn, const long dealOut, const int type, const datetime time1, const double price1, const datetime time2, const double price2, const string &description)
    {
      const string name = prefix + symbol + "#" + (string)dealIn + "-" + (string)dealOut;
      ObjectCreate(0, name, OBJ_TREND, ChartWindowFind(), time1, price1, time2, price2);
      ObjectSetInteger(0, name, OBJPROP_COLOR, type == +1 ? clrBlue : clrRed);
      ObjectSetInteger(0, name, OBJPROP_STYLE, STYLE_DOT);
      ObjectSetString(0, name, OBJPROP_TEXT, description);
      
      ObjectCreate(0, name + "in", type == +1 ? OBJ_ARROW_BUY : OBJ_ARROW_SELL, ChartWindowFind(), time1, price1);
      ObjectSetString(0, name + "in", OBJPROP_TEXT, (string)time1 + " @ " + (string)(float)price1);
      //ObjectSetInteger(0, name + "in", OBJPROP_HIDDEN, false);

      ObjectCreate(0, name + "out", type == +1 ? OBJ_ARROW_SELL : OBJ_ARROW_BUY, ChartWindowFind(), time2, price2);
      ObjectSetString(0, name + "out", OBJPROP_TEXT, (string)time2 + " @ " + (string)(float)price2);
      //ObjectSetInteger(0, name + "out", OBJPROP_HIDDEN, false);
    }
    
    void createButton(const int x, const int y, const int dx, const int dy, const string text, const bool selected)
    {
      const string name = prefix + "#" + text;
      ObjectCreate(0, name, OBJ_BUTTON, ChartWindowFind(), 0, 0);
      ObjectSetInteger(0, name, OBJPROP_CORNER, CORNER_LEFT_LOWER);

      ObjectSetInteger(0, name, OBJPROP_XDISTANCE, x);
      ObjectSetInteger(0, name, OBJPROP_YDISTANCE, y + dy);
      ObjectSetInteger(0, name, OBJPROP_XSIZE, dx);
      ObjectSetInteger(0, name, OBJPROP_YSIZE, dy);
      ObjectSetInteger(0, name, OBJPROP_BORDER_TYPE, BORDER_FLAT);
      ObjectSetString(0, name, OBJPROP_FONT, "Small Fonts");
      ObjectSetInteger(0, name, OBJPROP_FONTSIZE, 6);
      ObjectSetString(0, name, OBJPROP_TEXT, text);
      ObjectSetInteger(0, name, OBJPROP_STATE, false);
      ObjectSetInteger(0, name, OBJPROP_BGCOLOR, selected ? clrGreen : clrGray);
      ObjectSetInteger(0, name, OBJPROP_BORDER_COLOR, ChartGetInteger(0, CHART_COLOR_BACKGROUND));

      if(selected) pressed = name;
    }

    void controlPanel(const IndexMap &symbols)
    {
      if(internal) return;

      static int width = 75;
      static int height = 15;
      static int margin = 5;
      for(int i = 0; i < symbols.getSize(); ++i)
      {
        createButton(margin + i * width, margin, width, height, symbols.getKey(i), symbols.getKey(i) == symbol);
      }
    }
  
  public:
    Processor()
    {
      timestamp = (ulong)TimeLocal();
      prefix = "SCR" + (string)timestamp + "_";
      ObjectsDeleteAll(0, "SCR", ChartWindowFind(), -1);
      data = NULL;
      symbol = NULL;
      realsymbol = NULL;
    }
    
    ~Processor()
    {
      ObjectsDeleteAll(0, prefix, ChartWindowFind(), -1);
      if(data != NULL) delete data;
    }
    
    virtual IndexMap *load(const string file) = 0;
    virtual int getColumnCount() = 0;
    virtual int getSymbolColumn() = 0;
    virtual datetime getStart() = 0;
    virtual bool applyInit() { return true; }
    virtual void makeTrade(IndexMap *row) = 0;
    virtual int render() = 0;

    bool attach(const string file)
    {
      if(data != NULL)
      {
        delete data;
        data = NULL;
      }
      data = load(file);
      if(data == NULL)
      {
        Print("No data acquired");
        return false;
      }
      
      IndexMapInc symbols;
      
      for(int i = 0; i < data.getSize(); ++i)
      {
        IndexMap *row = data[i];
        if(CheckPointer(row) == POINTER_INVALID || row.getSize() != getColumnCount()) return false; // something is broken

        // collect all unique symbols
        string s = row[getSymbolColumn()].get<string>();
        StringTrimLeft(s);
        if(StringLen(s) > 0) symbols.inc(s);
      }

      if(symbols.getSize() > 0)
      {
        controlPanel(symbols);
      }
      
      for(int i = 0; i < symbols.getSize(); ++i)
      {
        Print(symbols.getKey(i) + "=" + symbols[i].asString());
      }
      
      return true;
    }

    bool apply(const string _s = NULL)
    {
      ObjectsDeleteAll(0, "SCR", ChartWindowFind(), OBJ_TREND);
      ObjectsDeleteAll(0, "SCR", ChartWindowFind(), OBJ_ARROW_BUY);
      ObjectsDeleteAll(0, "SCR", ChartWindowFind(), OBJ_ARROW_SELL);

      if(_s != NULL && _s != "") symbol = _s;
      if(symbol == NULL)
      {
        Print("No symbol selected");
        return false;
      }
      
      string real = findrealsymbol();
      if(real == NULL)
      {
        Print("No suitable symbol found. Try to use Prefix/Suffix");
        return false;
      }
      
      SymbolSelect(real, true);
#ifdef CHART_REPORTER_MAIN
      ChartSetSymbolPeriod(0, real, _Period);
#endif

      if(data == NULL)
      {
        Print("No data acquired");
        return false;
      }
      
      if(!applyInit()) return false;
      
      int selected = 0;
  
      for(int i = 0; i < data.getSize(); ++i)
      {
        IndexMap *row = data[i];
        if(CheckPointer(row) == POINTER_INVALID || row.getSize() != getColumnCount()) return false; // something is broken
        
        string s = row[getSymbolColumn()].get<string>();
        StringTrimLeft(s);
        
        if(s == symbol)
        {
          selected++;
          makeTrade(row);
        }
      }
      
      pressed = prefix + "#" + symbol;
      ObjectSetInteger(0, pressed, OBJPROP_BGCOLOR, clrGreen);
      
      int trends = render();
      Print(data.getSize(), " records in total");
      Print(selected, " trades for ", symbol);
      
      string title = CHART_REPORTER_TITLE + " (" + symbol + ", " + (string)selected + " records, " + (string)trends + " trades)";
      IndicatorSetString(INDICATOR_SHORTNAME, title);
#ifdef CHART_REPORTER_MAIN
      Comment(title);
#endif
      
      ChartRedraw();
      return true;
    }

    bool isEmpty() const
    {
      return data == NULL;
    }
    
    string findrealsymbol()
    {
      double temp;
      if(!SymbolInfoDouble(symbol, SYMBOL_BID, temp) && GetLastError() == ERR_MARKET_UNKNOWN_SYMBOL)
      {
        if(Suffix != "")
        {
          int pos = StringLen(symbol) - StringLen(Suffix);
          if((pos > 0) && (StringFind(symbol, Suffix) == pos))
          {
            string real = StringSubstr(symbol, 0, pos);
            if(SymbolInfoDouble(real, SYMBOL_BID, temp))
            {
              realsymbol = real;
              return real;
            }
          }
          if(StringFind(symbol, Suffix) == -1)
          {
            string real = symbol + Suffix;
            if(SymbolInfoDouble(real, SYMBOL_BID, temp))
            {
              realsymbol = real;
              return real;
            }
          }
        }
        if(Prefix != "")
        {
          int diff = StringLen(symbol) - StringLen(Prefix);
          if((diff > 0) && (StringFind(symbol, Prefix) == 0))
          {
            string real = StringSubstr(symbol, StringLen(Prefix));
            if(SymbolInfoDouble(real, SYMBOL_BID, temp))
            {
              realsymbol = real;
              return real;
            }
          }
          if(StringFind(symbol, Prefix) == -1)
          {
            string real = Prefix + symbol;
            if(SymbolInfoDouble(real, SYMBOL_BID, temp))
            {
              realsymbol = real;
              return real;
            }
          }
        }        
        realsymbol = NULL;
        return NULL;
      }
      realsymbol = symbol;
      return symbol;
    }
    
    string _symbol() const
    {
      return symbol;
    }
    
    string _realsymbol() const
    {
      return realsymbol;
    }

    void onChartEvent(const int id, const long& lparam, const double& dparam, const string& sparam)
    {
      if(id == CHARTEVENT_OBJECT_CLICK)
      {
        int x = StringFind(sparam, "_#"); // TODO: check prefix
        if(x != -1)
        {
          string s = StringSubstr(sparam, x + 2);
          Print(s, " ", sparam, " ", pressed);
          
          ObjectSetInteger(0, sparam, OBJPROP_STATE, false);
          ObjectSetInteger(0, pressed, OBJPROP_STATE, false);
          ObjectSetInteger(0, pressed, OBJPROP_BGCOLOR, clrGray);
          pressed = "";
          
          if(apply(s)) // will set pressed = sparam; and other properties
          {
#ifdef CHART_REPORTER_SUB
            ChartSetSymbolPeriod(0, _Symbol, _Period);
#endif
          }
        }
      }
    }
};

Processor *processor = NULL;

#include <Marketeer/HTMLcolumns.mqh>

class ReportProcessor: public Processor
{
  private:
    class Deal   // if MQL5 could respect private access specifier for classes,
    {            // Trades will be unreachable from outer world, so it would be fine to have
      public:    // fields made public for direct access from Processor only
        datetime time;
        double price;
        int type;      // +1 - buy, -1 - sell
        int direction; // +1 - in, -1 - out, 0 - in/out
        double volume;
        double profit;
        long deal;
        long order;
        string comment;
        
      public:
        Deal(const IndexMap *row) // this is MT5 deal
        {
          time = StringToTime(row[COLUMN_TIME].get<string>()) + TimeShift;
          price = StringToDouble(row[COLUMN_PRICE].get<string>());
          string t = row[COLUMN_TYPE].get<string>();
          type = t == "buy" ? +1 : (t == "sell" ? -1 : 0);
          t = row[COLUMN_DIRECTION].get<string>();
          direction = 0;
          if(StringFind(t, "in") > -1) ++direction;
          if(StringFind(t, "out") > -1) --direction;
          volume = StringToDouble(row[COLUMN_VOLUME].get<string>());
          t = row[COLUMN_PROFIT].get<string>();
          StringReplace(t, " ", "");
          profit = StringToDouble(t);
          deal = StringToInteger(row[COLUMN_DEAL].get<string>());
          order = StringToInteger(row[COLUMN_ORDER].get<string>());
          comment = row[COLUMN_COMMENT].get<string>();
        }
    
        bool isIn() const
        {
          return direction >= 0;
        }
        
        bool isOut() const
        {
          return direction <= 0;
        }
        
        bool isOpposite(const Deal *t) const
        {
          return type * t.type < 0;
        }
        
        bool isActive() const
        {
          return volume > 0;
        }
    };

    RubbArray<Deal *> array;
    RubbArray<Deal *> queue;

    bool isQueueEmpty() const
    {
      for(int i = 0; i < queue.size(); ++i)
      {
        Print("?? ", queue[i].deal, " ", queue[i].type, " ", queue[i].direction, " ", queue[i].volume);
      }
      return queue.size() == 0;
    }

  public:
    ReportProcessor()
    {
    }
    
    ~ReportProcessor()
    {
      // queue should be empty in theory, but may contain artifacts or open positions
      // since queue references the same pointers as array, don't delete them here,
      // they will be deleted in array destructor;
      // BaseArray will only resize the queue down to 0
      ((BaseArray<Deal *> *)&queue).clear();
      if(data != NULL) delete data;
    }
    
    virtual IndexMap *load(const string file) override
    {
      return HTMLConverter::convertReport2Map(file, true);
    }

    virtual int getColumnCount() override
    {
      return COLUMNS_COUNT;
    }

    virtual int getSymbolColumn() override
    {
      return COLUMN_SYMBOL;
    }

    virtual datetime getStart() override
    {
      if(array.size() > 0)
      {
        Deal *deal = array[0];
        return deal.time;
      }
      return NULL;
    }

    virtual bool applyInit() override
    {
      ((BaseArray<Deal *> *)&queue).clear();
      array.clear();
      return true;
    }
    
    virtual void makeTrade(IndexMap *row) override
    {
      array << new Deal(row);
    }

    virtual int render() override
    {
      int count = 0;
      // abstract:
      // if direction <= 0
      //   collect all Trades from the queue which have direction >= 0 and opposite type
      //   if this volume is greater than collected volumes
      //     reduce volume in this Deal by the total volume of collected Trades
      //   else if collected volumes are greater than this volume
      //     reduce volume in matched Trades in a loop until all volume of this Deal is exhausted
      //   create object-lines from all affected Trades to this Deal
      //   'delete' all affected Trades with zero volume from queue
      //   if volume == 0, 'delete' this Deal (disactivate)
      // if direction >= 0 push the new Deal object to the queue
      
      for(int i = 0; i < array.size(); ++i)
      {
        Deal *current = array[i];
        
        if(!current.isActive()) continue;
        
        if(current.isOut())
        {
          // first try to find exact match
          for(int j = 0; j < queue.size(); ++j)
          {
            if(queue[j].isIn() && queue[j].isOpposite(current) && queue[j].volume == current.volume)
            {
              string description;
              StringConcatenate(description, (float)queue[j].volume, "[", queue[j].deal, "/", queue[j].order, "-", current.deal, "/", current.order, "] ", (current.profit < 0 ? "-" : ""), current.profit, " ", current.comment);
              createTrend(queue[j].deal, current.deal, queue[j].type, queue[j].time, queue[j].price, current.time, current.price, description);
              current.volume = 0;
              queue >> j; // remove from queue
              ++count;
              break;
            }
          }

          if(!current.isActive()) continue;
          
          // second try to perform partial close
          for(int j = 0; j < queue.size(); ++j)
          {
            if(queue[j].isIn() && queue[j].isOpposite(current))
            {
              string description;
              if(current.volume >= queue[j].volume)
              {
                StringConcatenate(description, (float)queue[j].volume, "[", queue[j].deal, "/", queue[j].order, "-", current.deal, "/", current.order, "] ", (current.profit < 0 ? "-" : ""), current.profit, " ", current.comment);
                createTrend(queue[j].deal, current.deal, queue[j].type, queue[j].time, queue[j].price, current.time, current.price, description);

                current.volume -= queue[j].volume;
                queue[j].volume = 0;
                ++count;
              }
              else
              {
                StringConcatenate(description, (float)current.volume, "[", queue[j].deal, "/", queue[j].order, "-", current.deal, "/", current.order, "] ", (current.profit < 0 ? "-" : ""), current.profit, " ", current.comment);
                createTrend(queue[j].deal, current.deal, queue[j].type, queue[j].time, queue[j].price, current.time, current.price, description);

                queue[j].volume -= current.volume;
                current.volume = 0;
                ++count;
                break;
              }
            }
          }
          
          // purge all inactive from queue
          for(int j = queue.size() - 1; j >= 0; --j)
          {
            if(!queue[j].isActive())
            {
              queue >> j;
            }
          }
        }
        
        if(current.isActive()) // is _still_ active
        {
          if(current.isIn())
          {
            queue << current;
          }
        }
      }
      
      if(!isQueueEmpty())
      {
        Print("Warning: not all deals are processed (probably, open positions left).");
      }
      
      return count;
    }
    
};


#include <Marketeer/CSVcolumns.mqh>

class HistoryProcessor: public Processor
{
  private:
    class Trade
    {
      public:
        datetime time1;
        double price1;
        datetime time2;
        double price2;
        int type;      // +1 - buy, -1 - sell
        double volume;
        double profit;
        string comment; // optional (doesn't exist in positions.csv)
        
      public:
        Trade(const IndexMap *row)
        {
          int add = row.getSize() == 13 ? 2 : 0;
          time1 = StringToTime(row[CSV_COLUMN_TIME1].get<string>()) + TimeShift;
          time2 = StringToTime(row[CSV_COLUMN_TIME2 + add].get<string>()) + TimeShift;
          price1 = StringToDouble(row[CSV_COLUMN_PRICE1].get<string>());
          price2 = StringToDouble(row[CSV_COLUMN_PRICE2 + add].get<string>());
          string t = row[CSV_COLUMN_TYPE].get<string>();
          StringToLower(t);
          type = t == "buy" ? +1 : (t == "sell" ? -1 : 0);
          volume = StringToDouble(row[CSV_COLUMN_VOLUME].get<string>());
          t = row[CSV_COLUMN_PROFIT + add].get<string>();
          StringReplace(t, " ", "");
          profit = StringToDouble(t);
          comment = add > 0 ? row[CSV_COLUMN_COMMENT].get<string>() : "";
        }
    };
    
    RubbArray<Trade *> array;

  public:
    virtual IndexMap *load(const string file) override
    {
      return CSVConverter::ReadCSV(file);
    }

    virtual int getColumnCount() override
    {
      if(data != NULL && data.getSize() > 0)
      {
        IndexMap *row = data[0];
        return row.getSize();
      }
      return 0;
    }

    virtual int getSymbolColumn() override
    {
      return CSV_COLUMN_SYMBOL; // CSV column number with symbol
    }

    virtual datetime getStart() override
    {
      if(array.size() > 0)
      {
        Trade *trade = array[0];
        return trade.time1;
      }
      return NULL;
    }

    virtual bool applyInit() override
    {
      array.clear();
      return true;
    }

    virtual void makeTrade(IndexMap *row) override
    {
      array << new Trade(row);
    }

    virtual int render() override
    {
      int count = 0;
      for(int i = 0; i < array.size(); ++i)
      {
        Trade *current = array[i];
        if(current.volume > 0 && current.type != 0)
        {
          string description;
          StringConcatenate(description, (float)current.volume, " ", current.profit, " ", current.comment);
          createTrend((long)current.time1, (long)current.time2, current.type, current.time1, current.price1, current.time2, current.price2, description);
          ++count;
        }
      }

      return count;
    }
};
