//+------------------------------------------------------------------+
//|                                                         Prsr.mqh |
//|                                    Copyright (c) 2019, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                            https://www.mql5.com/ru/articles/5638 |
//|                                                   rev.2020.02.13 |
//+------------------------------------------------------------------+

#include "RubbArray.mqh"

#include "FileReader.mqh"

#include "Token.mqh"

// PRINTX is debug print out - very verbose!
// define PRINTX as Print to get debug outputs
// it's recommended to comment out all PRINTX in release
#ifndef PRINTX
#define PRINTX StringFormat // dumps prints into nowhere, but consumes time & memory
#endif

#define HALT(MSG) {Print(MSG); int x = 0; int y = 1 / x;}
#ifndef CLEAR
#define CLEAR(P) if(CheckPointer(P) == POINTER_DYNAMIC) delete P
#endif


class Terminal;


class TreeNode: public Clonable<TreeNode *>
{
  protected:
    const Terminal *symbol;
    const Token *token;
    const TreeNode *children[];

  public:
    TreeNode(const Terminal *_symbol, const Token *_token)
    {
      symbol = _symbol;
      token = _token;
    }

    TreeNode *clone() override
    {
      return new TreeNode(symbol, token);
    }
    
    const Terminal *key() const
    {
      return symbol;
    }
    
    const Token *value() const
    {
      return token;
    }
    
    void bind(const TreeNode *sub)
    {
      const int n = ArraySize(children);
      ArrayResize(children, n + 1);
      children[n] = sub;
    }
    
    int size() const
    {
      return ArraySize(children);
    }
    
    const TreeNode *operator[](const uint i) const
    {
      if(i < (uint)ArraySize(children)) return children[i];
      return NULL;
    }
    
    void clear()
    {
      for(int i = 0; i < ArraySize(children); i++)
      {
        CLEAR(children[i]);
      }
    }
    
    ~TreeNode()
    {
      clear();
    }
};


class Parser;


// the interface allows us to collect some state in custom object
interface Callback
{
  void produce(const Parser *parser, const Terminal *, const int index, const Token *, const Source *context, const int level);
  void backtrack(const int index);
};


// default tree printer
class TreePrinter
{
  private:
    Parser *parser;
    bool printHidden;
    bool printDetails;
    string gap;

  public:
    TreePrinter(Parser &p): parser(&p) {}

    virtual string print(const TreeNode *n)
    {
      if(CheckPointer(n) == POINTER_INVALID) return NULL;

      bool explicit = printHidden 
                   || dynamic_cast<const NonTerminal *>(n.key()) != NULL
                   || dynamic_cast<const HiddenNonTerminal *>(n.key()) == NULL;
      gap += explicit ? "|" : " ";
      
      string result = "";
      if(CheckPointer(n) == POINTER_DYNAMIC)
      {
        if(printDetails && explicit)
        {
          Print(gap, n.key().toString(false), " ", (n.value() != NULL ? n.value().toString(parser.getSource()) : ""), " ", parser.getTokenValue(n.value()));
        }
        if(n.size() > 0)
        {
          for(int i = 0; i < n.size(); i++)
          {
            if(result != "") result += " ";
            result += print(n[i]);
          }
        }
        else
        {
          result = parser.getTokenValue(n.value());
        }
        
        if(explicit)
        {
          Print(gap, n.key().getType(), " ", n.key().getName(), (dynamic_cast<const HiddenNonTerminal *>(n.key()) == NULL ? " @ " + parser.getTokenValue(n.value()) : ""));
          // NB: alternative:
          // Print(gap, n.key().getType(), " ", n.key().getName(), " @ ", result);
          // uncomment this to print combined strings for every NonTerminal,
          // this can be very lengthy for large input texts

          if(dynamic_cast<const Rule *>(n.key()) != NULL) result += "\n";
        }
      }
      else
      {
        Print(gap, "null");
      }
      
      StringSetCharacter(gap, StringLen(gap) - 1, 0);
      return result;
    }

    void printTree(const bool hidden = false, const bool details = false)
    {
      printHidden = hidden;
      printDetails = details;
      gap = "";
      print(parser.getCST());
    }
    
};

class Parser
{
  private:
    BaseArray<Token *> *tokens; // input stream
    int cursor;                 // current token
    int maxcursor;              // maximal token so far
    BaseArray<int> states;
    const Source *source;
    Callback *callback;

    // holds current stack on every problematic point
    Stack<Stack<TreeNode *> *> errors;
    
    Stack<TreeNode *> stack;    // current stack, how the grammar unwinds

    TreeNode *tree;             // concrete syntax tree (optional result)
    bool buildTree;
    
    uint tickCount;
    
  
  public:
    Parser(BaseArray<Token *> *_tokens, const Source *text, const bool _buildTree = false)
    {
      reset(_tokens, text, _buildTree);
    }
    
    ~Parser()
    {
      CLEAR(tree);
      reset();
    }
    
    const TreeNode *getCST() const
    {
      return tree;
    }
    
    const Source *getSource() const
    {
      return source;
    }

    void reset(BaseArray<Token *> *_tokens = NULL, const Source *text = NULL, const bool _buildTree = false)
    {
      if(_tokens != NULL) tokens = _tokens;
      if(text != NULL) source = text;
      cursor = 0;
      maxcursor = 0;
      callback = NULL;
      tree = NULL;
      buildTree = _buildTree;
      errors.clear();
      tickCount = 0;
    }
    
    void printStack() const
    {
      for(int x = 0; x < stack.size(); x++)
      {
        Print("  ", x, "] ", (stack[x].key() != NULL ? stack[x].key().toString() : "NULL"), " ", (stack[x].value() != NULL ? stack[x].value().toString(source) : "NULL"));
      }
    }
    
    bool match(Terminal *p, const bool bindable = true)
    {
      if(IsStopped())
      {
        static bool cancelled = false;
        if(!cancelled)
        {
          Print("Cancelled");
          cancelled = true;
        }
        return false;
      }

#ifdef MAX_TIMEOUT
      if(tickCount == 0) tickCount = GetTickCount();

      if(GetTickCount() - tickCount > MAX_TIMEOUT && !MQLInfoInteger(MQL_DEBUG))
      {
        Print("EQ: ", p != NULL ? p.toString() : "(null)");
        
        printStack();
        
        HALT("Timeout");
      }
#endif

#ifdef CHECK_FOR_LOOP
        for(int q = 0; q < stack.size(); q++)
        {
          if(stack[q].key() == p && stack[q].value() == getToken())
          {
            Print("EQ: ", p.toString(), " ", getToken().toString(source));
            
            printStack();
            
            HALT("Re-entrance");
          }
        }
#endif

      static uint tickMonitor;
      if(GetTickCount() - tickMonitor > 1000)
      {
        tickMonitor = GetTickCount();
        Comment("Tokens:", cursor, "/", tokens.size(), ", Stack:", stack.size(), ", Memory: ", MQLInfoInteger(MQL_MEMORY_USED));
      }

      // TreeNodes are used to maintain the stack and can optionally build syntax tree
      TreeNode *node = new TreeNode(p, getToken());
      
      if(buildTree)
      {
        if(tree == NULL) tree = node;
      }
      
      stack.push(node);

      int previous = cursor;
        
      bool result = p.parse(&this, bindable);
      
      stack.pop();
      
      if(result && buildTree && bindable) // success
      {
        if(stack.size() > 0) // there is a holder to bind to
        {
          if(cursor > previous) // token was consumed
          {
            stack.top().bind(node);
          }
          else
          {
            delete node;
          }
        }
      }
      else
      {
        delete node;
      }

      return result;
    }
    
    void setCallback(Callback *p)
    {
      callback = p;
    }
    
    Token *getToken() const
    {
      if(cursor < tokens.size())
      {
        return tokens[cursor];
      }
      return NULL;
    }
    
    string getTokenValue(const Token &t) const
    {
      return t.content(source);
    }
    
    int getTokenIndex() const
    {
      return cursor;
    }
    
    bool isAtEnd()
    {
      return cursor == tokens.size() - 1 && tokens[cursor].getType() == EOF;
    }
    
    string getContextDescription(const Stack<TreeNode *> *_stack) const
    {
      string context = "";
      int count = 0;
      for(int i = _stack.size() - 1; i >= 0; i--)
      {
        const Terminal *p = _stack[i].key();
        if(dynamic_cast<const NonTerminal *>(p) != NULL)
        {
          context = p.getName() + ";" + context;
          count++;
          if(count >= 7)
          {
            context = "..." + context;
            break;
          }
        }
      }
      
      return context;
    }
    
    void printState()
    {
      Print("First ", maxcursor, " tokens read out of ", tokens.size());
      Print("Source: ", tokens[maxcursor].toString(source, 40));
      Print("Expected:");
      for(int i = 0; i < errors.size(); i++)
      {
        string context = getContextDescription(errors[i]);
        
        TreeNode *l = errors[i].top();
        
        Print(l.key().getName() + (context != "" ? (" in " + context) : ";"));
      }
    }
    
    void advance(const Terminal *p, const bool internal = false)
    {
      if(!internal)
      {
        production(p, cursor, tokens[cursor], stack.size());
      }
      if(cursor < tokens.size() - 1) cursor++;
      
      if(cursor > maxcursor)
      {
        maxcursor = cursor;
        errors.clear();
        PRINTX("* clear ", states.size());
      }
      PRINTX(":: ", tokens[cursor - 1].content(source), " -> ", tokens[cursor].content(source));
    }
    
    const Terminal *getStackTop() const
    {
      const int n = stack.size();
      for(int i = n - 1; i >= 0; i--)
      {
        if(dynamic_cast<const NonTerminal *>(stack[i].key()) != NULL) return stack[i].key();
      }
      return stack.top().key();
    }
    
    void production(const Terminal *p, const int index, const Token *t, const int level)
    {
      if(callback) callback.produce(&this, p, index, t, source, level);
    }
    
    void notify(Callback *c, const Terminal *p, const bool internal = false)
    {
      if(!internal)
      {
        production(p, cursor - 1, tokens[cursor - 1], stack.size());
      }
      if(c != NULL)
      {
        // we send previous token because notification is made after advance
        c.produce(&this, p, cursor - 1, tokens[cursor - 1], source, stack.size());
      }
    }
    
    void pushState()
    {
      states.push(cursor);
    }
    
    void popState()
    {
      int previous = cursor;
      cursor = states.pop();
      if(cursor != previous)
      {
        if(callback) callback.backtrack(cursor);
        PRINTX(":: ", tokens[cursor].content(source), " <-X ");
      }
    }
    
    int topState() const
    {
      return states.top();
    }
    
    void commitState(const Terminal *p, const bool internal = false)
    {
      int x = states.pop();
      if(!internal)
      {
        for(int i = x; i < cursor; i++)
        {
          production(p, i, tokens[i], stack.size());
        }
      }
    }

    void trackError(const Terminal *pp)
    {
      if(cursor >= maxcursor)
      {
        for(int i = 0; i < errors.size(); i++)
        {
          TreeNode *l = errors[i].top();
          if(l.key().getName() == pp.getName())
          {
            return;
          }
        }
        
        Stack<TreeNode *> *current = new Stack<TreeNode *>();

        current << &stack;
        current << new TreeNode(pp, (Token *)NULL);
        
        errors << current;
      }
    }
};


// Auxiliary class to handle indentation of printouts
class Nesting
{
  private:
    static int level;
    
  public:
    Nesting()
    {
      level++;
    }
    
    ~Nesting()
    {
      level--;
    }
    
    static int getLevel()
    {
      return level;
    }
};

static int Nesting::level = 0;


class Terminal
{
  protected:
    TokenType me;
    int mult;
    bool completeOR;
    string name;
    string value;
    
    Terminal *eq;
    BaseArray<Terminal *> next;
    
    Terminal *parent;
    
    Callback *callback;
    
    string that;
    int srcline;
    
    const Terminal *getContainer() const
    {
      return (dynamic_cast<NonTerminal *>(parent) == NULL ? &this : parent);
    }
    
    bool isInternal() const
    {
      return dynamic_cast<NonTerminal *>(parent) == NULL;
    }
  
  public:
    Terminal(const string id = NULL)
    {
      me = 0;
      mult = -1;
      eq = NULL;
      parent = NULL;
      name = id;
      callback = NULL;
      completeOR = false;
      that = StringFormat("%d", &this);
      srcline = -1;
      value = NULL;
      PRINTX("", toString());
    }
    
    Terminal(const TokenType t, const string _value = NULL)
    {
      me = t;
      mult = -1;
      eq = NULL;
      parent = NULL;
      name = NULL;
      callback = NULL;
      completeOR = false;
      that = StringFormat("%d", &this);
      srcline = -1;
      value = _value;
      PRINTX("", toString());
    }
    
    ~Terminal()
    {
      Terminal *p = dynamic_cast<HiddenNonTerminal *>(parent);
      while(CheckPointer(p) != POINTER_INVALID)
      {
        Terminal *d = p;
        if(CheckPointer(p.parent) == POINTER_DYNAMIC)
        {
          p = dynamic_cast<HiddenNonTerminal *>(p.parent);
        }
        else
        {
          p = NULL;
        }
        CLEAR(d);
      }
    }
    
    virtual string getType() const
    {
      return typename(this);
    }

    string getName() const
    {
      return StringLen(name) > 0 ? name : (me != 0 ? EnumToString(me) : (srcline != -1 ? "#" + (string)srcline : "n/a"));
    }
    
    TokenType getTokenType() const
    {
      return me;
    }
    
    int getMult() const
    {
      return mult;
    }
    
    string toString(const bool recursion = true) const
    {
      string more = "";
      if(recursion && (next.size() > 0))
      {
        more += "{";
        for(int i = 0; i < next.size(); i++)
        {
          more += next[i].toString(false) + ";";
        }
        more += "}";
      }
      return StringFormat("%s%s %d -> %d [%d] %s", (me == null ? name : EnumToString(me)), (srcline != -1 ? "#" + (string)srcline : ""), &this, eq, next.size(), more);
    }
    
    BaseArray<Terminal *> *getNext()
    {
      return &next;
    }

    void setParent(Terminal &t)
    {
      if(&t == &this)
      {
        Print("self-assignment blocked in ", toString());
      }
      else
      {
        if(parent != NULL)
        {
          if(parent != &t)
          {
            Print("Bad grammar!");
            Print("Parent overwritten in: ", toString());
            Print("Old:", parent.toString());

            Terminal *p = parent;
            do
            {
              p = p.parent;
            }
            while(p != NULL);

            Print("New: ", t.toString());
            p = &t;
            do
            {
              p = p.parent;
            }
            while(p != NULL);
            
            HALT("Will stop execution by intentional error");
          }
        }
        parent = &t;
      }
    }
    
    Terminal *getParent() const
    {
      return parent;
    }
    
    virtual Terminal *operator<<(Terminal &t)
    {
      next << &t;
      return &this;
    }
    

    // check all alternatives, choose longest
    virtual Terminal *operator&(Terminal &t)
    {
      completeOR = true;
      PRINTX("  & all ", toString());
      return operator|(t);
    }
    
    virtual Terminal *operator|(Terminal &t)
    {
      Terminal *p = &t;
      if(dynamic_cast<HiddenNonTerminalOR *>(p.parent) != NULL)
      {
        p = p.parent;
      }

      if(dynamic_cast<HiddenNonTerminalOR *>(parent) != NULL)
      {
        parent.next << p;
        p.setParent(parent);
      }
      else
      {
        if(parent != NULL)
        {
          Print("Bad OR parent: ", parent.toString(), " in ", toString());
          
          HiddenNonTerminalOR *group = new HiddenNonTerminalOR("hiddenOR", completeOR);
          
          parent.setParent(group);
          p.setParent(group);
          group << parent;
          group << p;
          return group;
        }
        else
        {
          parent = new HiddenNonTerminalOR("hiddenOR", completeOR);

          p.setParent(parent);
          parent.next << &this;
          parent.next << p;
        }
      }
      return parent;
    }

    virtual Terminal *operator+(Terminal &t)
    {
      Terminal *p = &t;
      if(dynamic_cast<HiddenNonTerminalAND *>(p.parent) != NULL)
      {
        p = p.parent;
      }

      if(dynamic_cast<HiddenNonTerminalAND *>(parent) != NULL)
      {
        parent.next << p;
        p.setParent(parent);
      }
      else
      {
        if(parent != NULL)
        {
          Print("Bad AND parent:", parent.toString(), " in ", toString());
          
          HiddenNonTerminalAND *group = new HiddenNonTerminalAND("hiddenAND");
          
          parent.setParent(group);
          p.setParent(group);
          group << parent;
          group << p;
          return group;
        }
        parent = new HiddenNonTerminalAND("hiddenAND");
        p.setParent(parent);
        parent.next << &this;
        parent.next << p;
      }
      return parent;
    }

    virtual Terminal *operator*(const int times)
    {
      mult = times;
      return &this;
    }
    
    // operator[post-action()]
    virtual Terminal *operator[](Callback *action)
    {
      callback = action;
      return &this;
    }
    
    virtual bool hasAnd()
    {
      return false;
    }

    virtual bool hasOr()
    {
      return false;
    }
    
    virtual bool parse(Parser *parser, const bool commitable = true)
    {
      string prefix;
#ifdef MAX_LEVELS
      Nesting nesting;
      int level = Nesting::getLevel();
      
      if(level > MAX_LEVELS)
      {
        HALT("Too many nested levels - looping?");
      }
      StringInit(prefix, level * 2, ' '); 
#endif
      
      bool eqResult = true;

      Token *token = parser.getToken();
      
      PRINTX(prefix, "parse ", toString(), " ", parser.getTokenValue(token), " ", token.toString(NULL));

      if(token.getType() == EOF && mult == 0) return true; // expected end of program
      
      if(eq != NULL) // redirect
      {
        eqResult = parser.match(eq, commitable);
        
        PRINTX(prefix, "result eq ", eqResult, " ", toString(), " ", eq.toString(false));
        
        bool lastResult = eqResult;
        
        // if multiple tokens expected and while next tokens are successfully consumed
        while(eqResult && eq.mult == 1 && parser.getToken() != token && parser.getToken().getType() != EOF)
        {
          token = parser.getToken();
          PRINTX(prefix, "Repeat");
          eqResult = parser.match(eq, commitable);
          PRINTX(prefix, "result eq ", eqResult, " ", toString(), " ", eq.toString(false));
        }
        
        eqResult = lastResult || (mult == 0);
        
        if(eqResult)
        {
          parser.notify(callback, getContainer(), isInternal() || !commitable);
        }
        
        return eqResult; // redirect was fulfilled
      }

      if(token.getType() == me) // token match
      {
        if(value == NULL || value == parser.getTokenValue(token))
        {
          parser.advance(getContainer(), isInternal() || !commitable);
          return true;
        }
      }
      else
      if(hasAnd()) // check AND-ed conditions
      {
        PRINTX(prefix, "AND-ing");
        parser.pushState();
        for(int i = 0; i < next.size(); i++)
        {
          if(!parser.match(next[i], commitable))
          {
            if(mult == 0)
            {
              PRINTX(prefix, "Option skipped in AND: ", toString());
              parser.popState();
              return true;
            }
            else
            {
              PRINTX(prefix, "FALSE AND:", next[i].toString());
              parser.popState();
              return false;
            }
          }
        }

        PRINTX(prefix, "TRUE AND:", toString());

        parser.commitState(getContainer(), isInternal() || !commitable);
        return true;
      }
      else
      if(hasOr())
      {
        int maxLength = -1;
        int maxIndex = -1;
        bool orAll = completeOR;

        PRINTX(prefix, "OR-ing ", orAll);
        if(!orAll) parser.pushState();
        for(int i = 0; i < next.size(); i++)
        {
          if(orAll) parser.pushState();
          if(parser.match(next[i], !orAll && commitable))
          {
            if(!orAll)
            {
              PRINTX(prefix, "TRUE OR:", toString());
              parser.commitState(getContainer(), isInternal() || !commitable);
              return true;
            }
            else
            {
              int top = parser.getTokenIndex();
              PRINTX(prefix, "TRUE ORALL:", top, " ", next[i].toString());
              if(top > maxLength)
              {
                maxLength = top;
                maxIndex = i;
              }
            }
          }
          if(orAll) parser.popState();
        }
        if(!orAll) parser.popState();
        else if(maxIndex > -1)
        {
          parser.pushState();
          if(parser.match(next[maxIndex], commitable))
          {
            PRINTX(prefix, "Commit:", maxIndex, " ", next[maxIndex].toString());
            parser.commitState(getContainer(), isInternal() || !commitable);
            return true;
          }
          else
          {
            HALT("Missing completeOR");
          }
        }
        
        if(mult == 0)
        {
          PRINTX(prefix, "Option skipped in OR: ", toString());
          return true;
        }
        PRINTX(prefix, "FALSE OR:", toString());
      }
      else
      if(mult == 0) // last chance
      {
        // parser.advance(); - don't consume token and proceed to next production
        return true;
      }

      if(dynamic_cast<HiddenNonTerminal *>(&this) == NULL)
      {
        parser.trackError(&this);
      }
      
      PRINTX(prefix, "FALSE ", toString());
      return false;
    }
};


class HiddenNonTerminal: public Terminal
{
  private:
    static List<Terminal *> dynamic; // garbage collector

  public:
    HiddenNonTerminal(const string id = NULL): Terminal(id)
    {
    }

    HiddenNonTerminal(HiddenNonTerminal &ref)
    {
      eq = &ref;
    }

    HiddenNonTerminal(HiddenNonTerminal &ref, const int lineno)
    {
      eq = &ref;
      srcline = lineno;
      PRINTX(" ", toString());
    }

    virtual HiddenNonTerminal *operator~()
    {
      HiddenNonTerminal *p = new HiddenNonTerminal(this);
      dynamic.add(p);
      return p;
    }

    virtual HiddenNonTerminal *operator^(const int lineno) // __LINE__
    {
      HiddenNonTerminal *p = new HiddenNonTerminal(this, lineno);
      dynamic.add(p);
      return p;
    }

    virtual string getType() const override
    {
      return typename(this);
    }
};

class HiddenNonTerminalOR: public HiddenNonTerminal
{
  public:
    HiddenNonTerminalOR(const string id = NULL): HiddenNonTerminal(id)
    {
    }

    HiddenNonTerminalOR(const string id = NULL, const bool all = false): HiddenNonTerminal(id)
    {
      completeOR = all;
    }
    
    virtual Terminal *operator|(Terminal &t) override
    {
      Terminal *p = &t;
      next << p;
      p.setParent(this);
      return &this;
    }

    virtual bool hasOr() override
    {
      return next.size() > 0;
    }

    virtual string getType() const override
    {
      return typename(this);
    }
};

class HiddenNonTerminalAND: public HiddenNonTerminal
{
  public:
    HiddenNonTerminalAND(const string id = NULL): HiddenNonTerminal(id)
    {
    }
    
    ~HiddenNonTerminalAND()
    {
    }

    virtual Terminal *operator+(Terminal &t) override
    {
      Terminal *p = &t;
      next << p;
      p.setParent(this);
      return &this;
    }

    virtual bool hasAnd() override
    {
      return next.size() > 0;
    }

    virtual string getType() const override
    {
      return typename(this);
    }
};

class NonTerminal: public HiddenNonTerminal
{
  public:
    NonTerminal(const string id = NULL): HiddenNonTerminal(id)
    {
    }

    virtual string getType() const override
    {
      return typename(this);
    }
    
    virtual Terminal *operator=(Terminal &t) // production
    {
      Terminal *p = &t;
      while(p.getParent() != NULL)
      {
        p = p.getParent();
        if(p == &t)
        {
          Print("Cyclic dependency in assignment: ", toString(), " <<== ", t.toString());
          p = &t;
          break;
        }
      }
    
      if(dynamic_cast<HiddenNonTerminal *>(p) != NULL)
      {
        eq = p;
      }
      else
      {
        eq = &t;
      }
      eq.setParent(this);
      return &this;
    }

};


class Rule: public NonTerminal
{
  public:
    Rule(const string id = NULL): NonTerminal(id)
    {
    }

    virtual string getType() const override
    {
      return typename(this);
    }
};

class Keywords
{
  private:
    static List<Terminal *> keywords;

  public:
    static Terminal *get(const TokenType t, const string value = NULL)
    {
      Terminal *p = new Terminal(t, value);
      keywords.add(p);
      return p;
    }
};

static List<Terminal *> Keywords::keywords;
static List<Terminal *> HiddenNonTerminal::dynamic;


#define T(X) Keywords::get(X)
#define TC(X,Y) Keywords::get(X,Y)

// debug
#define R(Y) (Y^__LINE__)

// release
//#define R(Y) (~Y)

#define _DECLARE(Cls) NonTerminal Cls(#Cls); Cls
#define DECLARE(Cls) Rule Cls(#Cls); Cls
#define _FORWARD(Cls) NonTerminal Cls(#Cls);
#define FORWARD(Cls) Rule Cls(#Cls);

