//=====================================================================
//	Setting extremums on a chart in the manual mode
//=====================================================================

//---------------------------------------------------------------------
#property copyright 	"Dima S., 2010 ."
#property link      	"dimascub@mail.com"
#property version   	"1.03"
#property description "Extremum Hand Setting"
//---------------------------------------------------------------------
#property indicator_chart_window
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Version history:
//---------------------------------------------------------------------
//	15.11.2010. - V1.00
//	 - First version;
//
//	21.11.2010. - V1.01
//	 - Changed - the method of setting the extremums - implemented setting using the price
//		 labels, as a more universal way;
//	 - Added - saving price values of extremums in the global 
//		 variables;
//	 - Changed - names of global variables when saving;
//
//	22.11.2010. - V1.02
//	 - Added - creating/editing price labels;
//
//	29.11.2010. - V1.03
//	 - Changed - names of external parameters that move the position of displaying of
//		 information;
//
//---------------------------------------------------------------------


//---------------------------------------------------------------------
//	Connected libraries:
//---------------------------------------------------------------------
#include	<TextDisplay.mqh>
#include	<ExtremumClasses.mqh>
//---------------------------------------------------------------------

//=====================================================================
//	External parameters:
//=====================================================================
input string	PrefixString = "MainChannel";
//---------------------------------------------------------------------
input color		ExtremumPointColor = Yellow;
//---------------------------------------------------------------------
input bool		ShowInfo = true;
//---------------------------------------------------------------------
input int			UpDownInfoShift = 1;
input int			LeftRightInfoShift = 1;
input color		TitlesColor = LightCyan;
input color		FieldsColor = WhiteSmoke;
//---------------------------------------------------------------------


//---------------------------------------------------------------------
string	Titles_Array[ ] = { "Extr1:", "Extr2:", "Extr3:" };
//---------------------------------------------------------------------

//---------------------------------------------------------------------
TableDisplay*		titles_display_Ptr;				// displaying information on the screen
//---------------------------------------------------------------------
TExtremumList*	PrevExtr_List;				// list of previous extremums
TExtremumList*	CurrExtr_List;				// list of current extremums
TExtremumList*	NewExtr_List;				// list of new extremums
//---------------------------------------------------------------------

//---------------------------------------------------------------------
bool			is_first_init = true;
bool			is_first_draw = true;
//---------------------------------------------------------------------
#define		WIDTH			128
#define		HEIGHT		128
#define		FONTSIZE	10
//---------------------------------------------------------------------
//	Handle of the initialization event:
//---------------------------------------------------------------------
int
OnInit( )
{
	if( is_first_init != true )
	{
		DeleteGraphObjects( );
	}

	PrevExtr_List = new TExtremumList( );
	PrevExtr_List.SetChannalParams( PrefixString, Symbol( ), Period( ));
	PrevExtr_List.LoadExtremumList( );

//	If three extremums are loaded, move the price labels on them:
	if( PrevExtr_List.Total( ) == 3 )
	{
		DeleteAllArrows( 0 );
		for( int k = 0; k < 3; k++ )
		{
			CreateArrow( 0, PrevExtr_List.GetDateTime( k ), PrevExtr_List.GetPrice( k ), k + 1 );
		}
	}

	CurrExtr_List = PrevExtr_List;

	NewExtr_List = new TExtremumList( );
	NewExtr_List.SetChannalParams( PrefixString, Symbol( ), Period( ));

	InitGraphObjects( );
	is_first_init = false;

	RefreshInfo( );
	EventSetTimer( 1 );

	ChartRedraw( 0 );

	return( 0 );
}

//---------------------------------------------------------------------
//	Handle of the event of coming of new tick by the current symbol:
//---------------------------------------------------------------------
int
OnCalculate( const int rates_total, const int prev_calculated, const int begin, const double& price[ ] )
{
	return( rates_total );
}

//---------------------------------------------------------------------
//	Handler of timer events:
//---------------------------------------------------------------------
void
OnTimer( )
{
	if( CheakExtremumMoving( ) == true )
	{
		RefreshInfo( );
		ChartRedraw( 0 );
	}
}

//---------------------------------------------------------------------
//	Handle of deinitialization event:
//---------------------------------------------------------------------
void
OnDeinit( const int _reason )
{
	EventKillTimer( );
	DeleteGraphObjects( );
}

//---------------------------------------------------------------------
//	Initialization of graphical objects:
//---------------------------------------------------------------------
void
InitGraphObjects( )
{
//Print( "Creating..." );

	if( ShowInfo == true )
	{
		titles_display_Ptr = new TableDisplay( );
		titles_display_Ptr.SetParams( 0, 0, CORNER_LEFT_UPPER );
		for( int k = 0; k < 3; k++ )
		{
			titles_display_Ptr.AddFieldObject( WIDTH, HEIGHT, LeftRightInfoShift + 16, UpDownInfoShift + k * 2 + 8, FieldsColor, "Arial", FONTSIZE );
			titles_display_Ptr.SetAnchor( k, ANCHOR_RIGHT );
		}

//	Field headers:
		titles_display_Ptr.AddTitleObject( WIDTH, HEIGHT, LeftRightInfoShift + 16, UpDownInfoShift + 5, PrefixString, TitlesColor, "Arial", FONTSIZE );
		titles_display_Ptr.SetAnchor( 3, ANCHOR_RIGHT );
		for( int k = 0; k < 3; k++ )
		{
			titles_display_Ptr.AddTitleObject( WIDTH, HEIGHT, LeftRightInfoShift + 5, UpDownInfoShift + k * 2 + 8, Titles_Array[ k ], TitlesColor, "Arial", FONTSIZE );
			titles_display_Ptr.SetAnchor( k + 4, ANCHOR_RIGHT );
		}
	}
}

//---------------------------------------------------------------------
//	Deleting graphical objects:
//---------------------------------------------------------------------
void
DeleteGraphObjects( )
{
//Print( "Deleting..." );
	DeleteAllArrows( 0 );

	if( CheckPointer( PrevExtr_List ) == POINTER_DYNAMIC )
	{
		PrevExtr_List.Clear( );
		delete( PrevExtr_List );
	}
	if( CheckPointer( NewExtr_List ) == POINTER_DYNAMIC )
	{
		NewExtr_List.Clear( );
		delete( NewExtr_List );
	}
	if( ShowInfo == true )
	{
		if( CheckPointer( titles_display_Ptr ) == POINTER_DYNAMIC )
		{
			delete( titles_display_Ptr );
		}
	}
}

//---------------------------------------------------------------------
//	Refreshing dynamical information:
//---------------------------------------------------------------------
void
RefreshInfo( )
{
	if( ShowInfo == true )
	{
		string		dt_str;
		datetime	dt_int;

		int				count = CurrExtr_List.Total( );
		for( int i = 0; i < count; i++ )
		{
			dt_int = CurrExtr_List.GetDateTime( i );
			dt_str = TimeToString( dt_int );
			titles_display_Ptr.SetText( i, dt_str );
		}
	}
}

//---------------------------------------------------------------------
//	Checking the accordance of parameters of a graphical object to the required ones:
//---------------------------------------------------------------------
bool
IsGraphicObjectGood( long _chart_id, string _name, ENUM_OBJECT _type, color _color, int _width = -1, ENUM_LINE_STYLE _style = -1 )
{
//	Checking object type:
	
	if((( ENUM_OBJECT )ObjectGetInteger( _chart_id, _name, OBJPROP_TYPE )) != _type )
	{
		return( false );
	}

//	Checking object color:
	if((( color )ObjectGetInteger( _chart_id, _name, OBJPROP_COLOR )) != _color )
	{
		return( false );
	}

//	Checking line width:
	if( _width != -1 )
	{
		if((( int )ObjectGetInteger( _chart_id, _name, OBJPROP_WIDTH )) != _width )
		{
			return( false );
		}
	}

//	Checking line style:
	if( _style != -1 )
	{
		if((( ENUM_LINE_STYLE )ObjectGetInteger( _chart_id, _name, OBJPROP_STYLE )) != _style )
		{
			return( false );
		}
	}

	return( true );
}

//---------------------------------------------------------------------
//	Checking whether extremums points are found on the screen:
//---------------------------------------------------------------------
//	- should be three point exactly;
//	- the points must be marked with the left price labels;
//---------------------------------------------------------------------
bool
FindExtremumPoints( long _chart_id )
{
	string	name;

//	1. Calculate the total number of objects with specified parameters and write them to the list:
	int		total_objects = ObjectsTotal( _chart_id, -1, OBJ_ARROW_LEFT_PRICE );
	if( total_objects <= 0 )
	{
		return( false );
	}

	NewExtr_List.Clear( );
	for( int i = 0; i < total_objects; i++ )
	{
		name = ObjectName( _chart_id, i, -1, OBJ_ARROW_LEFT_PRICE );

		if( IsGraphicObjectGood( _chart_id, name, OBJ_ARROW_LEFT_PRICE, ExtremumPointColor ) == true )
		{
			NewExtr_List.AddExtremum( ObjectGetInteger( _chart_id, name, OBJPROP_TIME ), NormalizeDouble( ObjectGetDouble( _chart_id, name, OBJPROP_PRICE ), Digits( )));
		}
	}

//	2. If there are exactly three extremums found, we can try to draw a channel:
	if( NewExtr_List.Total( ) == 3 )
	{

//	Save the list of new extremums:
		NewExtr_List.SaveExtremumList( );
		return( true );
	}

	NewExtr_List.Clear( );
	return( false );
}

//---------------------------------------------------------------------
//	Check whether the extremums have been moved on the screen:
//---------------------------------------------------------------------
bool
CheakExtremumMoving( )
{
	if( FindExtremumPoints( 0 ) == true )
	{
		int		count = NewExtr_List.Total( );
		int		index;
		for( int i = 0; i < count; i++ )
		{
		 	index = CurrExtr_List.FindExtremum( NewExtr_List.GetDateTime( i ));

//	If a new extremum is found:
		 	if( index == -1 )
		 	{
		 		PrevExtr_List = CurrExtr_List;
		 		CurrExtr_List = NewExtr_List;
				return( true );
		 	}
		}
 		CurrExtr_List = PrevExtr_List;
	}

	return( false );
}

//---------------------------------------------------------------------
//	Delete all price labels:
//---------------------------------------------------------------------
void
DeleteAllArrows( long _chart_id )
{
	string	name;

//	1. Calculate the total number of objects with specified parameters and write them to the list:
	int		total_objects = ObjectsTotal( _chart_id, -1, OBJ_ARROW_LEFT_PRICE );
	if( total_objects <= 0 )																					// no labels
	{
		return;
	}

	for( int i = ( total_objects - 1 ); i >= 0; i-- )
	{
		name = ObjectName( _chart_id, i, -1, OBJ_ARROW_LEFT_PRICE );

		if( IsGraphicObjectGood( _chart_id, name, OBJ_ARROW_LEFT_PRICE, ExtremumPointColor ) == true )
		{
			ObjectDelete( _chart_id, name );
		}
	}
}

//---------------------------------------------------------------------
//	Creating a price label:
//---------------------------------------------------------------------
void
CreateArrow( long _chart_id, datetime _dt, double _price, int _nmb )
{
	string	name = PrefixString + "_" + Symbol( ) + "_" + IntegerToString( Period( )) + "_Arrow" + IntegerToString( _nmb );
	ObjectCreate( _chart_id, name, OBJ_ARROW_LEFT_PRICE, 0, _dt, _price );
	ObjectSetInteger( _chart_id, name, OBJPROP_COLOR, ExtremumPointColor );
	ObjectSetInteger( _chart_id, name, OBJPROP_WIDTH, 2 );
	ObjectSetInteger( _chart_id, name, OBJPROP_SELECTABLE, true );
	ObjectSetInteger( _chart_id, name, OBJPROP_SELECTED, true );
}
//---------------------------------------------------------------------
