//=====================================================================
//	Working with extremum on a chart:
//=====================================================================

//---------------------------------------------------------------------
#property copyright 	"Dima S., 2010 ."
#property link      	"dimascub@mail.com"
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Version history:
//---------------------------------------------------------------------
//	08.11.2010. - V1.00
//	 - First version;
//
//	16.11.2010. - V1.01
//	 - Changes - different updates;
//
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Connected libraries:
//---------------------------------------------------------------------

#include	<Object.mqh>
#include	<Arrays\List.mqh>
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	The extremum term:
//---------------------------------------------------------------------
class TExtremum : public CObject
{
private:
	datetime			extr_datetime;																	// date/time in extremum point
	double				extr_price;																	// price in extremum point
	
protected:
	virtual int		Compare(const CObject* _node, int _mode = 0) const;

public:
	void					TExtremum();																// constructor
	void					~TExtremum();																// destructor
	void					SetExtremum(datetime _time, double _price);						// change date/time and price in extremum point
	void					SetDateTime(datetime _time);											// change date/time in extremum point
	void					SetPrice(double _price);												// change price on extremum point

public:
	datetime			GetDateTime() const;															// get date/time in extremum point
	double				GetPrice() const;															// get price in extremum point

public:
  virtual bool	SaveExtremum(string _dt_name, string _p_name);				// save extremum
  virtual bool	LoadExtremum(string _dt_name, string _p_name);				// load extremum
  virtual bool	DeleteExtremum(string _dt_name, string _p_name);			// delete extremum
};
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Constructor:
//---------------------------------------------------------------------
void	TExtremum::TExtremum()
{
	this.extr_datetime = 0;
	this.extr_price = 0.0;
}

//---------------------------------------------------------------------
//	Destructor:
//---------------------------------------------------------------------
void	TExtremum::~TExtremum()
{
}

//---------------------------------------------------------------------
//	Save extremum (date/time):
//---------------------------------------------------------------------
bool	TExtremum::SaveExtremum(string _dt_name, string _p_name)
{
	datetime  dt_result = GlobalVariableSet(_dt_name, (double)this.extr_datetime);
	datetime  p_result = GlobalVariableSet(_p_name, (double)this.extr_price);
	if(dt_result != 0 && p_result != 0)
	{
		return(true);
	}

	return(false);
}

//---------------------------------------------------------------------
//	Load extremum (date/time):
//---------------------------------------------------------------------
bool	TExtremum::LoadExtremum(string _dt_name, string _p_name)
{
	double	dt_temp, p_temp;
	bool		result = GlobalVariableGet(_dt_name, dt_temp);
	result &= GlobalVariableGet(_p_name, p_temp);
	if(result != false)
	{
		this.extr_datetime = (datetime)dt_temp;
		this.extr_price = p_temp;
		return(true);
	}

	return(false);
}

//---------------------------------------------------------------------
//	Delete extremum:
//---------------------------------------------------------------------
bool	TExtremum::DeleteExtremum(string _dt_name, string _p_name)
{
	bool		result = GlobalVariableDel(_dt_name);
	result &= GlobalVariableDel(_p_name);
	return(result);
}

//---------------------------------------------------------------------
//	Comparing two extremums by time:
//---------------------------------------------------------------------
int	TExtremum::Compare(const CObject* _node, int _mode = 0) const
{
	datetime		temp = ((TExtremum*)_node).GetDateTime();
	datetime		curr = this.GetDateTime();
	if(curr > temp)
	{
		return(_mode > 0 ? 1 : -1);
	}
	else if(curr < temp)
	{
		return(_mode > 0 ? -1 : 1);
	}

	return(0);
}

//---------------------------------------------------------------------
//	Change date/time and price in extremum point:
//---------------------------------------------------------------------
void	TExtremum::SetExtremum(datetime _time, double _price)
{
	this.extr_datetime = _time;
	this.extr_price = _price;
}

//---------------------------------------------------------------------
//	Change date/time in extremum point:
//---------------------------------------------------------------------
void	TExtremum::SetDateTime(datetime _time)
{
	this.extr_datetime = _time;
}

//---------------------------------------------------------------------
//	Change price in extremum point:
//---------------------------------------------------------------------
void	TExtremum::SetPrice(double _price)
{
	this.extr_price = _price;
}

//---------------------------------------------------------------------
//	Get date/time in extremum point:
//---------------------------------------------------------------------
datetime	TExtremum::GetDateTime() const
{
	return(this.extr_datetime);
}

//---------------------------------------------------------------------
//	Get price in extremum point:
//---------------------------------------------------------------------
double	TExtremum::GetPrice() const
{
	return(this.extr_price);
}

//---------------------------------------------------------------------
//	Manipulations with extremums:
//---------------------------------------------------------------------
class TExtremumList : public CList
{
private:
	string							channel_prefix;															// channel name (prefix)
	ENUM_TIMEFRAMES			chart_timeframe;																// current timeframe
	string							chart_symbol;																// work symbol of the chart

protected:
	string	MakeDTimeName(int _nmb);																		// get name for saving/reading date/time of extremum
	string	MakePriceName(int _nmb);																		// get name for saving/reading price of extremum

public:
	void		TExtremumList();																					// constructor
	void		~TExtremumList();																					// destructor
	void		SetChannalParams(string _pref, string _symbol = NULL, ENUM_TIMEFRAMES _curr_tf = PERIOD_CURRENT);
	void		AddExtremum(datetime _time, double _price);
	void		DeleteAllExtremum();
	void		SaveExtremumList();
	void		LoadExtremumList();
	int			FindExtremum(datetime _dt);																// search extremums with a specified time

public:
	datetime	GetDateTime(int _index);
	double		GetPrice(int _index);
};
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Constructor:
//---------------------------------------------------------------------
void	TExtremumList::TExtremumList()
{
	this.chart_timeframe = PERIOD_CURRENT;
	this.channel_prefix = "";
	this.chart_symbol = NULL;
}

//---------------------------------------------------------------------
//	Destructor:
//---------------------------------------------------------------------
void	TExtremumList::~TExtremumList()
{
	this.Clear();
}

//---------------------------------------------------------------------
//	Setting general parameters of the channel:
//---------------------------------------------------------------------
void	TExtremumList::SetChannalParams(string _pref, string _symbol, ENUM_TIMEFRAMES _curr_tf)
{
	this.channel_prefix = _pref;

	if(_symbol == NULL)
	{
		this.chart_symbol = ChartSymbol();
	}
	else
	{
		this.chart_symbol = _symbol;
	}

	if(_curr_tf == 0)
	{
		this.chart_timeframe = ChartPeriod();
	}
	else
	{
		this.chart_timeframe = _curr_tf;
	}
}

//---------------------------------------------------------------------
//	Deleting all extremums from the list:
//---------------------------------------------------------------------
void	TExtremumList::DeleteAllExtremum()
{
	this.Clear();
}

//---------------------------------------------------------------------
//	Adding an extremum to the list with sorting by time:
//---------------------------------------------------------------------
void	TExtremumList::AddExtremum(datetime _time, double _price)
{
//	Creating extremum:
	TExtremum*	extr = new TExtremum();
	extr.SetExtremum(_time, _price);

//	Adding it to the list:
	this.Add(extr);

//	Sorting:
	this.Sort(1);
}

//---------------------------------------------------------------------
//	Saving parameters of extremums:
//---------------------------------------------------------------------
void	TExtremumList::SaveExtremumList()
{
	int		k = 1;
	TExtremum*	extr = (TExtremum*)(this.GetFirstNode());
	while(extr != NULL)
	{
		extr.SaveExtremum(MakeDTimeName(k), MakePriceName(k));
		k++;
		extr = (TExtremum*)(this.GetNextNode());
	}
}

//---------------------------------------------------------------------
//	Searching all extremums with a specified time:
//---------------------------------------------------------------------
int		TExtremumList::FindExtremum(datetime _dt)
{
	int		k = 0;
	TExtremum*	extr = (TExtremum*)(this.GetFirstNode());
	while(extr != NULL)
	{
		if(extr.GetDateTime() == _dt)
		{
			return(k);
		}
		extr = (TExtremum*)(this.GetNextNode());
	}
	return(-1);																					// extremum not found
}

//---------------------------------------------------------------------
//	Load extremums:
//---------------------------------------------------------------------
void	TExtremumList::LoadExtremumList()
{
//	Clear the list from previous extremums:
	this.DeleteAllExtremum();

	int		count = GlobalVariablesTotal();
	if(count <= 0)
	{
		return;
	}

	for(int i = (count - 1); i >= 0; i--)
	{
		string	name = GlobalVariableName(i);
		string	dt_name = MakeDTimeName(-1);
		int			index = StringFind(name, dt_name);
		if(index == -1 || index != 0)
		{
			continue;
		}

		string	p_name = MakePriceName(-1) + StringSubstr(name, StringLen(dt_name));
//	Creating extremum:
		TExtremum*	extr = new TExtremum();
		extr.LoadExtremum(name, p_name);
		
		this.Add(extr);
	}

//	Sorting:
	this.Sort(1);
}

//---------------------------------------------------------------------
//	Get the date/time value of an extremum:
//---------------------------------------------------------------------
datetime	TExtremumList::GetDateTime(int _index)
{
	TExtremum*	extr = (TExtremum*)(this.GetNodeAtIndex(_index));
	if(extr != NULL)
	{
		return(extr.GetDateTime());
	}

	return(0);	
}

//---------------------------------------------------------------------
//	Get the price value of an extremum:
//---------------------------------------------------------------------
double	TExtremumList::GetPrice(int _index)
{
	TExtremum*	extr = (TExtremum*)(this.GetNodeAtIndex(_index));
	if(extr != NULL)
	{
		return(extr.GetPrice());
	}

	return(0);	
}

//---------------------------------------------------------------------
//	Get the name for saving/reading the date/time of extremum:
//---------------------------------------------------------------------
string	TExtremumList::MakeDTimeName(int _nmb)
{
	string	name;
	if(_nmb > 0)
	{
		StringConcatenate(name, this.channel_prefix, "_", this.chart_symbol, "_", this.chart_timeframe, "_DTime_Extr", _nmb);
	}
	else
	{
		StringConcatenate(name, this.channel_prefix, "_", this.chart_symbol, "_", this.chart_timeframe, "_DTime_Extr");
	}
	return(name);
}

//---------------------------------------------------------------------
//	Get the name for saving/reading of price of extremum:
//---------------------------------------------------------------------
string	TExtremumList::MakePriceName(int _nmb)
{
	string	name;
	if(_nmb > 0)
	{
		StringConcatenate(name, this.channel_prefix, "_", this.chart_symbol, "_", this.chart_timeframe, "_Price_Extr", _nmb);
	}
	else
	{
		StringConcatenate(name, this.channel_prefix, "_", this.chart_symbol, "_", this.chart_timeframe, "_Price_Extr");
	}
	return(name);
}
//---------------------------------------------------------------------
