#include "ExecutionAlgorithm.mqh"
#include "TWAP.mqh"
#include "VWAP.mqh"
#include "IcebergOrder.mqh"
#include "PerformanceAnalyzer.mqh"
#include "ExecutionManager.mqh"
#include <Trade\Trade.mqh>

// Input parameters
input int    FastMA              = 20;    // Fast moving average period
input int    SlowMA              = 50;    // Slow moving average period
input double TradingVolume       = 0.1;   // Trading volume
input bool   UseAdaptiveExecution = true; // Use adaptive execution based on market conditions
input double EquityTPPercent     = 10.0;  // Equity Take Profit in percent
input double EquitySLPercent     = 5.0;   // Equity Stop Loss in percent

// Global variables
CExecutionManager *g_executionManager = NULL;
int                g_maHandle1       = INVALID_HANDLE;
int                g_maHandle2       = INVALID_HANDLE;
double             g_initialEquity   = 0.0;
CTrade             trade;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit() {
    // Record initial equity
    g_initialEquity = AccountInfoDouble(ACCOUNT_EQUITY);

    // Initialize execution manager
    g_executionManager = new CExecutionManager(Symbol(), 3, UseAdaptiveExecution);

    // Initialize indicators
    g_maHandle1 = iMA(Symbol(), Period(), FastMA, 0, MODE_SMA, PRICE_CLOSE);
    g_maHandle2 = iMA(Symbol(), Period(), SlowMA, 0, MODE_SMA, PRICE_CLOSE);

    if(g_maHandle1 == INVALID_HANDLE || g_maHandle2 == INVALID_HANDLE) {
        Print("Failed to create indicator handles");
        return INIT_FAILED;
    }

    return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {
    // Clean up
    if(g_executionManager != NULL) {
        delete g_executionManager;
        g_executionManager = NULL;
    }

    // Release indicator handles
    IndicatorRelease(g_maHandle1);
    IndicatorRelease(g_maHandle2);
}

//+------------------------------------------------------------------+
//| Check equity-based TP and SL, then reset baseline                 |
//+------------------------------------------------------------------+
void CheckEquityTPandSL() {
    double currentEquity = AccountInfoDouble(ACCOUNT_EQUITY);
    double tpEquity      = g_initialEquity * (1.0 + EquityTPPercent / 100.0);
    double slEquity      = g_initialEquity * (1.0 - EquitySLPercent / 100.0);

    if(currentEquity >= tpEquity) {
        Print("Equity Take Profit reached: ", currentEquity);
        CloseAllPositions();
        g_initialEquity = currentEquity;
        Print("Equity baseline reset to: ", g_initialEquity);
    } else if(currentEquity <= slEquity) {
        Print("Equity Stop Loss reached: ", currentEquity);
        CloseAllPositions();
        g_initialEquity = currentEquity;
        Print("Equity baseline reset to: ", g_initialEquity);
    }
}

//+------------------------------------------------------------------+
//| Close all open positions                                         |
//+------------------------------------------------------------------+
void CloseAllPositions() {
    CPositionInfo  m_position;
    CTrade m_trade;
    for(int i = PositionsTotal() - 1; i >= 0; i--) // loop all Open Positions
        if(m_position.SelectByIndex(i)) { // select a position
            m_trade.PositionClose(m_position.Ticket()); // then delete it --period
        }
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick() {
    // Check and reset equity thresholds
    CheckEquityTPandSL();

    // Update execution algorithms
    if(g_executionManager != NULL)
        g_executionManager.UpdateAlgorithms();

    // Only process at bar open
    if(iVolume(_Symbol, PERIOD_CURRENT, 0) > 1)
        return;

    // Get indicator values
    double fastMA[2], slowMA[2];

    if(CopyBuffer(g_maHandle1, 0, 0, 2, fastMA) <= 0 ||
            CopyBuffer(g_maHandle2, 0, 0, 2, slowMA) <= 0) {
        Print("Failed to copy indicator buffers");
        return;
    }

    // Check for trend signals
    bool buySignal  = (fastMA[0] > slowMA[0]) && (fastMA[1] <= slowMA[1]);
    bool sellSignal = (fastMA[0] < slowMA[0]) && (fastMA[1] >= slowMA[1]);

    // Execute signals using the execution manager
    if(buySignal) {
        Print("Buy signal detected");

        if(UseAdaptiveExecution) {
            // Let the execution manager select the best algorithm
            g_executionManager.ExecuteSignal(SIGNAL_TYPE_BUY, TradingVolume);
        } else {
            datetime currentTime = TimeCurrent();
            CTWAP *twap = g_executionManager.CreateTWAP(TradingVolume, currentTime,
                          currentTime + 3600, 6,
                          ORDER_TYPE_BUY, true);
        }
    } else if(sellSignal) {
        Print("Sell signal detected");

        if(UseAdaptiveExecution) {
            // Let the execution manager select the best algorithm
            g_executionManager.ExecuteSignal(SIGNAL_TYPE_SELL, TradingVolume);
        } else {
            datetime currentTime = TimeCurrent();
            CTWAP *twap = g_executionManager.CreateTWAP(TradingVolume, currentTime,
                          currentTime + 3600, 6,
                          ORDER_TYPE_SELL, true);
        }
    }
}
//+------------------------------------------------------------------+
