#pragma once

#ifdef ASYNCWEBSOCKETCLIENT_EXPORTS
#define WEBSOCK_API __declspec(dllexport)
#else
#define WEBSOCK_API __declspec(dllimport)
#endif


#include <vector>
#include <string>
#include <map>
#include <queue>
#include <mutex>
#include <windows.h>
#include <winhttp.h>

#define BUFFERSIZE 65535

// WinHTTP WebSocket client namespace
namespace WinHttpWebSocketClient
{
	// client state
	enum ENUM_WEBSOCKET_STATE
	{
		CLOSED = 0,
		CLOSING = 1,
		CONNECTING = 2,
		CONNECTED = 3,
		SENDING = 4,
		POLLING = 5
	};
	//websockets frame abstraction
	struct Frame
	{
		std::vector<BYTE>frame_buffer;
		WINHTTP_WEB_SOCKET_BUFFER_TYPE frame_type;
		DWORD frame_size;
	};

	// WebSocket client class
	class WebSocketClient {
	private:
		// Application session handle to use with this connection
		HINTERNET hSession;
		// Windows connect handle
		HINTERNET hConnect;
		// The initial HTTP request handle to start the WebSocket handshake
		HINTERNET hRequest;
		// Windows WebSocket handle
		HINTERNET hWebSocket;
		//initialization flag
		DWORD initialized;
		//sent bytes
		DWORD bytesTX;
		//last error code
		DWORD ErrorCode;
		//last completed websocket operation as indicated by callback function
		DWORD completed_websocket_operation;
		//internal queue of frames sent from a server
		std::queue<Frame>* frames;
		//client state;
		ENUM_WEBSOCKET_STATE status;
		//sets an hSession handle
		DWORD Initialize(VOID);
		// reset state of object
		/*
		 reset_error: boolean flag indicating whether to rest the
		 internal error buffers.
		*/
		VOID  Reset(bool reset_error = true);
		

	public:
		//constructor(s)
		WebSocketClient(VOID);
		WebSocketClient(const WebSocketClient&) = delete;
		WebSocketClient(WebSocketClient&&) = delete;
		WebSocketClient& operator=(const WebSocketClient&) = delete;
		WebSocketClient& operator=(WebSocketClient&&) = delete;
		//destructor
		~WebSocketClient(VOID);
		//received bytes;
		DWORD bytesRX;
		// receive buffer
		std::vector<BYTE> rxBuffer;
		// received frame type;
		WINHTTP_WEB_SOCKET_BUFFER_TYPE rxBufferType;
		// Get the winhttp websocket handle
		/*
		return: returns the hWebSocket handle which is used to 
		identify a websocket connection instance
		*/
		HINTERNET WebSocketHandle(VOID);

		// Connect to a server
		/* 
		   hsession: HINTERNET session handle
		   host: is the url
		   port: prefered port number to use
		   secure: 0 is false, non-zero is true
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
		*/
		DWORD Connect(const WCHAR* host, const INTERNET_PORT port, const DWORD secure);
		
		// Send data to the WebSocket server
		/*
		   bufferType: WINHTTP_WEB_SOCKET_BUFFER_TYPE enumeration of the frame type
		   pBuffer: pointer to the data to be sent
		   dwLength: size of pBuffer data
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
		*/
		DWORD Send(WINHTTP_WEB_SOCKET_BUFFER_TYPE bufferType, void* pBuffer, DWORD dwLength);

		// Close the connection to the server
		/*
		   status: WINHTTP_WEB_SOCKET_CLOSE_STATUS enumeration of the close notification to be sent
		   reason: character string of extra data sent with the close notification
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
		*/
		DWORD Close(WINHTTP_WEB_SOCKET_CLOSE_STATUS status, CHAR* reason = NULL);

		// Retrieve the close status sent by a server
		/*
		   pusStatus: pointer to a close status code that will be filled upon return.
		   pvReason: pointer to a buffer that will receive a close reason
		   dwReasonLength: The length of the pvReason buffer,
		   pdwReasonLengthConsumed:The number of bytes consumed. If pvReason is NULL and dwReasonLength is 0,
           pdwReasonLengthConsumed will contain the size of the buffer that needs to be allocated
           by the calling application.
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
		*/
		DWORD QueryCloseStatus(USHORT* pusStatus, PVOID pvReason, DWORD dwReasonLength, DWORD* pdwReasonLengthConsumed);

		// read from the server
		/*
		   bufferType: WINHTTP_WEB_SOCKET_BUFFER_TYPE enumeration of the frame type
		   pBuffer: pointer to the data to be sent
		   pLength: size of pBuffer
		   bytesRead: pointer to number bytes read from the server
		   pBufferType: pointer to type of frame sent from the server
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
		*/
		DWORD Receive(PVOID pBuffer, DWORD pLength, DWORD* bytesRead, WINHTTP_WEB_SOCKET_BUFFER_TYPE* pBufferType);

		// Check client state
		/*
		   return: ENUM_WEBSOCKET_STATE enumeration
		*/
		ENUM_WEBSOCKET_STATE Status(VOID);

		// get frames cached in the internal queue
		/*
		   pBuffer: User supplied container to which data is written to
		   pLength: size of pBuffer
		   pBufferType: WINHTTP_WEB_SOCKET_BUFFER_TYPE enumeration of frame type 
		*/
		VOID Read(BYTE* pBuffer, DWORD pLength, WINHTTP_WEB_SOCKET_BUFFER_TYPE* pBufferType);

		// get bytes received
		/*
		   return: Size of most recently cached frame sent from a server
		*/
		DWORD ReadAvailable(VOID);

		// get the last error
		/*
		   return: returns the last error code
		*/
		DWORD LastError(VOID);

		// activate callback function
		/*
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
		*/
		DWORD EnableCallBack(VOID);
		// set error 
		/*
		   message: Error description to be captured
		   errorcode: new user defined error code
		*/
		VOID SetError(const DWORD errorcode);
		 
		// get the last completed operation
		/*
		  returns: DWORD constant of last websocket operation
		*/
		DWORD LastOperation(VOID);

		//deinitialize the session handle and free up resources
		VOID Free(VOID);

		//the following methods define handlers meant to be triggered by the callback function//
		
		// on error 
		/*
		   result: pointer to WINHTTP_ASYNC_RESULT structure that 
		   encapsulates the specific event that triggered the error
		*/
		VOID OnError(const WINHTTP_ASYNC_RESULT* result);

		// read completion handler
		/*
		   read: Number of bytes of data successfully read from the server
		   buffertype: type of frame read-in.
		   Called when successfull read is completed
		*/
		VOID OnReadComplete(const DWORD read, const WINHTTP_WEB_SOCKET_BUFFER_TYPE buffertype);

		// websocket close handler
		/*
		   Handles the a successfull close request
		*/
		VOID OnClose(VOID);

		// Send operation handler
		/*
		   sent: the number of bytes successfully sent to the server if any
		   This is a handler for an asynchronous send that interacts with the
		   callback function
		*/
		VOID OnSendComplete(const DWORD sent);

		//set the last completed websocket operation 
		/*
		  operation : constant defining the operation flagged as completed by callback function
		*/
		VOID OnCallBack(const DWORD operation);
	};

    // container for  websocket objects accessible to callback function
	extern std::map<HINTERNET, std::shared_ptr<WebSocketClient>>clients;
	

	// deinitializes a session handle
	/*
	   websocket_handle: HINTERNET the websocket handle to close
	*/
	VOID WEBSOCK_API client_reset(HINTERNET websocket_handle);
	
	//creates a client connection to a server
	/*   
		   url: the url of the server
		   port: port
		   secure: use secure connection(non-zero) or not (zero)
		   websocket_handle: in-out,HINTERNET non NULL session handle
		   return: returns DWORD, zero if successful or non-zero on failure
		   
    */

	DWORD  WEBSOCK_API client_connect(const WCHAR* url, INTERNET_PORT port, DWORD secure, HINTERNET* websocket_handle);

	//destroys a client connection to a server
	/*
		   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
	*/

	void WEBSOCK_API client_disconnect(HINTERNET websocket_handle);

	//writes data to a server (non blocking)
	/*
		   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
		   bufferType: WINHTTP_WEB_SOCKET_BUFFER_TYPE enumeration of the frame type
		   message: pointer to the data to be sent
		   length: size of pBuffer data
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
	*/

	DWORD WEBSOCK_API client_send(HINTERNET websocket_handle, WINHTTP_WEB_SOCKET_BUFFER_TYPE buffertype, BYTE* message, DWORD length);

	//reads data sent from a server cached internally
	/*
		   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
		   out: User supplied container to which data is written to
		   out_size: size of out buffer
		   buffertype: WINHTTP_WEB_SOCKET_BUFFER_TYPE enumeration of frame type 
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
	*/

	DWORD WEBSOCK_API client_read(HINTERNET websocket_handle, BYTE* out, DWORD out_size, WINHTTP_WEB_SOCKET_BUFFER_TYPE* buffertype);

	//listens for a response from a server (non blocking)
	/*
		   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
	*/

	DWORD WEBSOCK_API client_poll(HINTERNET websocket_handle);

	//gets the last generated error
	/*
		   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
		   lasterror: container that will hold the error description
		   length: the size of lasterror container
		   lasterrornum: reference to which the last error code is written
		   return: DWORD error code, 0 indicates success
		   and non-zero for failure
	*/

	DWORD WEBSOCK_API  client_lasterror(HINTERNET websocket_handle);

	//checks whether there is any data cached internally
	/*
	 	   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
	 	   return: returns the size of the last received frame in bytes;
	*/

	DWORD WEBSOCK_API client_readable(HINTERNET websocket_handle);

	//return the state of a websocket connection
	/*
	 	   websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
	 	   return: ENUM_WEBSOCKET_STATE enumeration of the state of a client
	*/

	ENUM_WEBSOCKET_STATE WEBSOCK_API client_status(HINTERNET websocket_handle);

	//return the last websocket operation
	/*
	     websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
	     return : DWORD constant corresponding to a unique callback status value as defined in API    
	*/
	DWORD WEBSOCK_API client_lastcallback_notification(HINTERNET websocket_handle);

	//return the websocket handle
	/*
	     websocket_handle: a valid (non NULL) websocket handle created by calling client_connect()
	     return : HINTERNET returns the websocket handle for a client connection 
	*/
	HINTERNET WEBSOCK_API client_websocket_handle(HINTERNET websocket_handle);

}









