//+------------------------------------------------------------------+
//|                                       Optimization&&Tweaking.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"

//+------------------------------------------------------------------+
//|                           Includes                               |
//+------------------------------------------------------------------+
#include <Trade/Trade.mqh>
MqlTick CTick, PTick;
CTrade trade;

enum InLot{
   Lot_fixed,
   Lot_dynamic,
};

//+------------------------------------------------------------------+
//|                           Inputs                                 |
//+------------------------------------------------------------------+
input group "--------------General Inputs--------------"
input int RSI_Period = 14;
input int    TakeProfit   = 1000;     // TP in points
input int StopLoss = 400; // SL in points
input ulong  MagicNumber  = 888888;  // EA identifier
input int    MaxBarsCheck = 8;       // Historical bars to analyze
input InLot Lot_mode = Lot_fixed;
input double  In_Lot = 0.01;
input bool TrailYourStop = true;
input int trailingStop = 50;
//input int StopLoss = 234;

//+------------------------------------------------------------------+
//|                      Global Variables                            |
//+------------------------------------------------------------------+
int rsiHandle, macdHandle;
datetime lastBarTime;

double pointMultiplier;
//double StopLoss     = 150;

bool bullish_pattern_met = false;
bool bearish_pattern_met = false;

double first_bullish_low = 0.0;
double first_bearish_high = 0.0;

ENUM_TIMEFRAMES TimeFrame = PERIOD_CURRENT;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit(){

   // Create RSI and MACD indicator handles
   rsiHandle = iRSI(_Symbol, TimeFrame, RSI_Period, PRICE_CLOSE);
   macdHandle = iMACD(_Symbol, TimeFrame, 12, 26, 9, PRICE_CLOSE);

   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason){

   
}
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick(){

   if(!NewBarTrigger()) return;
   ThreeBar();
}
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Detects new bar formation                                        |
//+------------------------------------------------------------------+
bool NewBarTrigger(){

   datetime currentTime = iTime(_Symbol, PERIOD_CURRENT, 0);
   if(currentTime != lastBarTime){
      lastBarTime = currentTime;
      return true;
   }
   return false;
}

double getHigh(int index) {
    return iHigh(_Symbol, _Period, index);
}

double getLow(int index) {
    return iLow(_Symbol, _Period, index);
}

//+------------------------------------------------------------------+
//| Execute trade with risk parameters                               |
//+------------------------------------------------------------------+
void ExecuteTrade(ENUM_ORDER_TYPE tradeType){

   double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
   double price = (tradeType == ORDER_TYPE_BUY) ? SymbolInfoDouble(_Symbol, SYMBOL_ASK) :
                                                  SymbolInfoDouble(_Symbol, SYMBOL_BID);

   // Convert StopLoss and TakeProfit from pips to actual price distances
   double sl_distance = StopLoss * point;
   double tp_distance = TakeProfit * point;
   
   double sl = (tradeType == ORDER_TYPE_BUY) ? price - sl_distance :
                                               price + sl_distance;
   
   double tp = (tradeType == ORDER_TYPE_BUY) ? price + tp_distance :
                                               price - tp_distance;

   trade.PositionOpen(_Symbol, tradeType, In_Lot, price, sl, tp, NULL);
}


//+------------------------------------------------------------------+
//| Candle pattern detection Function                                |
//+------------------------------------------------------------------+
void ThreeBar() {
   // Settings
   string symbol = Symbol();
   ENUM_TIMEFRAMES timeframe = PERIOD_M5;

   // Pattern detection config
   int min_bullish_count = 3;
   int min_bearish_count = 3;
   int check_candles = 6;

   bool bullish_pattern = false;
   bool bearish_pattern = false;
   static bool bullish_pattern_detected = false;
   static bool bearish_pattern_detected = false;

   // Indicator buffers
   double rsi_value[1];    // Use array for buffer storage
   double macd_main[1];    
   double macd_signal[1];
   
   // Get indicator values - with error checking
   if (CopyBuffer(rsiHandle, 0, 0, 1, rsi_value) <= 0) {
       Print("RSI CopyBuffer error: ", GetLastError());
       return;
   }
   
   if (CopyBuffer(macdHandle, 0, 0, 1, macd_main) <= 0) {  
       Print("MACD Main Line error: ", GetLastError());
       return;
   }
   
   if (CopyBuffer(macdHandle, 1, 0, 1, macd_signal) <= 0) { 
       Print("MACD Signal Line error: ", GetLastError());
       return;
   }

   for (int i = check_candles; i >= min_bullish_count; i--) {
      bullish_pattern = false;
      bearish_pattern = false;

      int bullish_count = 0;
      int bearish_count = 0;

      // Check for bullish pattern
      for (int j = i; j >= i - min_bullish_count + 1; j--) {
         if (iClose(symbol, timeframe, j) > iOpen(symbol, timeframe, j)) {
            bullish_count++;
            first_bullish_low = getLow(5);
         } else break;
      }

      if (bullish_count >= min_bullish_count) {
         for (int j = i - bullish_count; j >= i - bullish_count - 1; j--) {
            if (iClose(symbol, timeframe, j) < iOpen(symbol, timeframe, j))
               bearish_count++;
            else break;
         }

         if ((bearish_count == 1 || bearish_count == 2) &&
             iClose(symbol, timeframe, i - bullish_count - bearish_count) > iOpen(symbol, timeframe, i - bullish_count - bearish_count)) {
            bullish_pattern = true;
         }
      }

      // Check for bearish pattern
      int bearish_candles_count = 0;
      int bullish_candles_count = 0;

      for (int j = i; j >= i - min_bearish_count + 1; j--) {
         if (iClose(symbol, timeframe, j) < iOpen(symbol, timeframe, j)) {
            bearish_candles_count++;
            first_bearish_high = getHigh(5);
         } else break;
      }

      if (bearish_candles_count >= min_bearish_count) {
         for (int j = i - bearish_candles_count; j >= i - bearish_candles_count - 1; j--) {
            if (iClose(symbol, timeframe, j) > iOpen(symbol, timeframe, j))
               bullish_candles_count++;
            else break;
         }

         if ((bullish_candles_count == 1 || bullish_candles_count == 2) &&
             iClose(symbol, timeframe, i - bearish_candles_count - bullish_candles_count) < iOpen(symbol, timeframe, i - bearish_candles_count - bullish_candles_count)) {
            bearish_pattern = true;
         }
      }

      // Filter + Trade Logic
      if (bullish_pattern && !bullish_pattern_detected &&
          rsi_value[0] > 50 && macd_main[0] > macd_signal[0])  
      {
          Print("Bullish pattern confirmed with RSI(", rsi_value[0], ") and MACD(", 
                macd_main[0], "/", macd_signal[0], ")");
          ExecuteTrade(ORDER_TYPE_BUY);
          bullish_pattern_detected = true;
          break;
      }
      else if (!bullish_pattern)
          bullish_pattern_detected = false;
      
      if (bearish_pattern && !bearish_pattern_detected &&
          rsi_value[0] < 50 && macd_main[0] < macd_signal[0])  
      {
          Print("Bearish pattern confirmed with RSI(", rsi_value[0], ") and MACD(", 
                macd_main[0], "/", macd_signal[0], ")");
          ExecuteTrade(ORDER_TYPE_SELL);
          bearish_pattern_detected = true;
          break;
      }
      else if (!bearish_pattern)
          bearish_pattern_detected = false;
   }
}




bool calculateLots(double slDs, double &lots){
   lots = 0.0;
   
   if(Lot_mode == Lot_fixed){
      lots = In_Lot;
   }else if(Lot_mode == Lot_dynamic){
   
      //------------------------------------------------------
      double Lmaxs = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
      double Lmins = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
      double Lstep = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
      
      //double volume = 0.0;
      double balance = AccountInfoDouble(ACCOUNT_EQUITY);
      
      lots = NormalizeDouble((0.01 / 100 * balance), _Digits);      
      // Ensure volume falls within permissible limits and conforms to the volume step
      lots = MathMax(Lmins, MathMin(Lmaxs, lots));
      lots = MathFloor(lots / Lstep) * Lstep;
   }
   
   if(!Checklots(lots)){return false;}
   return true;
}

bool Checklots(double &lots){
   double min = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   double max = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
   double step = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   if(lots < min){
      lots = min;
      return true;
   }
   if(lots > max){
      return false;
   }
   
   lots = (int)MathFloor(lots / step)  * step;
   
   return true;
}

